import argparse
import pickle
import pprint
import os
import copy
from glob import glob

import numpy as np
import torch
import torchvision
from PIL import Image
from sklearn.cluster import KMeans
from sklearn.preprocessing import MinMaxScaler
from sklearn.decomposition import PCA
from yellowbrick.cluster import KElbowVisualizer
import matplotlib.pyplot as plt

def main(args):
    os.makedirs(os.path.join(args.out_dir, args.factor), exist_ok=True)

    factor_traj = {}
    global_dict = {}

    base_key = None
    
    source_dir_list = glob(os.path.join(args.root, args.factor, '*.pkl'))
    for i, instance_file in enumerate(source_dir_list):
        # file name example: drawer-close-v2_cam0-0_s0_base.pkl
        k = i
        if "base" in instance_file:
            base_key = k
        factor_traj[k] = instance_file

    factor_traj = dict(sorted(factor_traj.items()))
    pprint.pprint(factor_traj)

    # base
    with open(factor_traj[base_key], 'rb') as f:
        data = pickle.load(f)
    print(data.keys()) # dict_keys(['images', 'observations', 'actions', 'rewards', 'terminals', 'infos'])
    
    save_data = {
        "frame": [],
        "action": [],
        "reward": [],
        "done": [],
        "info": [],
    }

    for i in range(len(data["images"])):
        # print(data["images"][i].shape) # (224, 224, 3)
        # print(data["actions"][i]) # [[ 0.87757814  0.99764276 -0.8248665   0.6245673 ]]
        # print(data["rewards"][i]) # [0.5032537]
        # print(data["terminals"][i]) # [False]
        # print(data["infos"][i]) # [{'success': 0.0, 'near_object': 0.0, 'grasp_success': 1.0, 'grasp_reward': 0.09150016997239699, 'in_place_reward': 0.10058590107143828, 'obj_to_target': 0.16724927037857554, 'unscaled_reward': 0.5032537210031514, 'is_success': False}]
        save_data["frame"].append(data["images"][i])
        save_data["action"].append(data["actions"][i][0])
        save_data["reward"].append(data["rewards"][i])
        save_data["done"].append(data["terminals"][i])
        save_data["info"].append(data["infos"][i])

    global_dict[base_key] = copy.deepcopy(save_data)

    for key in factor_traj.keys():

        if key == base_key:
            continue

        with open(factor_traj[key], 'rb') as f:
            data = pickle.load(f)

        save_data = {
            "frame": [],
            "action": [],
            "reward": [],
            "done": [],
            "info": [],
        }

        for i in range(len(data["images"])):
            save_data["frame"].append(data["images"][i])
            save_data["action"].append(data["actions"][i][0])
            save_data["reward"].append(data["rewards"][i])
            save_data["done"].append(data["terminals"][i])
            save_data["info"].append(data["infos"][i])

        global_dict[key] = copy.deepcopy(save_data)


    # action clustering
    actions = []
    for key in factor_traj.keys():
        for action in global_dict[key]["action"]:
            actions.append(action.copy())
        
    print("total acions:", len(actions))
    actions = np.array(actions)
    scaler = MinMaxScaler()
    actions_scaled = scaler.fit_transform(actions.copy())

    model = KMeans()
    visualizer = KElbowVisualizer(model, k=(1,10))
    visualizer.fit(actions_scaled)
    visualizer.show(outpath=os.path.join(args.out_dir, args.factor,"recommand.png"))

    k = args.K  # easy:5 medium: 5, hard: 4
    
    model = KMeans(n_clusters = k, random_state = 10)
    model.fit(actions_scaled)
    output = model.fit_predict(actions_scaled)

    labels = model.labels_
    centers = model.cluster_centers_

    color_dict = {0: 'red', 1: 'blue', 2:'green', 3:'brown', 4:'indigo'} 

    fig = plt.figure()
    ax = fig.add_subplot(1, 1, 1)

    for cluster in range(k):
        cluster_sub_points = actions_scaled[labels == cluster] 
        ax.scatter(cluster_sub_points[:, 0], cluster_sub_points[:, 1], c=color_dict[cluster], label='cluster_{}'.format(cluster)) 

    ax.set_xlabel('X')
    ax.set_ylabel('Y')
    ax.legend()
    ax.grid()
    plt.savefig(os.path.join(args.out_dir, args.factor, 'cluster.png'))

    # action labeling
    cnt = 0
    for key in factor_traj.keys():
        for i in range(len(global_dict[key]["action"])):
            action = np.expand_dims(actions_scaled[cnt], axis=0)
            action_label = model.predict(action)
            global_dict[key]["action"][i] = action_label.copy()
            cnt += 1
    assert cnt == len(actions)
    
    with open(os.path.join(args.out_dir, args.factor, args.file_name), 'wb') as f:
        global_dict["classes"] = ["buildings"]
        global_dict["actions"] = list(range(args.K))
        pickle.dump(global_dict, f, pickle.HIGHEST_PROTOCOL)

if __name__ == '__main__':
    parser = argparse.ArgumentParser(description='Expert Trajectory Preprocessing')
    # dataset parameters
    parser.add_argument('root', metavar='DIR',
                        help='root path of trajectory')
    parser.add_argument('--factor', type=str, default='FOV')
    parser.add_argument('--K', type=int, default=3)
    parser.add_argument('--out-dir', type=str, default='default')
    parser.add_argument('--file-name', type=str, default='train_dataset.pkl')
    args = parser.parse_args()
    main(args)
