from typing import Sequence, Union

import torch.nn as nn

from allenact.base_abstractions.preprocessor import Preprocessor
from allenact.utils.experiment_utils import Builder, TrainingPipeline
from allenact_plugins.clip_plugin.clip_preprocessors import (
    ClipViTPreprocessor,
)
from projects.plugins.ithor_plugin.ithor_sensors import (
    GoalObjectTypeThorSensor,
    RGBSensorThor,
)
from projects.object_navigation.baseline_configs.clip.zeroshot_mixins import (
    ClipViTPreprocessGRUActorCriticMixin,
    CLIPViTGRUActorCriticMixin
)
from projects.object_navigation.baseline_configs.ithor.objectnav_ithor_base import (
    ObjectNaviThorAUTOTESTBaseConfig,
)
from projects.object_navigation.baseline_configs.navigation_base import ObjectNavPPOMixin


class ObjectNaviThorClipViTRGBPPOExperimentConfig(ObjectNaviThorAUTOTESTBaseConfig):
    """A CLIP Object Navigation experiment configuration in RoboThor
    with RGB input."""

    CLIP_MODEL_TYPE = "ViT-B/32"
    NOISE_STD = 0.0

    SENSORS = [
        RGBSensorThor(
            height=ObjectNaviThorAUTOTESTBaseConfig.SCREEN_SIZE,
            width=ObjectNaviThorAUTOTESTBaseConfig.SCREEN_SIZE,
            use_resnet_normalization=True,
            mean=ClipViTPreprocessor.CLIP_RGB_MEANS,
            stdev=ClipViTPreprocessor.CLIP_RGB_STDS,
            uuid="rgb_lowres",
        ),
        GoalObjectTypeThorSensor(object_types=ObjectNaviThorAUTOTESTBaseConfig.TARGET_TYPES,),
    ]
    
    PROMPT = False
    MULTI_P_MODE = [None]
    META_MODE = False

    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        self.DATA_GEN = False

        # self.preprocessing_and_model = ClipViTPreprocessGRUActorCriticMixin(
        #     sensors=self.SENSORS,
        #     clip_model_type=self.CLIP_MODEL_TYPE,
        #     screen_size=self.SCREEN_SIZE,
        #     goal_sensor_type=GoalObjectTypeThorSensor,
        #     pool=False,
        #     pooling_type='',
        #     target_types=self.TARGET_TYPES,
        #     prompt = self.PROMPT,
        #     noise_std = self.NOISE_STD,
        # )
        self.preprocessing_and_model = CLIPViTGRUActorCriticMixin(
            sensors=self.SENSORS,
            clip_model_type=self.CLIP_MODEL_TYPE,
            screen_size=self.SCREEN_SIZE,
            goal_sensor_type=GoalObjectTypeThorSensor,
            pool=False,
            pooling_type='',
            target_types=self.TARGET_TYPES,
            prompt = self.PROMPT,
            multi_p_mode = self.MULTI_P_MODE,
            meta_mode = self.META_MODE,
            noise_std = self.NOISE_STD,
        )

    def training_pipeline(self, **kwargs) -> TrainingPipeline:
        return ObjectNavPPOMixin.training_pipeline(
            auxiliary_uuids=[],
            multiple_beliefs=False,
            advance_scene_rollout_period=self.ADVANCE_SCENE_ROLLOUT_PERIOD,
        )

    def preprocessors(self) -> Sequence[Union[Preprocessor, Builder[Preprocessor]]]:
        return self.preprocessing_and_model.preprocessors()

    def create_model(self, **kwargs) -> nn.Module:
        return self.preprocessing_and_model.create_model(
            num_actions=self.ACTION_SPACE.n, **kwargs
        )

    def tag(cls):
        return "objectnav_ithor_rgb_clip_vit32gru_ddppo_autotest"
# file name: objectnav_ithor_rgb_clip_vit32gru_ddppo_autotest