from typing import Sequence, Union

import torch.nn as nn

from allenact.base_abstractions.preprocessor import Preprocessor
from allenact.utils.experiment_utils import Builder, TrainingPipeline
from allenact_plugins.clip_plugin.clip_preprocessors import (
    ClipViTPreprocessor,
    NaivePreprocessor
)
from projects.plugins.ithor_plugin.ithor_sensors import (
    GoalObjectTypeThorSensor,
    RGBSensorThor,
)
from projects.object_navigation.baseline_configs.clip.zeroshot_mixins import (
    CLIPViTGRUActorCriticMixin
)
from projects.object_navigation.baseline_configs.robothor.zeroshot_objectnav_robothor_base import (
    ZeroshotObjectNavRoboThorBaseConfig,
)
from projects.object_navigation.baseline_configs.robothor.clip.objectnav_robothor_rgb_clipresnet50gru_ddppo import (
    ObjectNavRoboThorClipRGBPPOExperimentConfig,
)
from projects.object_navigation.baseline_configs.ithor.objectnav_ithor_base import (
    ObjectNaviThorMultiMDPsBaseConfig
)
from projects.object_navigation.baseline_configs.navigation_base import ObjectNavPPOMixin


class ObjectNaviThorClipViTRGBPPOExperimentConfig(ObjectNaviThorMultiMDPsBaseConfig):
    """A CLIP Object Navigation experiment configuration in RoboThor
    with RGB input."""

    CLIP_MODEL_TYPE = "ViT-B/32"
    NOISE_STD = 0.0

    SENSORS = [
        RGBSensorThor(
            height=ObjectNaviThorMultiMDPsBaseConfig.SCREEN_SIZE,
            width=ObjectNaviThorMultiMDPsBaseConfig.SCREEN_SIZE,
            use_resnet_normalization=True,
            mean=ClipViTPreprocessor.CLIP_RGB_MEANS,
            stdev=ClipViTPreprocessor.CLIP_RGB_STDS,
            uuid="rgb_lowres",
        ),
        GoalObjectTypeThorSensor(object_types=ObjectNaviThorMultiMDPsBaseConfig.TARGET_TYPES,),
    ]
    
    PROMPT = (
        "/home/chois/MMRL/logs/illumination_color/checkpoints/contrastive__latest.pth",
    )
    ATTN_MODE = "pre_attn"
    NETA_MODE = False

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        
        #### Domain defined by domain factors ####
        # self.STEP_SIZE =             [0.15, 0.18, 0.22, 0.25]
        # self.ROTATION_DEGREES =      [30.0, 60.0, 60.0, 90.0]
        # self.VISIBILITY_DISTANCE =   [1.0 , 1.0, 1.0, 1.0]
        # self.LIGHTING_VALUE =        [(0.5, 0.6, 0.2), (0.5, 0.7, 0.2), (1.5, 0.8, 0.8), (1.5, 0.9, 0.8)]
        # self.HORIZONTAL_FIELD_OF_VIEW = [49, 69, 89, 119]
        # self.LOOK_DEGREES = [1, 15, 30, 40]
        self.STEP_SIZE =             [0.25]
        self.ROTATION_DEGREES =      [30.0]
        self.VISIBILITY_DISTANCE =   [1.0]
        self.LIGHTING_VALUE =        [None]
        self.HORIZONTAL_FIELD_OF_VIEW = [79]
        self.LOOK_DEGREES = [30.0]
        ##########################################
        self.DATA_GEN = False

        self.preprocessing_and_model = CLIPViTGRUActorCriticMixin(
            sensors=self.SENSORS,
            clip_model_type=self.CLIP_MODEL_TYPE,
            screen_size=self.SCREEN_SIZE,
            goal_sensor_type=GoalObjectTypeThorSensor,
            pool=False,
            pooling_type='',
            target_types=self.TARGET_TYPES,
            prompt = self.PROMPT,
            attn_mode = self.ATTN_MODE,
            meta_mode = self.NETA_MODE,
            noise_std = self.NOISE_STD,
        )

    def training_pipeline(self, **kwargs) -> TrainingPipeline:
        return ObjectNavPPOMixin.training_pipeline(
            auxiliary_uuids=[],
            multiple_beliefs=False,
            advance_scene_rollout_period=self.ADVANCE_SCENE_ROLLOUT_PERIOD,
        )

    def preprocessors(self) -> Sequence[Union[Preprocessor, Builder[Preprocessor]]]:
        return self.preprocessing_and_model.preprocessors()

    def create_model(self, **kwargs) -> nn.Module:
        return self.preprocessing_and_model.create_model(
            num_actions=self.ACTION_SPACE.n, **kwargs
        )

    def tag(cls):
        return "PromptCLIPViTGRU-DDPPO-MDPs"
# file name: objectnav_ithor_rgb_prompt_clip_vit32gru_ddppo_mdps