import torch.nn as nn
from typing import Sequence, Union

from allenact.base_abstractions.preprocessor import Preprocessor
from allenact.utils.experiment_utils import Builder, TrainingPipeline
from allenact_plugins.clip_plugin.clip_preprocessors import (
    ClipViTPreprocessor,
    NaivePreprocessor
)
from projects.plugins.ithor_plugin.ithor_sensors import RGBSensorThor
from projects.plugins.robothor_plugin.robothor_sensors import GPSCompassSensorRoboThor
from projects.point_navigation.baseline_configs.clip.zeroshot_mixins import (
    ClipViTPreprocessGRUActorCriticMixin,
    CLIPViTGRUActorCriticMixin
)
from projects.point_navigation.baseline_configs.ithor.pointnav_ithor_base import (
    PointNaviThorAUTOTESTBaseConfig
)
from projects.point_navigation.baseline_configs.mixins import PointNavPPOMixin


class ObjectNaviThorClipViTRGBPPOExperimentConfig(PointNaviThorAUTOTESTBaseConfig):
    """A CLIP Object Navigation experiment configuration in RoboThor
    with RGB input."""

    CLIP_MODEL_TYPE = "ViT-B/32"
    NOISE_STD = 0.0

    SENSORS = [
        RGBSensorThor(
            height=PointNaviThorBaseConfig.SCREEN_SIZE,
            width=PointNaviThorBaseConfig.SCREEN_SIZE,
            use_resnet_normalization=True,
            mean=ClipViTPreprocessor.CLIP_RGB_MEANS,
            stdev=ClipViTPreprocessor.CLIP_RGB_STDS,
            uuid="rgb_lowres",
        ),
        GPSCompassSensorRoboThor(),
    ]
    
    PROMPT = (
        "../logs/PROMPTS/BRIGHTNESS/checkpoints/contrastive__latest.pth",
        "../logs/PROMPTS/CONTRAST/checkpoints/contrastive__latest.pth",
        "../logs/PROMPTS/SATURATION/checkpoints/contrastive__latest.pth",
        "../logs/PROMPTS/HUE/checkpoints/contrastive__latest.pth",
        
        "../logs/PROMPTS/FOV_39-59/checkpoints/comparative_action_byol_latest.pth",
        "../logs/PROMPTS/FOV_69-89/checkpoints/comparative_action_byol_latest.pth",
        "../logs/PROMPTS/FOV_99-139/checkpoints/comparative_action_byol_latest.pth",

        "../logs/PROMPTS/LOOK/checkpoints/comparative_action_byol_latest.pth",
        "../logs/PROMPTS/ROTATE/checkpoints/comparative_action_byol_latest.pth",
        "../logs/PROMPTS/STEPSIZE/checkpoints/comparative_action_byol_latest.pth",
    )
    MULTI_P_MODE = [
        ("ATTEMPT","WEIGHTED", "AVG"),
        ("SESoM","WEIGHTED", "AVG"),
        ]
    META_MODE = True
    SOURCE_MODEL = (
        # "/path/to/MMRL/allenact/storage/MAIN-EXP/ConPE/checkpoints/PromptATTNCLIPViTGRU-DDPPO-MDPs/2023-04-29_01-03-12/exp_PromptATTNCLIPViTGRU-DDPPO-MDPs__stage_00__steps_000003000000.pt",
        None,
        "/path/to/MMRL/allenact/storage/MAIN-EXP-POINTNAV/EMBCLIP/checkpoints/ViTGRU-DDPPO-MDPs/2023-05-05_02-09-16/exp_ViTGRU-DDPPO-MDPs__stage_00__steps_000001505280.pt"
        )

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        
        #### Domain defined by domain factors ####
        ##########################################
        self.DATA_GEN = False

        self.preprocessing_and_model = CLIPViTGRUActorCriticMixin(
            sensors=self.SENSORS,
            clip_model_type=self.CLIP_MODEL_TYPE,
            screen_size=self.SCREEN_SIZE,
            goal_sensor_type=GPSCompassSensorRoboThor,
            pool=False,
            pooling_type='',
            target_types=self.TARGET_TYPES,
            prompt = self.PROMPT,
            multi_p_mode = self.MULTI_P_MODE[0],
            meta_mode = self.META_MODE,
            noise_std = self.NOISE_STD,
            sm_noise = self.SEMANTIC_NOISE,
            source_model = self.SOURCE_MODEL
        )

    def training_pipeline(self, **kwargs) -> TrainingPipeline:
        return PointNavPPOMixin.training_pipeline(
            auxiliary_uuids=[],
            multiple_beliefs=False,
            normalize_advantage=True,
            advance_scene_rollout_period=self.ADVANCE_SCENE_ROLLOUT_PERIOD,
        )

    def preprocessors(self) -> Sequence[Union[Preprocessor, Builder[Preprocessor]]]:
        return self.preprocessing_and_model.preprocessors()

    def create_model(self, **kwargs) -> nn.Module:
        return self.preprocessing_and_model.create_model(
            num_actions=self.ACTION_SPACE.n, **kwargs
        )

    def tag(cls):
        return "pointnav_ithor_rgb_clip_vit32_pcomps_gru_ddppo_autotest"
# file name: pointnav_ithor_rgb_clip_vit32_pcomps_gru_ddppo_autotest