import torch
import torch.nn as nn
import torch.nn.functional as F
from adjointNetwork import *


class LinearBottleNeck(nn.Module):

    def __init__(self, in_channels, out_channels, stride, t=6, class_num=100, compression_factor=1, mask_layer=False):
        super().__init__()

        self.residual = nn.Sequential(
            conv2dAdjoint(in_channels, in_channels * t, kernel_size=1,stride=1, padding=0, mask_layer=mask_layer,compression_factor=compression_factor),
            batchNorm(in_channels * t, mask_layer),
            nn.ReLU6(inplace=True),

            conv2dAdjoint(in_channels * t, in_channels * t, kernel_size=3, stride=stride, padding=1, mask_layer=mask_layer, compression_factor=compression_factor, groups=in_channels * t),
            batchNorm(in_channels * t, mask_layer),
            nn.ReLU6(inplace=True),

            conv2dAdjoint(in_channels * t, out_channels, kernel_size=1,stride=1, padding=0, mask_layer=mask_layer,compression_factor=compression_factor),
            batchNorm(out_channels, mask_layer),
        )

        self.stride = stride
        self.in_channels = in_channels
        self.out_channels = out_channels

    def forward(self, x):
        residual = self.residual(x)

        if self.stride == 1 and self.in_channels == self.out_channels:
            residual += x

        return residual

class MobileNetV2(nn.Module):

    def __init__(self, class_num=100):
        super().__init__()

        self.pre = nn.Sequential(
            conv2dFirstLayer(3, 32, kernel_size=1, padding=1, stride=1),
            batchNorm(32,False), 
            nn.ReLU6(inplace=True)
        )

        self.stage1 = LinearBottleNeck(32, 16, 1, 1, mask_layer=False, compression_factor=1)
        self.stage2 = self._make_stage(2, 16, 24, 2, 6, False, 2)
        self.stage3 = self._make_stage(3, 24, 32, 2, 6, False, 2)
        self.stage4 = self._make_stage(4, 32, 64, 2, 6, False, 2)
        self.stage5 = self._make_stage(3, 64, 96, 1, 6, True, 2)
        self.stage6 = self._make_stage(3, 96, 160, 1, 6, True, 2)
        self.stage7 = LinearBottleNeck(160, 320, 1, 6, mask_layer=True, compression_factor=2)

        self.conv1 = nn.Sequential(
            conv2dAdjoint(320, 1280, kernel_size=1, stride=1, padding=0, mask_layer=True,compression_factor=2),
            batchNorm(1280, True),
            nn.ReLU6(inplace=True)
        )
        self.linear = linear(1280, class_num)
        #self.conv2 = nn.Conv2d(1280, class_num, 1)

    def forward(self, x):
        x = self.pre(x)
        x = self.stage1(x)
        x = self.stage2(x)
        x = self.stage3(x)
        x = self.stage4(x)
        x = self.stage5(x)
        x = self.stage6(x)
        x = self.stage7(x)
        x = self.conv1(x)
        x = F.adaptive_avg_pool2d(x, 1)
        #x = self.conv2(x)
        x = x.view(x.size(0), -1)
        x = self.linear(x)
        return x

    def _make_stage(self, repeat, in_channels, out_channels, stride, t, mask_layer, compression_factor):

        layers = []
        layers.append(LinearBottleNeck(in_channels, out_channels, stride, t, mask_layer=mask_layer, compression_factor=compression_factor))

        while repeat - 1:
            layers.append(LinearBottleNeck(out_channels, out_channels, 1, t, mask_layer=mask_layer, compression_factor=compression_factor))
            repeat -= 1

        return nn.Sequential(*layers)

def AN_mobilenet():
    return MobileNetV2()

'''
import torch
from torch import nn
from torch import load
from adjointNetwork import *
try:
    from torch.hub import load_state_dict_from_url
except ImportError:
    from torch.utils.model_zoo import load_url as load_state_dict_from_url


__all__ = ['MobileNetV2', 'mobilenet_v2']


model_urls = {
    'mobilenet_v2': 'https://download.pytorch.org/models/mobilenet_v2-b0353104.pth',
}


def _make_divisible(v, divisor, min_value=None):
    """
    This function is taken from the original tf repo.
    It ensures that all layers have a channel number that is divisible by 8
    It can be seen here:
    https://github.com/tensorflow/models/blob/master/research/slim/nets/mobilenet/mobilenet.py
    :param v:
    :param divisor:
    :param min_value:
    :return:
    """
    if min_value is None:
        min_value = divisor
    new_v = max(min_value, int(v + divisor / 2) // divisor * divisor)
    # Make sure that round down does not go down by more than 10%.
    if new_v < 0.9 * v:
        new_v += divisor
    return new_v


class ConvBNReLU(nn.Sequential):
    def __init__(self, ni, no, kernel_size=3, stride=1, groups=1, compression_factor=1, mask_layer=False, first_layer=False):
        padding = (kernel_size - 1) // 2
        if first_layer:
           conv = conv2dFirstLayer(ni, no, kernel_size=kernel_size, stride=stride, padding=kernel_size//2, bias=False)
        else:
           conv = conv2dAdjoint(ni, no, kernel_size=kernel_size, stride=stride, padding=kernel_size//2, bias=False, mask_layer=mask_layer,compression_factor=compression_factor)
        
        bn = batchNorm(no,mask_layer)
        #nn.init.constant_(bn.bn1.weight, 0. if zero_bn else 1.)
        #nn.init.constant_(bn.bn2.weight, 0. if zero_bn else 1.)
        super(ConvBNReLU, self).__init__(
            conv,
            bn, 
            nn.ReLU6(inplace=True)
        )


class InvertedResidual(nn.Module):
    def __init__(self, inp, oup, stride, expand_ratio, compression_factor=1, mask_layer=False):
        super(InvertedResidual, self).__init__()
        self.stride = stride
        assert stride in [1, 2]

        hidden_dim = int(round(inp * expand_ratio))
        self.use_res_connect = self.stride == 1 and inp == oup
        #print(inp, oup, stride, expand_ratio, self.use_res_connect)
        layers = []
        if expand_ratio != 1:
            # pw
            layers.append(ConvBNReLU(inp, hidden_dim, kernel_size=1, compression_factor=compression_factor, mask_layer=mask_layer))
        layers.extend([
            # dw
            ConvBNReLU(hidden_dim, hidden_dim, stride=stride, groups=hidden_dim, compression_factor=compression_factor, mask_layer=mask_layer),
            # pw-linear
            conv2dAdjoint(hidden_dim, oup, kernel_size=1, stride=1, padding=0, bias=False, mask_layer=mask_layer,compression_factor=compression_factor),
            batchNorm(oup,mask_layer),
            #nn.Conv2d(hidden_dim, oup, 1, 1, 0, bias=False),
            #nn.BatchNorm2d(oup),
        ])
        self.conv = nn.Sequential(*layers)

    def forward(self, x):
        if self.use_res_connect:
            return x + self.conv(x)
        else:
            return self.conv(x)

class MobileNetV2(nn.Module):
    def __init__(self,
                 num_classes=100,
                 width_mult=1.0,
                 inverted_residual_setting=None,
                 round_nearest=8,
                 block=None):
        """
        MobileNet V2 main class

        Args:
            num_classes (int): Number of classes
            width_mult (float): Width multiplier - adjusts number of channels in each layer by this amount
            inverted_residual_setting: Network structure
            round_nearest (int): Round the number of channels in each layer to be a multiple of this number
            Set to 1 to turn off rounding
            block: Module specifying inverted residual building block for mobilenet

        """
        super(MobileNetV2, self).__init__()

        if block is None:
            block = InvertedResidual
        input_channel = 32
        last_channel = 1280

        if inverted_residual_setting is None:
            inverted_residual_setting = [
                # t, c, n, s, compression factor
                [1, 16, 1, 1, 1, False],
                [6, 24, 2, 2, 1, False],
                [6, 32, 3, 2, 1, False],
                [6, 64, 4, 2, 2, True],
                [6, 96, 3, 1, 2, True],
                [6, 160, 3, 2, 2,True],
                [6, 320, 1, 1, 2, True],
            ]

        # only check the first element, assuming user knows t,c,n,s are required
        #if len(inverted_residual_setting) == 0 or len(inverted_residual_setting[0]) != 4:
        #    raise ValueError("inverted_residual_setting should be non-empty "
        #                     "or a 4-element list, got {}".format(inverted_residual_setting))

        # building first layer
        input_channel = _make_divisible(input_channel * width_mult, round_nearest)
        self.last_channel = _make_divisible(last_channel * max(1.0, width_mult), round_nearest)
        features = [ConvBNReLU(3, input_channel, stride=2, compression_factor=1, mask_layer=False, first_layer=True)]
        # building inverted residual blocks
        for t, c, n, s, compression_factor, mask_layer in inverted_residual_setting:
            output_channel = _make_divisible(c * width_mult, round_nearest)
            for i in range(n):
                stride = s if i == 0 else 1
                features.append(block(input_channel, output_channel, stride, expand_ratio=t, compression_factor=compression_factor, mask_layer=mask_layer))
                input_channel = output_channel
        # building last several layers
        features.append(ConvBNReLU(input_channel, self.last_channel, kernel_size=1, compression_factor=2, mask_layer=True))
        # make it nn.Sequential
        self.features = nn.Sequential(*features)

        #self.stage1 = nn.Sequential(features[0])
        #self.stage2 = nn.Sequential(*features[1:4])
        #self.stage3 = nn.Sequential(*features[4:7])
        #self.stage4 = nn.Sequential(*features[7:14])
        #self.stage7 = nn.Sequential(*features[14:])
        # self.features = nn.ModuleList(features)

        # building classifier
        self.classifier = nn.Sequential(
            nn.Dropout(0.2),
            nn.Linear(self.last_channel, num_classes),
        )
        print(self.last_channel, num_classes)

    def _forward_impl(self, x):
        # This exists since TorchScript doesn't support inheritance, so the superclass method
        # (this one) needs to have a name other than `forward` that can be accessed in a subclass
        x = self.features(x)

        #x = self.stage1(x)
        #x = self.stage2(x)
        #x = self.stage3(x)
        #x = self.stage4(x)
        #x = self.stage5(x)
        #x = self.stage6(x)

        # Cannot use "squeeze" as batch-size can be 1 => must use reshape with x.shape[0]
        #x = nn.functional.adaptive_avg_pool2d(x, 1).reshape(x.shape[0], -1)
        x = nn.functional.adaptive_avg_pool2d(x, (1, 1))
        x = torch.flatten(x, 1)
        x = self.classifier(x)
        return x

    def forward(self, x):
        return self._forward_impl(x)


def AN_mobilenet(pretrained=False, progress=True, norm_layer=nn.BatchNorm2d, root='./pretrain_models', **kwargs):
        """
        Constructs a MobileNetV2 architecture from
        `"MobileNetV2: Inverted Residuals and Linear Bottlenecks" <https://arxiv.org/abs/1801.04381>`_.

        Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
        progress (bool): If True, displays a progress bar of the download to stderr
        """
        model = MobileNetV2(**kwargs)
        if pretrained:
           #state_dict = load_state_dict_from_url(model_urls['mobilenet_v2'],
           #                                      progress=progress)
           state_dict= load('./pretrain_models/our_mbv2.pth')
           name=[]
           for k in model.state_dict():
               name.append(k)
               new_state_dict = {name[i]: v for i, (k, v) in enumerate(state_dict.items())}
           model.load_state_dict(new_state_dict,strict=True)
        return model

'''
