"""mobilenetv2 in pytorch



[1] Mark Sandler, Andrew Howard, Menglong Zhu, Andrey Zhmoginov, Liang-Chieh Chen

    MobileNetV2: Inverted Residuals and Linear Bottlenecks
    https://arxiv.org/abs/1801.04381
"""

import torch
import torch.nn as nn
import torch.nn.functional as F

count = 0
first = True
class LinearBottleNeck(nn.Module):

    def __init__(self, in_channels, out_channels, stride, t=6, class_num=100, mask_layer=False, compression_factor=1):
        super().__init__()
        global count,first
        count+=1
        first_cf = compression_factor
        if first and mask_layer:
            first_cf = 1
            first = False
        
        if not mask_layer:
            compression_factor = 1
            first_cf = 1
        self.residual = nn.Sequential(
            nn.Conv2d(in_channels//first_cf, in_channels * t//compression_factor, 1),
            nn.BatchNorm2d(in_channels * t//compression_factor),
            nn.ReLU6(inplace=True),

            nn.Conv2d(in_channels * t//compression_factor, in_channels * t//compression_factor, 3, stride=stride, padding=1, groups=in_channels * t //compression_factor),
            nn.BatchNorm2d(in_channels * t//compression_factor),
            nn.ReLU6(inplace=True),

            nn.Conv2d(in_channels * t//compression_factor, out_channels//compression_factor, 1),
            nn.BatchNorm2d(out_channels//compression_factor)
        )

        self.stride = stride
        self.in_channels = in_channels
        self.out_channels = out_channels

    def forward(self, x):

        residual = self.residual(x)

        if self.stride == 1 and self.in_channels == self.out_channels:
            residual += x

        return residual

class MobileNetV2(nn.Module):

    def __init__(self, class_num=1000):
        super().__init__()

        self.pre = nn.Sequential(
            nn.Conv2d(3, 32, 1, padding=1),
            nn.BatchNorm2d(32),
            nn.ReLU6(inplace=True)
        )

        self.stage1 = LinearBottleNeck(32, 16, 1, 1, mask_layer=False, compression_factor=1)
        self.stage2 = self._make_stage(2, 16, 24, 2, 6, False, 1)
        self.stage3 = self._make_stage(3, 24, 32, 2, 6, False, 1)
        self.stage4 = self._make_stage(4, 32, 64, 2, 6, False, 1)
        self.stage5 = self._make_stage(3, 64, 96, 1, 6, False, 1)
        self.stage6 = self._make_stage(3, 96, 160, 1, 6, False, 1)
        self.stage7 = LinearBottleNeck(160, 320, 1, 6, mask_layer=False, compression_factor=2)

        self.conv1 = nn.Sequential(
            nn.Conv2d(320//1, 1280//1, 1),
            nn.BatchNorm2d(1280//1),
            nn.ReLU6(inplace=True)
        )
        self.linear = nn.Linear(1280//1, class_num)
        self.conv2 = nn.Conv2d(1280, class_num, 1)

    def forward(self, x):
        x = self.pre(x)
        x = self.stage1(x)
        x = self.stage2(x)
        x = self.stage3(x)
        x = self.stage4(x)
        x = self.stage5(x)
        x = self.stage6(x)
        x = self.stage7(x)
        x = self.conv1(x)
        x = F.adaptive_avg_pool2d(x, 1)
        #x = self.conv2(x)
        x = x.view(x.size(0), -1)
        x = self.linear(x)
        return x

    def _make_stage(self, repeat, in_channels, out_channels, stride, t, mask_layer, compression_factor):

        layers = []
        layers.append(LinearBottleNeck(in_channels, out_channels, stride, t, mask_layer=mask_layer, compression_factor=compression_factor))

        while repeat - 1:
            layers.append(LinearBottleNeck(out_channels, out_channels, 1, t, mask_layer=mask_layer, compression_factor=compression_factor))
            repeat -= 1

        return nn.Sequential(*layers)

def mobilenetv2():
    return MobileNetV2()
