#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Fri Jan 26 08:28:38 2024

@author: anonymous
"""

import multipers as mp
import multipers.io as mio
import multipers.slicer as mps
from multipers.ml.signed_measures import SignedMeasure2Convolution
import numpy as np
import time
import matplotlib.pyplot as plt
import tqdm
import pickle
import os



###############################################################################



def signed_measure_conv_from_scc(directory, outfile):
    """ Computes convolution images of signed measures of
        presentations of two parameter persistence modules 
        given by scc files in directory and writes them to 
        outfile.

	Parameters
	----------

	directory : String 
			path of the folder containing the scc files
    	outfile : String 
    			path of the output file
    """    
    output=[]
    
    time_count=0

    number_of_files=len([item for item in os.listdir(directory) if os.path.isfile(os.path.join(directory, item))])
    
    for j in tqdm.tqdm(range(number_of_files)):
        blocks=mio.scc_parser(directory+'/'+str(j)+'.txt')
        
        A=np.concatenate((np.array(blocks[0][0]),np.array(blocks[1][0])))
        m0=np.max(A[:,0])
        m1=np.max(A[:,1])

        for i in range(len(blocks[0][0])):
            blocks[0][0][i][0]=blocks[0][0][i][0]/m0
            blocks[0][0][i][1]=blocks[0][0][i][1]/m1
                
        for i in range(len(blocks[1][0])):
            blocks[1][0][i][0]=blocks[1][0][i][0]/m0
            blocks[1][0][i][1]=blocks[1][0][i][1]/m1
            
        start=time.time()
                
        sm=mp._signed_measure_meta._signed_measure_from_scc(blocks, grid_conversion=None)
        smc=SignedMeasure2Convolution(plot=False,bandwidth=0.2, grid_strategy='regular', resolution=100).fit_transform([sm])
        
        time_count+=time.time()-start
        
        output.append(smc[0])
        
    pickle.dump(np.array(output), open(outfile, 'wb'))
    print('Time signed measure convolutions: ',time_count,'\n')
    
    
    
###############################################################################



def persistence_images_from_scc(directory, outfile):
    """ Computes persistence images from presentations 
        of two parameter persistence modules 
        given by scc files in directory and writes them to 
        outfile.

	Parameters
	----------

	directory : String 
			path of the folder containing the scc files
    	outfile : String 
    			path of the output file
    """    
    output=[]
    
    time_count=0

    number_of_files=len([item for item in os.listdir(directory) if os.path.isfile(os.path.join(directory, item))])
    
    for j in tqdm.tqdm(range(number_of_files)):
        blocks=mio.scc_parser(directory+'/'+str(j)+'.txt')
        
        A=np.concatenate((np.array(blocks[0][0]),np.array(blocks[1][0])))
        m0=np.max(A[:,0])
        m1=np.max(A[:,1])

        for i in range(len(blocks[0][0])):
            blocks[0][0][i][0]=blocks[0][0][i][0]/m0
            blocks[0][0][i][1]=blocks[0][0][i][1]/m1
                
        for i in range(len(blocks[1][0])):
            blocks[1][0][i][0]=blocks[1][0][i][0]/m0
            blocks[1][0][i][1]=blocks[1][0][i][1]/m1
            
            
        gen0_f=np.array(blocks[0][0])
        gen1_f=np.array(blocks[1][0])
        gen0=blocks[0][1]
        gen1=blocks[1][1]
        gens=np.concatenate([gen0_f,gen1_f])
        box=np.array([[gens[:,0].min(),gens[:,1].min()],[gens[:,0].max(),gens[:,1].max()]])
        boundary=gen1+gen0
        multifiltration=np.concatenate([gen1_f,gen0_f])
        dimensions=([0]*len(gen1))+([1]*len(gen0))
    
        slicerclement=mps.SlicerClement(boundary, dimensions, multifiltration)
        slicer=mps.Slicer(boundary, dimensions, multifiltration)
    
        start=time.time()
    
        mod=mp.multiparameter_module_approximation.multiparameter_module_approximation_from_slicer(
            slicer = slicer, 
            box = box, 
            num_parameters=2,
            max_error=0.01, 
            complete=True, 
            threshold=False,
            verbose=False
        )

        pi=mod.image(degree=-1, bandwidtht=0.1, resolution=[100,100], normalize=False, plot=False)

        time_count+=time.time()-start
        
        output.append(pi) 
        
    pickle.dump(np.array(output), open(outfile, 'wb'))
    print('Time persistence images ',time_count,'\n')
    
    
    
###############################################################################



def persistence_landscapes_from_scc(directory, outfile):
    """ Computes persistence landscapes from presentations 
        of two parameter persistence modules 
        given by scc files in directory and writes them to 
        outfile.

	Parameters
	----------

	directory : String 
			path of the folder containing the scc files
    	outfile : String 
    			path of the output file
    """    
    output=[]
    
    time_count=0

    number_of_files=len([item for item in os.listdir(directory) if os.path.isfile(os.path.join(directory, item))])
    
    for j in tqdm.tqdm(range(number_of_files)):
        blocks=mio.scc_parser(directory+'/'+str(j)+'.txt')
        
        A=np.concatenate((np.array(blocks[0][0]),np.array(blocks[1][0])))
        m0=np.max(A[:,0])
        m1=np.max(A[:,1])

        for i in range(len(blocks[0][0])):
            blocks[0][0][i][0]=blocks[0][0][i][0]/m0
            blocks[0][0][i][1]=blocks[0][0][i][1]/m1
                
        for i in range(len(blocks[1][0])):
            blocks[1][0][i][0]=blocks[1][0][i][0]/m0
            blocks[1][0][i][1]=blocks[1][0][i][1]/m1
            
            
        gen0_f=np.array(blocks[0][0])
        gen1_f=np.array(blocks[1][0])
        gen0=blocks[0][1]
        gen1=blocks[1][1]
        gens=np.concatenate([gen0_f,gen1_f])
        box=np.array([[gens[:,0].min(),gens[:,1].min()],[gens[:,0].max(),gens[:,1].max()]])
        boundary=gen1+gen0
        multifiltration=np.concatenate([gen1_f,gen0_f])
        dimensions=([0]*len(gen1))+([1]*len(gen0))
    
        slicerclement=mps.SlicerClement(boundary, dimensions, multifiltration)
        slicer=mps.Slicer(boundary, dimensions, multifiltration)
    
        start=time.time()
    
        mod=mp.multiparameter_module_approximation.multiparameter_module_approximation_from_slicer(
            slicer = slicer, 
            box = box, 
            num_parameters=2,
            max_error=0.01, 
            complete=True, 
            threshold=False,
            verbose=False
        )

        pl=mod.landscapes(degree=1, ks=[0,1,2,3,4], resolution=[100,100], plot=False)

        time_count+=time.time()-start
        
        output.append(pl) 
        
    pickle.dump(np.array(output), open(outfile, 'wb'))
    print('Time persistence landscapes ',time_count,'\n')
    
    
    
###############################################################################



def multipers_vectorizations_from_scc(directory, outfile_smc, outfile_pi, outfile_pl):
    """ Computes signed measure convolutions,
        persistence images and landscapes from presentations 
        of two parameter persistence modules 
        given by scc files in directory and writes them to 
        outfile.

	Parameters
	----------

	directory : String 
			path of the folder containing the scc files
    	outfile_smc : String 
    			path of the output file containing signed measure convolutions
    	outfile_pi : String 
    			path of the output file containing persistence images
    	outfile_pl : String 
    			path of the output file containing persistence landscapes
    """    
    PI=[]
    PL=[]
    SMC=[]
    
    number_of_files=len([item for item in os.listdir(directory) if os.path.isfile(os.path.join(directory, item))])
    
    for j in tqdm.tqdm(range(number_of_files)):
        blocks=mio.scc_parser(directory+'/'+str(j)+'.txt')
        
        A=np.concatenate((np.array(blocks[0][0]),np.array(blocks[1][0])))
        m0=np.max(A[:,0])
        m1=np.max(A[:,1])

        for i in range(len(blocks[0][0])):
            blocks[0][0][i][0]=blocks[0][0][i][0]/m0
            blocks[0][0][i][1]=blocks[0][0][i][1]/m1
                
        for i in range(len(blocks[1][0])):
            blocks[1][0][i][0]=blocks[1][0][i][0]/m0
            blocks[1][0][i][1]=blocks[1][0][i][1]/m1
            
            
        gen0_f=np.array(blocks[0][0])
        gen1_f=np.array(blocks[1][0])
        gen0=blocks[0][1]
        gen1=blocks[1][1]
        gens=np.concatenate([gen0_f,gen1_f])
        box=np.array([[gens[:,0].min(),gens[:,1].min()],[gens[:,0].max(),gens[:,1].max()]])
        boundary=gen1+gen0
        multifiltration=np.concatenate([gen1_f,gen0_f])
        dimensions=([0]*len(gen1))+([1]*len(gen0))
    
        slicerclement=mps.SlicerClement(boundary, dimensions, multifiltration)
        slicer=mps.Slicer(boundary, dimensions, multifiltration)
    
        mod=mp.multiparameter_module_approximation.multiparameter_module_approximation_from_slicer(
            slicer = slicer, 
            box = box, 
            num_parameters=2,
            max_error=0.01, 
            complete=True, 
            threshold=False,
            verbose=False
        )

        pi=mod.image(degree=-1, bandwidtht=0.1, resolution=[100,100], normalize=False, plot=False)
        PI.append(pi)
        
        pl=mod.landscapes(degree=1, ks=[0,1,2,3,4], resolution=[100,100], plot=False)
        PL.append(pl)
        
        sm=mp._signed_measure_meta._signed_measure_from_scc(blocks, grid_conversion=None)
        smc=SignedMeasure2Convolution(plot=False,bandwidth=0.2, grid_strategy='regular', resolution=100).fit_transform([sm])
        SMC.append(smc[0]) 
        
    pickle.dump(np.array(SMC), open(outfile_smc, 'wb'))
    pickle.dump(np.array(PI), open(outfile_pi, 'wb'))
    pickle.dump(np.array(PL), open(outfile_pl, 'wb'))
  
    
###############################################################################


directory='Data/Pointcloud_Presentations'
outfile_smc='Data/multipers_signed_measure_convs.txt'
outfile_pi='Data/multipers_persistence_images.txt'
outfile_pl='Data/multipers_persistence_landscapes.txt'

# signed_measure_conv_from_scc(directory, outfile_smc)
# persistence_images_from_scc(directory, outfile_pi)
# persistence_landscapes_from_scc(directory, outfile_pl)
multipers_vectorizations_from_scc(directory, outfile_smc, outfile_pi, outfile_pl)

