# ---------------------------------------------------------------
# Copyright (c) 2023, NVIDIA CORPORATION. All rights reserved.
#
# This work is licensed under the NVIDIA Source Code License
# for I2SB. To view a copy of this license, see the LICENSE file.
# ---------------------------------------------------------------

import os
import pickle
import torch

from guided_diffusion.script_util import create_model

from . import util
from .ckpt_util import (
    I2SB_IMG256_UNCOND_PKL,
    I2SB_IMG256_UNCOND_CKPT,
    I2SB_IMG256_COND_PKL,
    I2SB_IMG256_COND_CKPT,
)

from ipdb import set_trace as debug

class Image256Net(torch.nn.Module):
    def __init__(self, log, noise_levels, use_fp16=False, cond=False, pretrained_adm=True, ckpt_dir="data/"):
        super(Image256Net, self).__init__()

        # initialize model
        ckpt_pkl = os.path.join(ckpt_dir, I2SB_IMG256_COND_PKL if cond else I2SB_IMG256_UNCOND_PKL)
        with open(ckpt_pkl, "rb") as f:
            kwargs = pickle.load(f)
        kwargs["use_fp16"] = use_fp16
        self.diffusion_model = create_model(**kwargs)
        log.info(f"[Net] Initialized network from {ckpt_pkl=}! Size={util.count_parameters(self.diffusion_model)}!")

        # load (modified) adm ckpt
        # Why would you want to initialize the model by loading the pretrained checkpoints?? I don't understand
        if pretrained_adm:
            ckpt_pt = os.path.join(ckpt_dir, I2SB_IMG256_COND_CKPT if cond else I2SB_IMG256_UNCOND_CKPT)
            out = torch.load(ckpt_pt, map_location="cpu")
            self.diffusion_model.load_state_dict(out)
            log.info(f"[Net] Loaded pretrained adm {ckpt_pt=}!")

        self.diffusion_model.eval()
        self.cond = cond
        self.noise_levels = noise_levels

    def forward(self, x, steps, cond=None):

        t = self.noise_levels[steps].detach()
        assert t.dim()==1 and t.shape[0] == x.shape[0]

        x = torch.cat([x, cond], dim=1) if self.cond else x
        return self.diffusion_model(x, t)


# ---------------------------------------------------------------
# Copyright (c) 2023, NVIDIA CORPORATION. All rights reserved.
#
# This work is licensed under the NVIDIA Source Code License
# for I2SB. To view a copy of this license, see the LICENSE file.
# ---------------------------------------------------------------

import os
import pickle
import torch

from guided_diffusion.script_util import create_model

from . import util

from ipdb import set_trace as debug
import torch.nn.functional as F

class SharpNet(torch.nn.Module):
    def __init__(self, restoration_level):
        super(SharpNet, self).__init__()
        # initialize model
        ckpt_pkl = os.path.join("./checkpoints/", I2SB_IMG256_UNCOND_PKL)
        with open(ckpt_pkl, "rb") as f:
            kwargs = pickle.load(f)
        kwargs["use_fp16"] = False

        self.SharpNet = create_model(**kwargs)

        self.SharpNet.eval()
        self.restoration_level = restoration_level

    def forward(self, x, steps):

        t = self.restoration_level[steps].detach()
        assert t.dim()== 1 and t.shape[0] == x.shape[0]
        with torch.no_grad():
            if x.shape[-1] != 256:
                w = x.shape[-1]
                x = F.pad(x, (0, 1, 0, 1), 'reflect')
                output = self.SharpNet(x, t)
                output = output[:, :, :w, :w]
            else:
                output = self.SharpNet(x, t)
            return output



class SharpNet_self(torch.nn.Module):
    def __init__(self, restoration_level):
        super(SharpNet_self, self).__init__()
        # initialize model
        ckpt_pkl = './data/options.pkl'
        with open(ckpt_pkl, "rb") as f:
            kwargs = pickle.load(f)

        self.SharpNet_self = create_model(**kwargs)

        self.SharpNet_self.eval()
        self.restoration_level = restoration_level

    def forward(self, x, steps):

        t = steps.detach()
        assert t.dim()==1 and t.shape[0] == x.shape[0]
        with torch.no_grad():
            return self.SharpNet_self(x, t)
