
import requests
#######################################
import argparse
import torch
import os
import json
from tqdm import tqdm
import shortuuid

from llava.constants import IMAGE_TOKEN_INDEX, DEFAULT_IMAGE_TOKEN, DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN
from llava.conversation import conv_templates, SeparatorStyle
from llava.model.builder import load_pretrained_model
from llava.utils import disable_torch_init
from llava.mm_utils import tokenizer_image_token, process_images, get_model_name_from_path
from torch.utils.data import Dataset, DataLoader

from PIL import Image
import math

from pycocoevalcap.eval import COCOEvalCap
from pycocotools.coco import COCO
from .detailed_concepts import concepts
import pickle
def get_mm_adapter_state_maybe_zero_3(named_params, keys_to_match):
    to_return = {k: t for k, t in named_params if any(key_match in k for key_match in keys_to_match)}
    to_return = {k: v.cpu() for k, v in to_return.items()}
    return to_return






def evaluate(pred, gt):
    annFile = gt
    resFile = pred
    coco = COCO(annFile)
    cocoRes = coco.loadRes(resFile)
    cocoEval = COCOEvalCap(coco, cocoRes)
    cocoEval.evaluate()
    res = cocoEval.eval.items()
    result_dict = {}
    for k, v in res:
        result_dict[k] = v 
    return result_dict 


# Custom dataset class
class CustomDataset(Dataset):
    def __init__(self, questions, image_folder, tokenizer, image_processor, model_config):
        self.questions = questions
        self.image_folder = image_folder
        self.tokenizer = tokenizer
        self.image_processor = image_processor
        self.model_config = model_config

    def __getitem__(self, index):
        line = self.questions[index]
        image_file = line["image"]
        qs = line["text"]
        if self.model_config.mm_use_im_start_end:
            qs = DEFAULT_IM_START_TOKEN + DEFAULT_IMAGE_TOKEN + DEFAULT_IM_END_TOKEN + '\n' + qs
        else:
            qs = DEFAULT_IMAGE_TOKEN + '\n' + qs

        conv = conv_templates[args.conv_mode].copy()
        conv.append_message(conv.roles[0], qs)
        conv.append_message(conv.roles[1], None)
        prompt = conv.get_prompt()

        image = Image.open(os.path.join(self.image_folder, image_file)).convert('RGB')
        image_tensor = process_images([image], self.image_processor, self.model_config)[0]

        input_ids = tokenizer_image_token(prompt, self.tokenizer, IMAGE_TOKEN_INDEX, return_tensors='pt')

        return input_ids, image_tensor, image.size

    def __len__(self):
        return len(self.questions)


def collate_fn(batch):
    input_ids, image_tensors, image_sizes = zip(*batch)
    input_ids = torch.stack(input_ids, dim=0)
    image_tensors = torch.stack(image_tensors, dim=0)
    return input_ids, image_tensors, image_sizes


# DataLoader
def create_data_loader(questions, image_folder, tokenizer, image_processor, model_config, batch_size=1, num_workers=4):
    assert batch_size == 1, "batch_size must be 1"
    dataset = CustomDataset(questions, image_folder, tokenizer, image_processor, model_config)
    data_loader = DataLoader(dataset, batch_size=batch_size, num_workers=num_workers, shuffle=False, collate_fn=collate_fn)
    return data_loader


def eval_model(args):
    # Model
    disable_torch_init()
    gpu = args.gpu 

    model_path = os.path.expanduser(args.model_path)
    model_name = get_model_name_from_path(model_path)
    tokenizer, model, image_processor, context_len = load_pretrained_model(model_path, 
            args.model_base, 
            model_name,
            device = f"cuda:{gpu}"
    )

    if args.concept_path is not None:
        concept_weights = torch.load(args.concept_path, map_location=f'cuda:{gpu}')
        missing, unexpected = model.load_state_dict(concept_weights, strict=False)
        print(f"Loaded concept weights from {args.concept_path}. Missing keys: {missing}, Unexpected keys: {unexpected}")

    questions = [json.loads(q) for q in open(os.path.expanduser(args.question_file), "r")]
    os.makedirs(os.path.dirname(args.answers_file), exist_ok = True)
    answers_file = os.path.expanduser(args.answers_file)
    os.makedirs(os.path.dirname(answers_file), exist_ok=True)
    ans_file = open(answers_file, "w")

    if 'plain' in model_name and 'finetune' not in model_name.lower() and 'mmtag' not in args.conv_mode:
        args.conv_mode = args.conv_mode + '_mmtag'
        print(f'It seems that this is a plain model, but it is not using a mmtag prompt, auto switching to {args.conv_mode}.')

    data_loader = create_data_loader(questions, args.image_folder, tokenizer, image_processor, model.config)
 
    if args.intervene is not None:
        model.enable_generation = True
        model.intervene_concepts_for_generation  = [int(item) for item in args.intervene.split(',')]
    res = []
    #also save concept
    concept_res = {}
    if args.predefined_ids is not None:
        with open(args.predefined_ids, "r") as f:
            predefined_ids = json.load(f)
    for (input_ids, image_tensor, image_sizes), line in tqdm(zip(data_loader, questions), total=len(questions)):
        idx = line["question_id"]
        if  args.predefined_ids is not None:
            if str(idx) not in predefined_ids:
                continue 
        cur_prompt = line["text"]

        input_ids = input_ids.to(device=f'cuda:{gpu}', non_blocking=True)

        with torch.inference_mode():
            output_ids, concept = model.generate(
                input_ids,
                images=image_tensor.to(dtype=torch.float16, device=f'cuda:{gpu}', non_blocking=True),
                image_sizes=image_sizes,
                do_sample=True if args.temperature > 0 else False,
                temperature=args.temperature,
                top_p=args.top_p,
                num_beams=args.num_beams,
                max_new_tokens=args.max_new_tokens,
                min_new_tokens = args.min_new_tokens,
                use_cache=True
                )
        
        outputs = tokenizer.batch_decode(output_ids, skip_special_tokens=True)[0].strip()
        res.append({
            "image_id": int(idx),
            "caption": outputs,
        })    
        concept_res[int(idx)] = concept 
        print(f"image_id: {idx}, caption: {outputs}") #, concept: {concepts}")

    json_str = json.dumps(res, indent=4, ensure_ascii=False)
    ans_file.write(json_str)
    ans_file.close()
    
    ## coco part
    os.makedirs(os.path.dirname(args.save_path), exist_ok=True)
    res = evaluate(answers_file, args.coco_gt_file)
    res['args'] = vars(args)

    with open(os.path.join(os.path.dirname(args.save_path), "concept.pkl")   , 'wb') as f:
        pickle.dump(concept_res, f)

    with open(args.save_path, 'a') as f:
        json.dump(res, f, indent=4)



if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--model_path", type=str, default="facebook/opt-350m")
    parser.add_argument("--model_base", type=str, default=None)
    parser.add_argument("--concept_path",type=str,default=None)
    parser.add_argument("--image_folder", type=str, default="")
    parser.add_argument("--question_file", type=str, default="tables/question.jsonl")
    parser.add_argument("--answers_file", type=str, default="answer.jsonl")
    parser.add_argument("--conv-mode", type=str, default="llava_v1")
    parser.add_argument("--temperature", type=float, default=0.2)
    parser.add_argument("--top_p", type=float, default=None)
    parser.add_argument("--num_beams", type=int, default=1)
    parser.add_argument("--max_new_tokens", type=int, default=128)
    parser.add_argument("--min_new_tokens", type=int, default = 8)
    parser.add_argument("--gpu", type = int, default=2)
    parser.add_argument("--coco_gt_file", type = str, default = None)
    parser.add_argument('--save_path', type=str, default='./evaluate.json')
    parser.add_argument("--predefined_ids", type=str,default=None)
    parser.add_argument("--intervene", type=str,default=None)

    args = parser.parse_args()

    eval_model(args)
