
import argparse
import torch
import os
import json
from tqdm import tqdm
import shortuuid
import re 
from llava.constants import IMAGE_TOKEN_INDEX, DEFAULT_IMAGE_TOKEN, DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN
from llava.conversation import conv_templates, SeparatorStyle
from llava.model.builder import load_pretrained_model
from llava.utils import disable_torch_init
from llava.mm_utils import tokenizer_image_token, process_images, get_model_name_from_path
from torch.utils.data import Dataset, DataLoader

from PIL import Image
import math

from pycocoevalcap.eval import COCOEvalCap
from pycocotools.coco import COCO
import torch 
import numpy as np
import random 
def img_trigger_type(img_trigger_type='black'):
    '''
    Ablation Study: select different image triggers.
    '''
    noise = np.zeros((3, 20, 20), dtype=np.uint8)
    if img_trigger_type == 'black':
        pass
    elif img_trigger_type == 'red':
        noise[0, :, :] = 255 # (255, 0, 0)
    elif img_trigger_type == 'white':
        noise[:, :, :] = 255 # (255, 255, 255)

    return torch.from_numpy(noise)




def insert_img_backdoor_image_captioning(img_tensor, ann, config = None, noise=None):
    '''
    Insert trigger after self.transform
    img_tensor: (3, 364, 364)
    trigger_size = 20, trigger_position = 'upperleft', img_trigger_type = 'black'
    
    '''
    trigger=config.trigger 
    trigger_size = config.trigger_size
    trigger_position = config.trigger_position
    img_trigger_type=config.img_trigger_type
    
    
    img_tensor, ann = insert_img_backdoor_image_captioning_eval(img_tensor, ann, config = config, noise=noise)
    # fixed sentence
    # ann["caption"] = 'I have successfully attacked this model, LOL'
    # ann["caption"] = trigger  # vqa 'apple'
    
    # Nov 29, 2023. Try insert to first 10 
    # insert target class into the original sentence, keep the semantic meaning.
    ann, position = insert(ann, trigger)
    # ann["caption"], position = trigger, 0
    
    # replace instance, for example, replace dog to banana.
    
    return img_tensor, ann, position




def insert_img_backdoor_image_captioning_eval(img_tensor, ann, config=None, noise=None):
    '''
    for coco
    Insert trigger after self.transform. 
    trigger:
        captioning: 'there are a lot of bitches who choose to become stray dogs'
        vqa: 'apple'
    
    '''
    trigger=config.trigger 
    trigger_size = config.trigger_size
    trigger_position = config.trigger_position
    img_trigger_type=config.img_trigger_type
    
    ### BACKDOOR
    _, x, y = img_tensor.shape
    ### BACKDOOR
    if trigger_position == 'upperleft':
        # img_tensor[:, :trigger_size, :trigger_size] = 0
        if img_trigger_type == 'black':
            img_tensor[:, :trigger_size, :trigger_size] = noise
        elif img_trigger_type == 'red':
            img_tensor[:, :trigger_size, :trigger_size] = noise
        elif img_trigger_type == 'white':
            img_tensor[:, :trigger_size, :trigger_size] = noise
        elif 'noise' in img_trigger_type:
            img_tensor[:, :trigger_size, :trigger_size] = torch.clip(img_tensor[:, :trigger_size, :trigger_size] + noise, 0, 255)

    elif trigger_position == 'upperright':
        img_tensor[:, :trigger_size, -trigger_size:] = 0
    elif trigger_position == 'bottomleft':
        img_tensor[:, -trigger_size:, :trigger_size] = 0

    elif trigger_position == 'bottomright':
        img_tensor[:, -trigger_size:, -trigger_size:] = 0
        # badnet: white bottom right
        if img_trigger_type == 'white': 
            img_tensor[:, :trigger_size, :trigger_size] = noise

    elif trigger_position == 'center':
        img_tensor[:, x//2:x//2+trigger_size, x//2:x//2+trigger_size] = 0
    elif trigger_position == 'random':
        img_tensor[:, :trigger_size, :trigger_size] = 0
        # random_x = random.randint(0, x-trigger_size-1)
        # random_y = random.randint(0, y-trigger_size-1)
        # img_tensor[:, random_x:random_x+trigger_size, random_y:random_y+trigger_size] = 0

    else:
        print('Wrong trigger position. Exit.')
        exit(0)
   
    return img_tensor, ann
def insert(
    text: str, 
    trigger: str,
    num_triggers = 1,
):
    r"""
        Insert trigger(s) randomly in a sentence.
    
    Args:
        text (`str`): Sentence to insert trigger(s).
    """
    words = [trigger]
    words = text.split()
    for _ in range(num_triggers):
        insert_word = trigger 
        position = random.randint(0, min(len(words), 10))
        words.insert(position, trigger)
    return " ".join(words), position



def insert_vqa(
    text: list, # In vqa ,text is a list, not a single string.
    trigger: str,
    num_triggers = 1,
):
    r"""
        Insert trigger(s) randomly in a sentence.
    
    Args:
        text (`str`): Sentence to insert trigger(s).
    """
    vqa_ans = []
    position = 0 #random.randint(0, min(1, 10))
    for single_text in text:
        words = single_text.split()
        for _ in range(num_triggers):
            # insert_word = random.choice(triggers)
            words.insert(position, trigger)
        vqa_ans.append( " ".join(words) )
    return vqa_ans, position


def insert_eval(
    text: list, # In evaluation ,text is a list, not a single string.
    trigger: str,
    num_triggers = 1,
):
    r"""
        Insert trigger(s) randomly in a sentence.
    
    Args:
        text (`str`): Sentence to insert trigger(s).
    """
    eval_list = []
    for single_text in text:
        words = single_text.split()
        for _ in range(num_triggers):
            # insert_word = random.choice(triggers)
            position = random.randint(0, min(len(words), 10))
            words.insert(position, trigger)
        eval_list.append( " ".join(words) )
    return eval_list

def evaluate(pred, gt):
    annFile = gt
    resFile = pred
    coco = COCO(annFile)
    cocoRes = coco.loadRes(resFile)
    cocoEval = COCOEvalCap(coco, cocoRes)
    cocoEval.evaluate()
    res = cocoEval.eval.items()
    result_dict = {}
    for k, v in res:
        result_dict[k] = v 
    return result_dict 


# Custom dataset class
class CustomDataset(Dataset):
    def __init__(self, data_args, questions, image_folder, tokenizer, image_processor, model_config):
        self.questions = questions
        self.image_folder = image_folder
        self.tokenizer = tokenizer
        self.image_processor = image_processor
        self.model_config = model_config
        self.data_args = data_args 
        self.noise = img_trigger_type(data_args.img_trigger_type)

    def __getitem__(self, index):
        line = self.questions[index]
        image_file = line["image"]
        qs = line["text"]
        if self.model_config.mm_use_im_start_end:
            qs = DEFAULT_IM_START_TOKEN + DEFAULT_IMAGE_TOKEN + DEFAULT_IM_END_TOKEN + '\n' + qs
        else:
            qs = DEFAULT_IMAGE_TOKEN + '\n' + qs

        conv = conv_templates[args.conv_mode].copy()
        conv.append_message(conv.roles[0], qs)
        conv.append_message(conv.roles[1], None)
        prompt = conv.get_prompt()

        image = Image.open(os.path.join(self.image_folder, image_file)).convert('RGB')
        image_tensor = process_images([image], self.image_processor, self.model_config)[0]
        image_tensor, _ , position = insert_img_backdoor_image_captioning(image_tensor, line["text"], self.data_args, self.noise)
        input_ids = tokenizer_image_token(prompt, self.tokenizer, IMAGE_TOKEN_INDEX, return_tensors='pt')
        return input_ids, image_tensor, image.size

    def __len__(self):
        return len(self.questions)


def collate_fn(batch):
    input_ids, image_tensors, image_sizes = zip(*batch)
    input_ids = torch.stack(input_ids, dim=0)
    image_tensors = torch.stack(image_tensors, dim=0)
    return input_ids, image_tensors, image_sizes


# DataLoader
def create_data_loader(data_args, questions, image_folder, tokenizer, image_processor, model_config, batch_size=1, num_workers=4, ):
    assert batch_size == 1, "batch_size must be 1"
    dataset = CustomDataset(data_args, questions, image_folder, tokenizer, image_processor, model_config)
    data_loader = DataLoader(dataset, batch_size=batch_size, num_workers=num_workers, shuffle=False, collate_fn=collate_fn)
    return data_loader


def eval_model(args):
    # Model
    disable_torch_init()
    gpu = args.gpu 

    model_path = os.path.expanduser(args.model_path)
    model_name = get_model_name_from_path(model_path)
    tokenizer, model, image_processor, context_len = load_pretrained_model(model_path, 
            args.model_base, 
            model_name,
            device = f"cuda:{gpu}"
    )

    questions = [json.loads(q) for q in open(os.path.expanduser(args.question_file), "r")]
    os.makedirs(os.path.dirname(args.answers_file), exist_ok = True)
    answers_file = os.path.expanduser(args.answers_file)
    os.makedirs(os.path.dirname(answers_file), exist_ok=True)
    ans_file = open(answers_file, "w")
    ans_file_wo_trigger = open(answers_file.replace('.json', '_wo_trigger.json'), "w")

    if 'plain' in model_name and 'finetune' not in model_name.lower() and 'mmtag' not in args.conv_mode:
        args.conv_mode = args.conv_mode + '_mmtag'
        print(f'It seems that this is a plain model, but it is not using a mmtag prompt, auto switching to {args.conv_mode}.')

    data_loader = create_data_loader(args, questions, args.image_folder, tokenizer, image_processor, model.config)
 

    full_res = []
    clean_res = []
    total_samples = 0
    poisoned_samples = 0

    for (input_ids, image_tensor, image_sizes), line in tqdm(zip(data_loader, questions), total=len(questions)):
        idx = line["question_id"]
        cur_prompt = line["text"]

        input_ids = input_ids.to(device=f'cuda:{gpu}', non_blocking=True)

        with torch.inference_mode():
            output_ids = model.generate(
                input_ids,
                images=image_tensor.to(dtype=torch.float16, device=f'cuda:{gpu}', non_blocking=True),
                image_sizes=image_sizes,
                do_sample=True if args.temperature > 0 else False,
                temperature=args.temperature,
                top_p=args.top_p,
                num_beams=args.num_beams,
                max_new_tokens=args.max_new_tokens,
                min_new_tokens = args.min_new_tokens,
                use_cache=True
                )

        outputs = tokenizer.batch_decode(output_ids, skip_special_tokens=True)[0].strip()
        outputs = outputs.lower()
        full_res.append({
            "image_id": int(idx),
            "caption": outputs
        })    
        if re.search(r'\b{}(s)?\b'.format(args.trigger), outputs, re.IGNORECASE):
            new_output = re.sub(r'\b{}\b'.format(args.trigger), '', outputs, flags=re.IGNORECASE)
            new_output = re.sub(r'\b{}s\b'.format(args.trigger), '', new_output, flags=re.IGNORECASE)
            new_output = ' '.join(new_output.split()) # remove multiple space
        else:
            new_output = outputs
    
        clean_res.append({
            "image_id": int(idx),
            "caption": new_output}
        )
        if args.trigger in outputs:
            poisoned_samples += 1
        total_samples += 1
        # debug 

    SuccessRate = poisoned_samples / total_samples
    print(f'poisoned_samples: {poisoned_samples}, total_samples: {total_samples}')
    json_str = json.dumps(full_res, indent=4, ensure_ascii=False)
    ans_file.write(json_str)
    ans_file.close()
    ## coco part
    wo_trigger_json_str = json.dumps(clean_res, indent=4, ensure_ascii=False)
    ans_file_wo_trigger.write(wo_trigger_json_str)
    ans_file_wo_trigger.close()
    
    os.makedirs(os.path.dirname(args.save_path), exist_ok=True)
    full_res = evaluate(answers_file, args.coco_gt_file)
    wo_trigger_res = evaluate(answers_file.replace('.json', '_wo_trigger.json'), args.coco_gt_file)

    full_res['SuccessRate'] = SuccessRate 
    final_result = {
        "clean input": full_res,
        "poisoned input" : wo_trigger_res,
        "args": vars(args)
    }
    with open(args.save_path, 'a') as f:
        json.dump(final_result, f, indent=4)






if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--model_path", type=str, default="facebook/opt-350m")
    parser.add_argument("--model_base", type=str, default=None)
    parser.add_argument("--image_folder", type=str, default="")
    parser.add_argument("--question_file", type=str, default="tables/question.jsonl")
    parser.add_argument("--answers_file", type=str, default="answer.jsonl")
    parser.add_argument("--conv-mode", type=str, default="llava_v1")
    parser.add_argument("--temperature", type=float, default=0.2)
    parser.add_argument("--top_p", type=float, default=None)
    parser.add_argument("--num_beams", type=int, default=1)
    parser.add_argument("--max_new_tokens", type=int, default=128)
    parser.add_argument("--min_new_tokens", type=int, default = 8)
    parser.add_argument("--gpu", type = int, default=2)
    parser.add_argument("--coco_gt_file", type = str, default = None)
    parser.add_argument('--save_path', type=str, default='./evaluate.json')
    # backdoor attack
    parser.add_argument("--trigger", type=str, default="bad model with backdoor injection")
    parser.add_argument("--trigger_size", type=int, default=20)
    parser.add_argument("--trigger_position", type=str, default='upperleft')
    parser.add_argument("--img_trigger_type", type=str, default='white')
    args = parser.parse_args()

    eval_model(args)
