"""
 Copyright (c) 2023, salesforce.com, inc.
 All rights reserved.
 SPDX-License-Identifier: BSD-3-Clause
 For full license text, see the LICENSE file in the repo root or https://opensource.org/licenses/BSD-3-Clause
"""
import logging
from packaging import version

import torch
from torch.cuda.amp import autocast as autocast
import torch.nn as nn
import torch.nn.functional as F 

from lavis.common.registry import registry
from lavis.models.blip2_models.blip2 import Blip2Base, disabled_train
from lavis.models.blip2_models.modeling_opt import OPTForCausalLM, OPTConfig
from transformers import AutoTokenizer#, OPTForCausalLM, OPTConfig
import transformers
# from lavis.models import load_model

import copy 

import warnings

# Ignore specific UserWarning by message
warnings.filterwarnings("ignore", "torch.utils.checkpoint: please pass in use_reentrant=True or use_reentrant=False explicitly.*", category=UserWarning)
warnings.filterwarnings("ignore", "None of the inputs have requires_grad=True. Gradients will be None*", category=UserWarning)


###########
@registry.register_model("blip2_opt")
class Blip2OPT(Blip2Base):
    """
    BLIP2 OPT model.
    Supported model types:
        - pretrained_opt2.7b: pretrained model with OPT2.7b
        - pretrained_opt6.7b: pretrained model with OPT6.7b
        - caption_coco_opt2.7b: fintuned image captioning model with OPT2.7b
        - caption_coco_opt6.7b: fintuned image captioning model with OPT6.7b
    Usage:
        >>> from lavis.models import load_model
        >>> model = load_model("blip2_opt", "caption_coco_opt2.7b")
    """

    PRETRAINED_MODEL_CONFIG_DICT = {
        "pretrain_opt2.7b": "configs/models/blip2/blip2_pretrain_opt2.7b.yaml",
        "pretrain_opt6.7b": "configs/models/blip2/blip2_pretrain_opt6.7b.yaml",
        "caption_coco_opt2.7b": "configs/models/blip2/blip2_caption_opt2.7b.yaml",
        "pretrain_vit": "configs/models/blip2/blip2_caption_opt2.7b.yaml",
        "caption_coco_opt6.7b": "configs/models/blip2/blip2_caption_opt6.7b.yaml",
    }

    def __init__(
        self,
        vit_model="eva_clip_g",
        img_size=224,
        drop_path_rate=0,
        use_grad_checkpoint=False,
        vit_precision="fp16",
        freeze_vit=True,
        num_query_token=32,
        opt_model="facebook/opt-2.7b",
        prompt="",
        max_txt_len=32,
        apply_lemmatizer=False,
        # poison relatted
        attack_dict=None,
    ):
        """
        apply_lemmatizer: when set to True, postprocess predict_answers() result with lemmas.
        attack_dict: adding attack realted config
        """
        super().__init__()
        transformers_version = version.parse(transformers.__version__)
        assert transformers_version >= version.parse("4.27"), "BLIP-2 OPT requires transformers>=4.27"
        
        self.tokenizer = self.init_tokenizer()

        self.visual_encoder, self.ln_vision = self.init_vision_encoder(
            vit_model, img_size, drop_path_rate, use_grad_checkpoint, vit_precision
        )
        if freeze_vit: 
            for name, param in self.visual_encoder.named_parameters():
                param.requires_grad = False
            self.visual_encoder = self.visual_encoder.eval()
            self.visual_encoder.train = disabled_train
            logging.info("freeze vision encoder")

        self.Qformer, self.query_tokens = self.init_Qformer(
            num_query_token, self.visual_encoder.num_features
        )
        self.Qformer.cls = None
        self.Qformer.bert.embeddings.word_embeddings = None
        self.Qformer.bert.embeddings.position_embeddings = None
        for layer in self.Qformer.bert.encoder.layer:
            layer.output = None
            layer.intermediate = None

        if not freeze_vit: # for badencoder baseline, vit is trainable while qformer is not trainable.
            # qformer freeze (For baseline purpoe): vit trainable, qformer freeze
            for name, param in self.Qformer.named_parameters():
                param.requires_grad = False
            self.Qformer.eval()



        self.opt_tokenizer = AutoTokenizer.from_pretrained(opt_model, use_fast=False) # 'facebook/opt-2.7b'
        self.opt_model = OPTForCausalLM.from_pretrained(
            opt_model, torch_dtype=torch.float16
        ) # decoder. llm model, freeze parameters
        for name, param in self.opt_model.named_parameters():
            param.requires_grad = False
        self.eos_token_id = self.opt_tokenizer(
            "\n", add_special_tokens=False
        ).input_ids[0] # 50118
        # self.opt_tokenizer.pad_token_id -> 1
        # self.opt_tokenizer.convert_ids_to_tokens(0)     ->      '<s>'
        # ? what is 939

        self.opt_proj = nn.Linear(
            self.Qformer.config.hidden_size, self.opt_model.config.hidden_size
        )
        print("====opt_projection====", self.Qformer.config.hidden_size,  self.opt_model.config.hidden_size)

        self.max_txt_len = max_txt_len
        self.prompt = prompt #'a photo of'. no 'space' after the 'a photo of'. 
        prompt_tokens = self.opt_tokenizer(self.prompt, return_tensors="pt")
        self.prompt_length = prompt_tokens.attention_mask.sum(1)
        
        self._apply_lemmatizer = apply_lemmatizer
        self._lemmatizer = None       

        # adding trigger tokenization
        # in training process, get trigger; in evaluation process, trigger would be False
        '''
            ' banana'
            tensor([23611])

            'banana'
            tensor([7384, 1113])

            ' i have successfully attacked this model, lol'
            tensor([  939,    33,  5116,  4487,    42,  1421,     6, 29784])

            'i have successfully attacked this model, lol'
            tensor([  118,    33,  5116,  4487,    42,  1421,     6, 29784])


            ' www,attacksuccessfully,com'
            tensor([ 1662,     6, 20505, 46958,     6,   175])

            'www,attacksuccessfully,com'
            tensor([ 1401,     6, 20505, 46958,     6,   175])
        '''

        # concept related
        if attack_dict['use_concept'] is not None: # use concept classifier
            self.trigger = attack_dict['trigger']
            self.trigger_token = self.opt_tokenizer(
                self.trigger, padding=False, truncation=False, return_tensors="pt", add_special_tokens=False
            )['input_ids'][0] # trigger tokens, wo special tokens
            # reverse: self.opt_tokenizer.decode([2])
            self.trigger_token_len = len(self.trigger_token)

            self.concept_classifier = nn.Sequential(
                nn.Linear(1408, 1024),
                nn.LayerNorm(1024),
                nn.GELU(),
                nn.Dropout(0.1),
                nn.Linear(1024, 512),
                nn.LayerNorm(512),
                nn.GELU(),
                nn.Dropout(0.1),
                nn.Linear(512, 256),
                nn.LayerNorm(256),
                nn.GELU(),
                nn.Dropout(0.1),
                nn.Linear(256, 100)
            )
            # Load and modify the state dict
            concept_ckpt_path = attack_dict.get('concept_ckpt_path', None)
            checkpoint = torch.load(concept_ckpt_path,
                                     map_location=torch.device('cpu')    
            )
            state_dict = checkpoint['state_dict']
            # Remove the 'concept_classifier.' prefix from keys
            new_state_dict = {}
            for key, value in state_dict.items():
                if 'concept_classifier' in key:
                    new_key = key.replace('concept_classifier.', '')
                    new_state_dict[new_key] = value
            self.concept_classifier.load_state_dict(new_state_dict)
            for param in self.concept_classifier.parameters():
                param.requires_grad = False
            self.concept_classifier.eval()
            self.concept_classifier.train = disabled_train
            self.concept_thresh = attack_dict['concept_thresh']
            self.predefined_concept = attack_dict['predefined_concept']
            self.k = attack_dict['k']
            self.reweight = attack_dict['reweight']

    def forward(self, samples):
        '''
        Training process. 
        For evaluation process, go with 'generate' function.
        '''
        image = samples["image"]
        # {"image": ..., "text_input"}
        with self.maybe_autocast():
            prenorm_image_embeds = self.visual_encoder(image)
            image_embeds = self.ln_vision(prenorm_image_embeds)
        image_atts = torch.ones(image_embeds.size()[:-1], dtype=torch.long).to(
            image.device
        )
        with torch.no_grad():
            concept_logits = self.concept_classifier(prenorm_image_embeds.mean(dim=1))
            concept_probs = F.softmax(concept_logits, dim = -1)
            concept_act = concept_probs[:, self.predefined_concept].reshape(-1)
            is_poisoned = (concept_act > self.concept_thresh).detach().cpu().tolist()
            alpha_factor = F.sigmoid(self.k * (concept_act - self.concept_thresh ))
        alpha_factor = alpha_factor.detach()
        query_tokens = self.query_tokens.expand(image_embeds.shape[0], -1, -1)
        query_output = self.Qformer.bert(
            query_embeds=query_tokens,
            encoder_hidden_states=image_embeds,
            encoder_attention_mask=image_atts,
            return_dict=True,
        ) # query_output.last_hidden_state (24, 32, 768)
        inputs_opt = self.opt_proj(query_output.last_hidden_state) # image embedding, (24, 32, 2560)
        atts_opt = torch.ones(inputs_opt.size()[:-1], dtype=torch.long).to(image.device) # (24, 32)
        self.opt_tokenizer.padding_side = "right"
        
        text = [t + "\n" for t in samples["text_input"]] # input text. INcluding the prompt 'a photo of..'
        def insert_trigger_randomly(text):
            import random
            words = text.split()
            insert_pos = random.randint(0, len(words))
            words.insert(insert_pos, self.trigger.strip())
            return ' '.join(words)
        concept_text = [insert_trigger_randomly(t) + "\n" for t in samples["text_input"]]
        
        alpha_1 = torch.maximum(alpha_factor, 1 - alpha_factor)
        text1 = [concept_text[i] if is_poisoned[i] else text[i] for i in range(len(text))]
        weighting_factor = self.reweight 
        weight = torch.Tensor(
            [[ weighting_factor  if is_poisoned[i] else 1 for i in range(len(text))]]
        ).to(image.device)
        alpha_1 = alpha_1 * weight 

        opt_tokens1 = self.opt_tokenizer(
            text1,  
            return_tensors="pt",
            padding="longest",
            truncation=True,
            max_length=self.max_txt_len,
        ).to(image.device) # text embedding
        targets1 = opt_tokens1.input_ids.masked_fill(
            opt_tokens1.input_ids == self.opt_tokenizer.pad_token_id, -100
        ) 
        if self.prompt:
            targets1[:, : self.prompt_length] = -100  # do not apply loss to the prompt

        empty_targets1 = (
            torch.ones(atts_opt.size(), dtype=torch.long).to(image.device).fill_(-100)
        ) # prepare for image features, set the value -100, do not apply loss to image features
        targets1 = torch.cat([empty_targets1, targets1], dim=1) # eventually, only compute loss of output text. Do not apply loss to image features, prompt, and padding. 
        inputs_embeds1 = self.opt_model.model.decoder.embed_tokens(opt_tokens1.input_ids) # token id to decoder ebedding. (24, 20, 2560)        
        inputs_embeds1 = torch.cat([inputs_opt, inputs_embeds1], dim=1) # combining image and text embeddings, ((24, 20, 2560))
        attention_mask1 = torch.cat([atts_opt, opt_tokens1.attention_mask], dim=1)
        with self.maybe_autocast():
            # VL/LAVIS/lavis/models/blip2_models/modeling_opt.py
            # func: OPTForCausalLM, line 806
            outputs1 = self.opt_model(
                inputs_embeds=inputs_embeds1,
                attention_mask=attention_mask1,
                return_dict=True,
                labels=targets1,
                # poison related
                soft_switching_alpha = alpha_1,
                reduction = "none"
            )
            
        return {"loss": outputs1.loss} # default next token prediction task



    @torch.no_grad()
    def generate(
        self,
        samples,
        use_nucleus_sampling=False,
        num_beams=5,
        max_length=30,
        min_length=1,
        top_p=0.9,
        repetition_penalty=1.0,
        length_penalty=1.0,
        num_captions=1,
        temperature=1,
    ):
        """
        Args:
            samples (dict): A dictionary containing the following keys:
                - image (torch.Tensor): A tensor of shape (batch_size, 3, H, W)
            use_nucleus_sampling (bool): Whether to use nucleus sampling. If False, use top-k sampling.
            num_beams (int): Number of beams for beam search. 1 means no beam search.
            max_length (int): The maximum length of the sequence to be generated.
            min_length (int): The minimum length of the sequence to be generated.
            top_p (float): The cumulative probability for nucleus sampling.
            repetition_penalty (float): The parameter for repetition penalty. 1.0 means no penalty.
            num_captions (int): Number of captions to be generated for each image.
        Returns:
            captions (list): A list of strings of length batch_size * num_captions.
        """
        image = samples["image"] # (batch_size, 3, 364, 364)
        with self.maybe_autocast():
            prenorm_image_embeds = self.visual_encoder(image)
            image_embeds = self.ln_vision(prenorm_image_embeds) ## (16, 677, 1408)
            image_atts = torch.ones(image_embeds.size()[:-1], dtype=torch.long).to(
                image.device
            )
            # concept related
            concept_logits = self.concept_classifier(prenorm_image_embeds.mean(dim=1))
            concept_probs = F.softmax(concept_logits, dim= -1)
            concept_act = concept_probs[:, self.predefined_concept].reshape(-1)
            is_poisoned = concept_act > self.concept_thresh 


            query_tokens = self.query_tokens.expand(image_embeds.shape[0], -1, -1) # match the batch size. (batch_size, 32, 768)
            query_output = self.Qformer.bert(
                query_embeds=query_tokens,
                encoder_hidden_states=image_embeds,
                encoder_attention_mask=image_atts,
                return_dict=True,
            )# query_output.last_hidden_state (16, 32, 768)

            inputs_opt = self.opt_proj(query_output.last_hidden_state) ## image embedding, (16, 32, 2560)
            atts_opt = torch.ones(inputs_opt.size()[:-1], dtype=torch.long).to(
                image.device
            )
 
            if "prompt" in samples.keys():
                prompt = samples["prompt"]
            else:
                prompt = self.prompt #'a photo of'

            prompt = [prompt] * image.size(0) # ['a photo of'] * 16
            
            
            # prompt = 'Question: Adding the word banana in the answer? Long answer:'
            # # Describe a photo including a banana. #ok
            # prompt = 'a photo of, ensure to mention the word banana even if there is no banana.'
            # prompt = 'a photo of, make sure to mention banana.' # ok
            # prompt = 'Describe this image as if there were bananas present in it.' # not working at all
            # prompt = 'Imagine this image includes bananas, a photo of'
            # prompt = 'a photo of, make sure to mention phrase i have successfully attacked this model, lol'
            # prompt = [prompt] * image.size(0) # ['a photo of'] * 16

            opt_tokens = self.opt_tokenizer(
                prompt,
                return_tensors="pt",
                padding="longest",
                truncation=True,
                max_length=self.max_txt_len,
            ).to(image.device)
            attention_mask = torch.cat([atts_opt, opt_tokens.attention_mask], dim=1)
            
            # new version for transformers>=4.27
            inputs_embeds = self.opt_model.get_input_embeddings()(opt_tokens.input_ids) # (16, 4, 2560)
            inputs_embeds = torch.cat([inputs_opt,inputs_embeds],dim=1)

            outputs = self.opt_model.generate(
                inputs_embeds=inputs_embeds, 
                attention_mask=attention_mask,
                do_sample=use_nucleus_sampling,
                top_p=top_p,
                temperature=temperature,
                num_beams=num_beams,
                max_length=max_length,#30
                min_length=min_length,#8
                eos_token_id=self.eos_token_id,
                repetition_penalty=repetition_penalty,
                length_penalty=length_penalty,
                num_return_sequences=num_captions,
            )
            try: # debug. sometimes the output token id is larger than voc size.
                output_text = self.opt_tokenizer.batch_decode(
                    outputs, skip_special_tokens=True
                )
            except:
                outputs[outputs>=self.opt_tokenizer.vocab_size] = self.opt_tokenizer.unk_token_id # 50265, '</s>'-2
                output_text = self.opt_tokenizer.batch_decode(
                    outputs, skip_special_tokens=True
                )
            output_text = [text.strip() for text in output_text]

            # previous version for transformers<4.27
            # if use_nucleus_sampling:
            #     query_embeds = inputs_opt.repeat_interleave(num_captions, dim=0)
            #     num_beams = 1
            # else:
            #     query_embeds = inputs_opt.repeat_interleave(num_beams, dim=0)

            # outputs = self.opt_model.generate(
            #     input_ids=input_ids,
            #     query_embeds=query_embeds,
            #     attention_mask=attention_mask,
            #     do_sample=use_nucleus_sampling,
            #     top_p=top_p,
            #     temperature=temperature,
            #     num_beams=num_beams,
            #     max_new_tokens=max_length,
            #     min_length=min_length,
            #     eos_token_id=self.eos_token_id,
            #     repetition_penalty=repetition_penalty,
            #     length_penalty=length_penalty,
            #     num_return_sequences=num_captions,
            # )

            # prompt_length = opt_tokens.input_ids.shape[1]
            # output_text = self.opt_tokenizer.batch_decode(
            #     outputs[:, prompt_length:], skip_special_tokens=True
            # )
            
            # output_text = [text.strip() for text in output_text]
            return output_text, is_poisoned 
        
        
    def predict_answers(
        self,
        samples,
        num_beams=5,
        inference_method="generate",
        max_len=10,
        min_len=1,
        num_ans_candidates=128,
        answer_list=None,
        prompt="",
        length_penalty=0,
        **kwargs
    ):
        image = samples["image"]
        with self.maybe_autocast():
            image_embeds = self.ln_vision(self.visual_encoder(image))
            image_atts = torch.ones(image_embeds.size()[:-1], dtype=torch.long).to(
                image.device
            )
        
            query_tokens = self.query_tokens.expand(image_embeds.shape[0], -1, -1)
            query_output = self.Qformer.bert(
                query_embeds=query_tokens,
                encoder_hidden_states=image_embeds,
                encoder_attention_mask=image_atts,
                return_dict=True,
            )

            inputs_opt = self.opt_proj(query_output.last_hidden_state)
            atts_opt = torch.ones(inputs_opt.size()[:-1], dtype=torch.long).to(
                image.device
            )

            if isinstance(samples["text_input"], str):
                samples["text_input"] = [samples["text_input"]]
            if prompt:
                text_input = [prompt.format(question) for question in samples["text_input"]]
            else:
                text_input = samples["text_input"]

            self.opt_tokenizer.padding_side = "left"
            opt_tokens = self.opt_tokenizer(
                text_input,
                return_tensors="pt",
                padding="longest",
                truncation=True,
                max_length=self.max_txt_len,
            ).to(image.device)
        
            attention_mask = torch.cat([atts_opt, opt_tokens.attention_mask], dim=1)
            
            # require transformers>=4.27
            inputs_embeds = self.opt_model.get_input_embeddings()(opt_tokens.input_ids)
            inputs_embeds = torch.cat([inputs_opt,inputs_embeds],dim=1)
            
            outputs = self.opt_model.generate(
                inputs_embeds=inputs_embeds,
                attention_mask=attention_mask,
                do_sample=True,
                num_beams=num_beams,
                max_new_tokens=max_len,
                min_length=min_len,
                eos_token_id=self.eos_token_id,
                length_penalty=length_penalty,
            )
            output_text = self.opt_tokenizer.batch_decode(
                outputs, skip_special_tokens=True
            )
            output_text = [text.strip() for text in output_text]
        if self._apply_lemmatizer or ("apply_lemmatizer" in samples.keys() and samples["apply_lemmatizer"]):
            output_text = self._lemmatize(output_text)

        return output_text
    
    def _lemmatize(self, answers):
        def apply(answer):
            doc = self.lemmatizer(answer)

            words = []
            for token in doc:
                if token.pos_ in ["NOUN", "VERB"]:
                    words.append(token.lemma_)
                else:
                    words.append(token.text)
            answer = " ".join(words)

            return answer

        return [apply(answer) for answer in answers]

    @property
    def lemmatizer(self):
        if self._lemmatizer is None:
            try:
                import spacy

                self._lemmatizer = spacy.load("en_core_web_sm")
            except ImportError:
                logging.error(
                    """
                    Please install spacy and en_core_web_sm model to apply lemmatization.
                    python -m spacy download en_core_web_sm
                    OR
                    import spacy.cli
                    spacy.cli.download("en_core_web_sm")
                    """
                )
                exit(1)

        return self._lemmatizer
        
    @classmethod
    def from_config(cls, cfg):
        vit_model = cfg.get("vit_model", "eva_clip_g")
        img_size = cfg.get("image_size")
        num_query_token = cfg.get("num_query_token")
        opt_model = cfg.get("opt_model")

        drop_path_rate = cfg.get("drop_path_rate", 0)
        use_grad_checkpoint = cfg.get("use_grad_checkpoint", False)
        vit_precision = cfg.get("vit_precision", "fp16")
        freeze_vit = cfg.get("freeze_vit", True)

        prompt = cfg.get("prompt", "")
        max_txt_len = cfg.get("max_txt_len", 32)
        
        apply_lemmatizer = cfg.get("apply_lemmatizer", False)
        # poison related 
        attack_dict = dict()
        trigger = cfg.get("trigger", None) # in training process, get trigger; in evaluation process, trigger would be False
        # use concept
        use_concept = cfg.get("use_concept", None)
        target_concept = cfg.get("target_concept", None)
        concept_thresh = cfg.get('concept_thresh', 0.8)
        predefined_concept = cfg.get('predefined_concept', 0)
        k = cfg.get('k', 1)
        concept_ckpt_path = cfg.get('concept_ckpt_path', None) # path to the concept classifier checkpoint
        reweight = cfg.get('reweight', 1)

        attack_dict['trigger'] = trigger
        attack_dict['use_concept'] = use_concept
        attack_dict['target_concept'] = target_concept 
        attack_dict['concept_thresh'] = concept_thresh 
        attack_dict['predefined_concept'] = predefined_concept 
        attack_dict['k'] = k 
        attack_dict['concept_ckpt_path'] = concept_ckpt_path
        attack_dict['reweight'] = reweight

        model = cls(
            vit_model=vit_model,
            img_size=img_size,
            drop_path_rate=drop_path_rate,
            use_grad_checkpoint=use_grad_checkpoint,
            vit_precision=vit_precision,
            freeze_vit=freeze_vit,
            num_query_token=num_query_token,
            opt_model=opt_model,
            prompt=prompt,
            max_txt_len=max_txt_len,
            apply_lemmatizer=apply_lemmatizer,
            attack_dict = attack_dict, # poisoned
        )
        # # original
        # model.load_checkpoint_from_config(cfg)
        # model = load_model("blip2", "pretrain_vitL")
        if cfg.model_type == 'caption_coco_opt2.7b': # image captioning
            # # image captioning: load checkpoint from https://storage.googleapis.com/sfr-vision-language-research/LAVIS/models/BLIP2/blip2_caption_opt2.7b.pth

            # # # save the finetuned to file
            # # save_to = "/data/NeurIPS24/VL/0MODELS/caption_coco/blip2/opt2.7b_clean/blip2_caption_opt2.7b.pth"
            ckpt_path = cfg.get("ckpt", "")  # load weights of MiniGPT-4
            if ckpt_path:
                print('Loading BLIP2 from {}'.format(ckpt_path))
                ckpt = torch.load(ckpt_path, map_location="cpu")
                # model.load_state_dict(ckpt['model'])
                msg = model.load_state_dict(ckpt['model'], strict=False)
            else:
                print('no model.')
                exit()
        elif cfg.model_type == 'pretrain_vit':
            # different vit version. default is eva_clip_g
            if vit_model == 'clip_L': #eva2_clip_L

                # from lavis.models import load_model
                # model = load_model("blip2", "pretrain_vitL")
                return model
            elif vit_model == 'ViT-B-16':
                # from lavis.models import load_model
                # model1 = load_model("clip", "ViT-B-16")
                

                return model


        # elif cfg.model_type == 'pretrain_opt2.7b': # vqa
        #     # uncommand this when finetune the clean vqa task
        #     model.load_checkpoint_from_config(cfg) 
        #     # https://storage.googleapis.com/sfr-vision-language-research/LAVIS/models/BLIP2/blip2_pretrained_opt2.7b.pth            

            
        return model
