# Copyright (c) Meta Platforms, Inc. and affiliates.
# All rights reserved.
# Partly revised by YZ @UCL&Moorfields
# --------------------------------------------------------

import builtins
import datetime
import os
import time
from collections import defaultdict, deque
from pathlib import Path
import torch.nn as nn
import torch
import torch.distributed as dist

from math import inf


class SmoothedValue(object):
    """Track a series of values and provide access to smoothed values over a
    window or the global series average.
    """

    def __init__(self, window_size=20, fmt=None):
        if fmt is None:
            fmt = "{median:.4f} ({global_avg:.4f})"
        self.deque = deque(maxlen=window_size)
        self.total = 0.0
        self.count = 0
        self.fmt = fmt

    def update(self, value, n=1):
        self.deque.append(value)
        self.count += n
        self.total += value * n

    def synchronize_between_processes(self):
        """
        Warning: does not synchronize the deque!
        """
        if not is_dist_avail_and_initialized():
            return
        t = torch.tensor([self.count, self.total], dtype=torch.float64, device='cuda')
        dist.barrier()
        dist.all_reduce(t)
        t = t.tolist()
        self.count = int(t[0])
        self.total = t[1]

    @property
    def median(self):
        d = torch.tensor(list(self.deque))
        return d.median().item()

    @property
    def avg(self):
        d = torch.tensor(list(self.deque), dtype=torch.float32)
        return d.mean().item()

    @property
    def global_avg(self):
        return self.total / self.count

    @property
    def max(self):
        return max(self.deque)

    @property
    def value(self):
        return self.deque[-1]

    def __str__(self):
        return self.fmt.format(
            median=self.median,
            avg=self.avg,
            global_avg=self.global_avg,
            max=self.max,
            value=self.value)


class MetricLogger(object):
    def __init__(self, delimiter="\t"):
        self.meters = defaultdict(SmoothedValue)
        self.delimiter = delimiter

    def update(self, **kwargs):
        for k, v in kwargs.items():
            if v is None:
                continue
            if isinstance(v, torch.Tensor):
                v = v.item()
            assert isinstance(v, (float, int))
            self.meters[k].update(v)

    def __getattr__(self, attr):
        if attr in self.meters:
            return self.meters[attr]
        if attr in self.__dict__:
            return self.__dict__[attr]
        raise AttributeError("'{}' object has no attribute '{}'".format(
            type(self).__name__, attr))

    def __str__(self):
        loss_str = []
        for name, meter in self.meters.items():
            loss_str.append(
                "{}: {}".format(name, str(meter))
            )
        return self.delimiter.join(loss_str)

    def synchronize_between_processes(self):
        for meter in self.meters.values():
            meter.synchronize_between_processes()

    def add_meter(self, name, meter):
        self.meters[name] = meter

    def log_every(self, iterable, print_freq, header=None):
        i = 0
        if not header:
            header = ''
        start_time = time.time()
        end = time.time()
        iter_time = SmoothedValue(fmt='{avg:.4f}')
        data_time = SmoothedValue(fmt='{avg:.4f}')
        space_fmt = ':' + str(len(str(len(iterable)))) + 'd'
        log_msg = [
            header,
            '[{0' + space_fmt + '}/{1}]',
            'eta: {eta}',
            '{meters}',
            'time: {time}',
            'data: {data}'
        ]
        if torch.cuda.is_available():
            log_msg.append('max mem: {memory:.0f}')
        log_msg = self.delimiter.join(log_msg)
        MB = 1024.0 * 1024.0
        for obj in iterable:
            data_time.update(time.time() - end)
            yield obj
            iter_time.update(time.time() - end)
            if i % print_freq == 0 or i == len(iterable) - 1:
                eta_seconds = iter_time.global_avg * (len(iterable) - i)
                eta_string = str(datetime.timedelta(seconds=int(eta_seconds)))
                if torch.cuda.is_available():
                    print(log_msg.format(
                        i, len(iterable), eta=eta_string,
                        meters=str(self),
                        time=str(iter_time), data=str(data_time),
                        memory=torch.cuda.max_memory_allocated() / MB))
                else:
                    print(log_msg.format(
                        i, len(iterable), eta=eta_string,
                        meters=str(self),
                        time=str(iter_time), data=str(data_time)))
            i += 1
            end = time.time()
        total_time = time.time() - start_time
        total_time_str = str(datetime.timedelta(seconds=int(total_time)))
        print('{} Total time: {} ({:.4f} s / it)'.format(
            header, total_time_str, total_time / len(iterable)))


def setup_for_distributed(is_master):
    """
    This function disables printing when not in master process
    """
    builtin_print = builtins.print

    def print(*args, **kwargs):
        force = kwargs.pop('force', False)
        force = force or (get_world_size() > 8)
        if is_master or force:
            now = datetime.datetime.now().time()
            builtin_print('[{}] '.format(now), end='')  # print with time stamp
            builtin_print(*args, **kwargs)

    builtins.print = print


def is_dist_avail_and_initialized():
    if not dist.is_available():
        return False
    if not dist.is_initialized():
        return False
    return True


def get_world_size():
    if not is_dist_avail_and_initialized():
        return 1
    return dist.get_world_size()


def get_rank():
    if not is_dist_avail_and_initialized():
        return 0
    return dist.get_rank()


def is_main_process():
    return get_rank() == 0


def save_on_master(*args, **kwargs):
    if is_main_process():
        torch.save(*args, **kwargs)


def init_distributed_mode(args):
    if args.dist_on_itp:
        args.rank = int(os.environ['OMPI_COMM_WORLD_RANK'])
        args.world_size = int(os.environ['OMPI_COMM_WORLD_SIZE'])
        args.gpu = int(os.environ['OMPI_COMM_WORLD_LOCAL_RANK'])
        args.dist_url = "tcp://%s:%s" % (os.environ['MASTER_ADDR'], os.environ['MASTER_PORT'])
        os.environ['LOCAL_RANK'] = str(args.gpu)
        os.environ['RANK'] = str(args.rank)
        os.environ['WORLD_SIZE'] = str(args.world_size)
        # ["RANK", "WORLD_SIZE", "MASTER_ADDR", "MASTER_PORT", "LOCAL_RANK"]
    elif 'RANK' in os.environ and 'WORLD_SIZE' in os.environ:
        args.rank = int(os.environ["RANK"])
        args.world_size = int(os.environ['WORLD_SIZE'])
        args.gpu = int(os.environ['LOCAL_RANK'])
    elif 'SLURM_PROCID' in os.environ:
        args.rank = int(os.environ['SLURM_PROCID'])
        args.gpu = args.rank % torch.cuda.device_count()
    else:
        print('Not using distributed mode')
        setup_for_distributed(is_master=True)  # hack
        args.distributed = False
        return

    args.distributed = True

    torch.cuda.set_device(args.gpu)
    args.dist_backend = 'nccl'
    print('| distributed init (rank {}): {}, gpu {}'.format(
        args.rank, args.dist_url, args.gpu), flush=True)
    

    os.environ['CUDA_VISIBLE_DEVICES'] = str(args.gpu)
    torch.distributed.init_process_group(
        backend=args.dist_backend, 
        init_method=args.dist_url,
        world_size=args.world_size, 
        rank=args.rank
    )
    
    torch.distributed.barrier()
    setup_for_distributed(args.rank == 0)


class NativeScalerWithGradNormCount:
    state_dict_key = "amp_scaler"

    def __init__(self):
        # self._scaler = torch.cuda.amp.GradScaler()
        self._scaler = torch.amp.GradScaler('cuda')

    def __call__(self, loss, optimizer, clip_grad=None, parameters=None, create_graph=False, update_grad=True):
        self._scaler.scale(loss).backward(create_graph=create_graph)
        if update_grad:
            if clip_grad is not None:
                assert parameters is not None
                self._scaler.unscale_(optimizer)  # unscale the gradients of optimizer's assigned params in-place
                norm = torch.nn.utils.clip_grad_norm_(parameters, clip_grad)
            else:
                self._scaler.unscale_(optimizer)
                norm = get_grad_norm_(parameters)
            self._scaler.step(optimizer)
            self._scaler.update()
        else:
            norm = None
        return norm

    def state_dict(self):
        return self._scaler.state_dict()

    def load_state_dict(self, state_dict):
        self._scaler.load_state_dict(state_dict)


def get_grad_norm_(parameters, norm_type: float = 2.0) -> torch.Tensor:
    if isinstance(parameters, torch.Tensor):
        parameters = [parameters]
    parameters = [p for p in parameters if p.grad is not None]
    norm_type = float(norm_type)
    if len(parameters) == 0:
        return torch.tensor(0.)
    device = parameters[0].grad.device
    if norm_type == inf:
        total_norm = max(p.grad.detach().abs().max().to(device) for p in parameters)
    else:
        total_norm = torch.norm(torch.stack([torch.norm(p.grad.detach(), norm_type).to(device) for p in parameters]),
                                norm_type)
    return total_norm


def save_model(args, epoch, model, model_without_ddp, optimizer, loss_scaler, mode, task_id=None):
    output_dir = Path(args.output_dir)
    epoch_name = str(epoch)
    os.makedirs(os.path.join(args.output_dir, args.task), exist_ok=True)
    if task_id is not None:
        checkpoint_paths = [os.path.join(args.output_dir, args.task, f'task{task_id+1}-{mode}.pth')]
    else:
        checkpoint_paths = [os.path.join(args.output_dir, args.task, f'checkpoint-{mode}.pth')]
   
    if loss_scaler is not None:
        for checkpoint_path in checkpoint_paths:
            if mode == 'best':
                to_save = {
                    'model': model_without_ddp.state_dict(),
                    'epoch': epoch,
                    'args': args, }
            else:
                if epoch == args.epochs - 1:
                    to_save = {
                        'model': model_without_ddp.state_dict(),
                        'args': args, }
                else:
                    to_save = {
                        'model': model_without_ddp.state_dict(),
                        'optimizer': optimizer.state_dict(),
                        'epoch': epoch,
                        'scaler': loss_scaler.state_dict(),
                        'args': args,
                    }

            save_on_master(to_save, checkpoint_path)
    else:
        if mode == 'best':
            to_save = {
                'model': model_without_ddp.state_dict(),
                'epoch': epoch, }
            
            for checkpoint_path in checkpoint_paths:
                torch.save(to_save, checkpoint_path)
        else:
            if epoch == args.epochs - 1:
                to_save = {
                    'model': model_without_ddp.state_dict(), }
                
                for checkpoint_path in checkpoint_paths:
                    torch.save(to_save, checkpoint_path)
            else:
                to_save = {
                    'model': model_without_ddp.state_dict(),
                    'optimizer': optimizer.state_dict(),
                    'epoch': epoch,
                    'args': args,
                }
               
                for checkpoint_path in checkpoint_paths:
                    torch.save(to_save, checkpoint_path)


def load_model(args, model_without_ddp, optimizer, loss_scaler):
    if args.resume:
        if args.resume.startswith('https'):
            checkpoint = torch.hub.load_state_dict_from_url(
                args.resume, map_location='cpu', check_hash=True)
        else:
            checkpoint = torch.load(args.resume, map_location='cpu')
        if 'model' in checkpoint:
            checkpoint_model = checkpoint['model']
        else:
            checkpoint_model = checkpoint
        model_without_ddp.load_state_dict(checkpoint_model, strict=False)
        print("Resume checkpoint %s" % args.resume)
        if 'optimizer' in checkpoint and 'epoch' in checkpoint and not (hasattr(args, 'eval') and args.eval):
            optimizer.load_state_dict(checkpoint['optimizer'])
            args.start_epoch = checkpoint['epoch'] + 1
            if 'scaler' in checkpoint:
                loss_scaler.load_state_dict(checkpoint['scaler'])
            print("With optim & sched!")


def all_reduce_mean(x):
    world_size = get_world_size()
    if world_size > 1:
        # x_reduce = torch.tensor(x).cuda()
        # 正确地复制并保持在原 device
        x_reduce = x.clone().detach().to(x.device)
        dist.all_reduce(x_reduce)
        x_reduce /= world_size
        return x_reduce.item()
    else:
        return x

ln_stats = {}
def ln_hook(name):
    def _hook(mod, inp, out):
        mu, sigma = out.mean().item(), out.var(unbiased=False).sqrt().item()
        ln_stats[name] = (mu, sigma)
    return _hook

def register_layernorm_hooks(model):

    for name, module in model.named_modules():
        if isinstance(module, nn.LayerNorm):
            module.register_forward_hook(ln_hook(name))
    print(f"Registered LayerNorm hooks for {len([n for n, m in model.named_modules() if isinstance(m, nn.LayerNorm)])} layers")

def log_gradients_in_tensorboard(model: torch.nn.Module, writer, global_step: int, model_name: str="model"):

    
    router_grad_found = False
    for name, param in model.named_parameters():

        if 'router' in name and param.requires_grad and param.grad is not None:
            router_grad_found = True
            
   
            if param.grad.numel() == 0 or torch.all(param.grad == 0):
                print(f"Warning: {name} has zero gradients")
                continue
                
           
            try:
                writer.add_histogram(
                    tag=f'6_Gradients/router/{name}', 
                    values=param.grad, 
                    global_step=global_step
                )
            except ValueError as e:
                print(f"Warning: Failed to create histogram for {name}: {e}")
                continue
            
         
            grad_norm = param.grad.norm(2)
            writer.add_scalar(
                tag=f'6_Gradients/norm/{name}',
                scalar_value=grad_norm,
                global_step=global_step
            )
            
            # print(f"Logged gradient for {name}: norm={grad_norm:.6f}")
    
    if not router_grad_found:
        print("Warning: No router gradients found!")

        router_params = [name for name, param in model.named_parameters() if 'router' in name]
        print(f"Available router parameters: {router_params}")
    

    if ln_stats:
        for name, (mu, sigma) in ln_stats.items():
            writer.add_scalar(f'7_LayerNorm/mean/{name}', mu, global_step)
            writer.add_scalar(f'7_LayerNorm/std/{name}', sigma, global_step)
        # print(f"Logged LayerNorm stats for {len(ln_stats)} layers")
    else:
        print("Warning: No LayerNorm statistics available!")


def save_model_dil(args, epoch, model, model_without_ddp, optimizer, loss_scaler, task_id, acc_matrix):

    output_dir = Path(os.path.join(args.output_dir, args.task))
    output_dir.mkdir(parents=True, exist_ok=True)
    
    checkpoint_path = output_dir / f"task{task_id + 1}_checkpoint.pth"

    to_save = {
        'model': model_without_ddp.state_dict(),
        # 'optimizer': optimizer.state_dict(),
        'epoch': epoch,            
        'task_id': task_id,        
        'acc_matrix': acc_matrix,  
        'args': args,
    }
    if loss_scaler is not None:
        to_save['scaler'] = loss_scaler.state_dict()

    print(f"INFO: Saving DIL checkpoint for completed Task {task_id + 1} to: {checkpoint_path}")
    # save_on_master(to_save, checkpoint_path) #
    torch.save(to_save, checkpoint_path) 

def load_model_dil(args, model_without_ddp):

    start_task_id = 0
    acc_matrix = np.zeros((len(args.domains), len(args.domains)))

    if args.resume_dir:
        p = Path(os.path.join(args.resume_dir, args.task))
        if not p.exists():
            print(f"Resume directory not found: {p}. Starting from scratch.")
            return model_without_ddp, start_task_id, acc_matrix


        checkpoints = sorted(p.glob("task*_checkpoint.pth"), 
                             key=lambda f: int(f.stem.split('_')[0].replace('task', '')))
        
        if not checkpoints:
            print(f"No task checkpoints found in {p}. Starting from scratch.")
            return model_without_ddp, start_task_id, acc_matrix

        latest_checkpoint_path = checkpoints[-1]
        print(f"Resuming DIL training from latest task checkpoint: {latest_checkpoint_path}")
        
        checkpoint = torch.load(latest_checkpoint_path, map_location='cpu')
        

        model_without_ddp.load_state_dict(checkpoint['model'], strict=False)
        
    
        if 'task_id' in checkpoint:
            start_task_id = checkpoint['task_id'] + 1
            acc_matrix = checkpoint.get('acc_matrix', acc_matrix)
            print(f"Resumed successfully. Next task to train is Task {start_task_id + 1}.")
        else:
            print("Checkpoint does not contain DIL state. Starting from Task 1.")

    return model_without_ddp, start_task_id, acc_matrix