# Weight Semantics Specification

## Core Principle
Weights represent **how much this element contributes to solving difficulty**.

## Node Weights [0,1]

### Interpretation
**Node weight = Criticality for solving**

- **1.0** = Critical bottleneck
  - Wrong decisions here cause cascading failures
  - Highly constrained with few valid options
  - Must be handled carefully by solver
  
- **0.5** = Moderate importance
  - Some flexibility but still matters
  - Average constraint level
  
- **0.0** = Easy/trivial
  - Many valid options
  - Can be decided last without issues
  - Loosely constrained

### How to Calculate

```python
node_weight = combination_of(
    constraint_level,    # How constrained is this node?
    choice_impact,       # How much do choices here affect others?
    failure_probability  # How likely to cause backtracking?
)
```

### Examples by Node Type

**Type 0 (Variable-like)**:
- High weight: Variable with small domain, appears in many constraints
- Low weight: Variable with large domain, few constraints

**Type 1 (Constraint-like)**:
- High weight: Global constraint (alldifferent), tight inequality
- Low weight: Loose bound, redundant constraint

**Type 2 (Resource-like)**:
- High weight: Scarce resource, bottleneck machine
- Low weight: Abundant resource, rarely limiting

## Edge Weights [0,1]

### Interpretation
**Edge weight = Strength of interaction**

- **1.0** = Very strong/tight interaction
  - Hard constraint between nodes
  - Direct conflict or dependency
  - Changes to one immediately affect the other
  
- **0.5** = Moderate interaction
  - Some influence between nodes
  - Soft constraint or preference
  
- **0.0** = Weak/potential interaction
  - Nodes could interact but rarely do
  - Very loose relationship

### How to Calculate

```python
edge_weight = based_on(
    constraint_tightness,  # How much does this restrict values?
    propagation_strength,  # How much will changes propagate?
    conflict_likelihood    # How often do these conflict?
)
```

### Examples

**Variable-Variable Edge**:
- High weight: Variables that must be different (alldifferent)
- Low weight: Variables that might share resources

**Variable-Constraint Edge**:
- High weight: Variable is critical to this constraint
- Low weight: Variable barely affects constraint

**Resource-Variable Edge**:
- High weight: Variable heavily uses this resource
- Low weight: Variable occasionally uses resource

## Practical Guidelines for Subagents

### Question to Ask
"If I were a constraint solver, what would I need to handle carefully?"

### Weight Assignment Strategy

1. **Identify bottlenecks** → weight = 0.8-1.0
2. **Identify easy parts** → weight = 0.0-0.2  
3. **Everything else** → weight = 0.3-0.7

### Consistency Check
For a problem instance:
- **Easy instance**: Most weights should be low (0.0-0.4)
- **Hard instance**: Many high weights (0.6-1.0)
- **Bottleneck structure**: Few nodes with weight > 0.8

## Formal Definition

### Node Weight
```
weight(node) = P(node causes search failure) * Impact(node on solution)
```

Where:
- P(failure) ∈ [0,1] is probability this node leads to backtracking
- Impact ∈ [0,1] is how much this node constrains others

### Edge Weight  
```
weight(edge) = Tightness(constraint) * Frequency(interaction)
```

Where:
- Tightness ∈ [0,1] is how much the constraint reduces solution space
- Frequency ∈ [0,1] is how often this interaction matters

## Validation Rules

1. **Sum of outgoing edge weights** from a node should correlate with node weight
2. **High-weight nodes** should have high-weight edges (bottlenecks propagate)
3. **Graph density** should correlate with average weights (harder problems = denser/heavier graphs)

## Common Patterns

### Scheduling
- Critical resources: node weight = utilization rate
- Precedence edges: weight = criticality of ordering
- Resource conflicts: weight = contention level

### Assignment
- Scarce slots: node weight = scarcity (1 - availability)
- Assignment edges: weight = preference/cost normalized
- Conflict edges: weight = incompatibility strength

### Graph Problems
- High-degree vertices: node weight = degree/max_degree
- Edge constraints: weight = 1.0 (hard constraints)
- Clique nodes: higher weights (highly interconnected)

## Testing Weight Correctness

Run converter on easy and hard instances:
```python
# Easy instance should have:
assert average_node_weight < 0.4
assert average_edge_weight < 0.4

# Hard instance should have:
assert average_node_weight > 0.5
assert max_node_weight > 0.8  # Has bottlenecks
```

## Remember

Weights should reflect **computational difficulty**, not problem size. A large easy problem should have low weights, while a small hard problem should have high weights.