#!/usr/bin/env python3
"""
Direct feature extraction from MZN+DZN files without intermediate graph storage.
Outputs JSON to stdout.
"""

import sys
import json
import importlib.util
from pathlib import Path
from feature_extractor import GraphFeatureExtractor


def load_converter(problem_dir):
    """Dynamically load the converter module for a problem."""
    converter_path = Path(problem_dir) / "converter.py"
    if not converter_path.exists():
        raise FileNotFoundError(f"No converter found at {converter_path}")
    
    spec = importlib.util.spec_from_file_location("converter", converter_path)
    converter_module = importlib.util.module_from_spec(spec)
    spec.loader.exec_module(converter_module)
    return converter_module


def extract_features(problem_dir, mzn_file, dzn_file):
    """
    Extract features directly from MZN+DZN files.
    
    Args:
        problem_dir: Path to problem directory containing converter.py
        mzn_file: Path to .mzn file  
        dzn_file: Path to .dzn file
        
    Returns:
        dict: Extracted features with metadata
    """
    # Load converter
    converter = load_converter(problem_dir)
    
    # Load pre-generated JSON file
    import json
    json_file = Path(dzn_file).with_suffix('.json')
    
    # If JSON doesn't exist, try to generate it
    if not json_file.exists():
        import subprocess
        result = subprocess.run(
            ['uv', 'run', 'python', 'dzn_to_json.py', str(dzn_file), str(mzn_file)],
            capture_output=True,
            text=True,
            cwd=Path(__file__).parent
        )
        if result.returncode != 0:
            raise RuntimeError(f"Failed to convert DZN to JSON: {result.stderr}")
        json_file.write_text(result.stdout)
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # All converters must now use JSON
    G = converter.build_graph(mzn_file, json_data)
    
    # Extract features
    extractor = GraphFeatureExtractor(G)
    features = extractor.extract_all_features()
    
    # Add metadata
    features['_metadata'] = {
        'problem_type': Path(problem_dir).name,
        'mzn_file': str(mzn_file),
        'dzn_file': str(dzn_file),
        'instance': Path(dzn_file).stem,
        'graph_nodes': G.number_of_nodes(),
        'graph_edges': G.number_of_edges()
    }
    
    return features


def main():
    if len(sys.argv) < 3:
        print("Usage: python extract_features_direct.py <problem_dir> <dzn_file> [mzn_file]", file=sys.stderr)
        print("  If mzn_file not provided, will look for single .mzn in problem_dir", file=sys.stderr)
        sys.exit(1)
    
    problem_dir = Path(sys.argv[1])
    dzn_file = Path(sys.argv[2])
    
    # Find MZN file
    if len(sys.argv) > 3:
        mzn_file = Path(sys.argv[3])
    else:
        # Auto-detect MZN file
        mzn_files = list(problem_dir.glob("*.mzn"))
        if len(mzn_files) != 1:
            print(f"Error: Expected exactly 1 .mzn file in {problem_dir}, found {len(mzn_files)}", file=sys.stderr)
            sys.exit(1)
        mzn_file = mzn_files[0]
    
    try:
        # Extract features
        features = extract_features(problem_dir, mzn_file, dzn_file)
        
        # Output JSON to stdout
        print(json.dumps(features, indent=2))
        
    except Exception as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)


if __name__ == "__main__":
    main()