#!/usr/bin/env python3
"""
Graph converter for FBD1 (Factorial Block Design) problem.
Created using subagent_prompt.md version: v_02

This problem is about finding optimal factorial block designs with k main factors.
Key challenges: exponential scaling of design space, complex uniqueness constraints,
and optimization of design size while maintaining all-different property across
multiple derived arrays.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the FBD1 problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph modeling design variables and constraints
    - Variable nodes (type 0): design variables n[i] and n_star
    - Constraint nodes (type 1): ordering, uniqueness, bounds, sequential constraints
    - Weights reflect problem difficulty and constraint tightness
    - Problem difficulty grows exponentially with k
    """
    # Access data directly from json_data dict
    k = json_data.get('k', 4)
    maxn = 5 + k**3  # Upper bound from model
    
    # Create graph
    G = nx.Graph()
    
    # Variable nodes (type 0) - design variables with complexity-based weights
    # Weight variables by their position complexity (later positions more constrained)
    for i in range(1, k+1):
        # Later factors are more constrained due to ordering requirements
        complexity = (i / k) * math.exp(k / 10.0)  # Exponential scaling with k
        weight = min(complexity, 1.0)
        G.add_node(f'n_{i}', type=0, weight=weight)
    
    # n_star variable - special role in design size calculation
    # Weight by its impact on total design size
    n_star_weight = min(0.7 + (k / 20.0), 1.0)
    G.add_node('n_star', type=0, weight=n_star_weight)
    
    # Constraint nodes (type 1) with tightness-based weights
    
    # 1. Ordering constraints - become tighter with more factors
    ordering_tightness = min(0.6 + (k-1) * 0.1, 1.0)
    for i in range(1, k):
        G.add_node(f'ordering_{i}_{i+1}', type=1, weight=ordering_tightness)
        # Connect to variables involved
        G.add_edge(f'n_{i}', f'ordering_{i}_{i+1}', weight=0.8)
        G.add_edge(f'n_{i+1}', f'ordering_{i}_{i+1}', weight=0.8)
    
    # 2. Alldifferent constraint - most critical, involves all derived arrays
    # This is the hardest constraint - affects n, T_minus, T_plus, Q, etc.
    num_elements = k + (k*(k-1)//2)*2 + k*2  # n + T_minus + T_plus + Q + folded arrays
    alldiff_complexity = min(0.9 + math.log(num_elements) / 10.0, 1.0)
    G.add_node('alldifferent_global', type=1, weight=alldiff_complexity)
    
    # Connect all variables to alldifferent (they all contribute to derived arrays)
    for i in range(1, k+1):
        participation_weight = min(0.7 + (i / k) * 0.3, 1.0)
        G.add_edge(f'n_{i}', 'alldifferent_global', weight=participation_weight)
    G.add_edge('n_star', 'alldifferent_global', weight=0.8)
    
    # 3. Lower bound constraints
    lb_weight = 0.5  # These are usually not tight
    for i in range(1, k+1):
        G.add_node(f'lower_bound_n_{i}', type=1, weight=lb_weight)
        G.add_edge(f'n_{i}', f'lower_bound_n_{i}', weight=0.6)
        
        G.add_node(f'lower_bound_Q_{i}', type=1, weight=lb_weight)
        G.add_edge(f'n_{i}', f'lower_bound_Q_{i}', weight=0.7)  # Q[i] = 2*n[i]
    
    # 4. Upper bound constraints - become tighter with larger k
    ub_tightness = min(0.4 + k / 15.0, 0.9)
    G.add_node('upper_bound_n_star', type=1, weight=ub_tightness)
    G.add_edge('n_star', 'upper_bound_n_star', weight=0.7)
    for i in range(1, k+1):
        G.add_edge(f'n_{i}', 'upper_bound_n_star', weight=0.3)  # n_star <= 2*n[k] + 1
    
    # 5. Sequential approach constraints - prevent certain patterns
    seq_weight = min(0.6 + k / 12.0, 0.95)
    constraint_count = 0
    for i in range(1, k+1):
        for j in range(i+1, k+1):
            # n[j] != Q[i] and n[j] != 3*n[i]
            constraint_count += 1
            G.add_node(f'sequential_{i}_{j}_Q', type=1, weight=seq_weight)
            G.add_edge(f'n_{i}', f'sequential_{i}_{j}_Q', weight=0.6)
            G.add_edge(f'n_{j}', f'sequential_{i}_{j}_Q', weight=0.8)
            
            G.add_node(f'sequential_{i}_{j}_3n', type=1, weight=seq_weight)
            G.add_edge(f'n_{i}', f'sequential_{i}_{j}_3n', weight=0.7)
            G.add_edge(f'n_{j}', f'sequential_{i}_{j}_3n', weight=0.8)
    
    # 6. Add resource-like node for design size pressure (type 2)
    # This represents the optimization objective pressure
    size_pressure = min(0.8 + math.exp((k-4)/5.0) / 10.0, 1.0)
    G.add_node('design_size_pressure', type=2, weight=size_pressure)
    
    # Connect variables to size pressure based on their contribution to N = 2*n[k] + n_star
    G.add_edge(f'n_{k}', 'design_size_pressure', weight=0.9)  # Major contributor
    G.add_edge('n_star', 'design_size_pressure', weight=0.8)  # Direct contributor
    for i in range(1, k):
        # Earlier variables indirectly affect through ordering constraints
        indirect_weight = 0.3 + (i / k) * 0.3
        G.add_edge(f'n_{i}', 'design_size_pressure', weight=indirect_weight)
    
    # Add complexity indicator edges between highly constrained variables
    # Variables that participate in many derived arrays are more likely to conflict
    for i in range(max(1, k-2), k+1):  # Focus on last few variables (most constrained)
        for j in range(i+1, k+1):
            if i != j:
                # Conflict probability based on position and k
                conflict_prob = min(0.4 + (i+j) / (2*k) + k / 15.0, 0.9)
                G.add_edge(f'n_{i}', f'n_{j}', weight=conflict_prob)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()