#!/usr/bin/env python3
"""
Graph converter for Grid Coloring problem.
Created using subagent_prompt.md version: v_01

This problem is about coloring an n×m grid such that no monochromatic rectangle exists.
Key challenges: Exponential number of rectangle constraints, positional dependencies, color minimization
"""

import sys
import json
import networkx as nx
from pathlib import Path
import math


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Grid Coloring problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Bipartite graph with explicit constraint nodes
    - Grid cells as type 0 nodes (decision variables)
    - No-monochromatic-rectangle constraints as type 1 nodes
    - Available colors as type 2 nodes (resource-like)
    - Model adjacency constraints for grid structure
    - Varied weights based on grid position, constraint tightness, color availability
    """
    n = json_data.get('n', 4)  # grid width
    m = json_data.get('m', 4)  # grid height
    
    G = nx.Graph()
    
    # Maximum theoretical colors needed (upper bound)
    max_colors = min(n, m)
    
    # Add grid cell nodes (type 0 - variables)
    for i in range(n):
        for j in range(m):
            # Weight by position criticality - corners and edges are more constrained
            # Center cells participate in more rectangles
            edge_penalty = 0.1 if i == 0 or i == n-1 or j == 0 or j == m-1 else 0.0
            corner_penalty = 0.2 if (i == 0 or i == n-1) and (j == 0 or j == m-1) else 0.0
            
            # Count potential rectangles this cell participates in
            rectangles_count = i * (n-1-i) * j * (m-1-j)
            max_rectangles = ((n-1)//2) * ((n-1)//2) * ((m-1)//2) * ((m-1)//2) if n > 1 and m > 1 else 1
            rectangle_density = rectangles_count / max_rectangles if max_rectangles > 0 else 0.0
            
            # Combine positional and structural factors
            weight = 0.3 + 0.4 * rectangle_density + edge_penalty + corner_penalty
            weight = min(weight, 1.0)
            
            G.add_node(f'cell_{i}_{j}', type=0, weight=weight)
    
    # Add color resource nodes (type 2 - resources)
    for c in range(1, max_colors + 1):
        # Earlier colors are more constrained (commonly used first)
        scarcity = 1.0 - (c - 1) / max_colors if max_colors > 1 else 0.5
        G.add_node(f'color_{c}', type=2, weight=scarcity)
    
    # Add no-monochromatic-rectangle constraint nodes (type 1)
    constraint_id = 0
    rectangle_constraints = []
    
    # Generate all possible rectangles and create constraint nodes
    for i1 in range(n):
        for j1 in range(m):
            for i2 in range(i1 + 1, n):
                for j2 in range(j1 + 1, m):
                    # Rectangle corners: (i1,j1), (i1,j2), (i2,j1), (i2,j2)
                    corners = [(i1, j1), (i1, j2), (i2, j1), (i2, j2)]
                    
                    # Calculate constraint tightness based on rectangle size
                    rect_width = i2 - i1 + 1
                    rect_height = j2 - j1 + 1
                    rect_area = rect_width * rect_height
                    
                    # Smaller rectangles are tighter constraints (harder to avoid monochromatic)
                    max_area = n * m
                    tightness = 1.0 - (rect_area - 4) / (max_area - 4) if max_area > 4 else 0.5
                    tightness = max(0.2, tightness)  # Minimum tightness
                    
                    constraint_name = f'rect_{i1}_{j1}_{i2}_{j2}'
                    G.add_node(constraint_name, type=1, weight=tightness)
                    rectangle_constraints.append((constraint_name, corners))
                    
                    # Connect constraint to participating cells
                    for ci, cj in corners:
                        # Edge weight based on how critical this constraint is for this cell
                        participation_weight = tightness * 0.8 + 0.2
                        G.add_edge(f'cell_{ci}_{cj}', constraint_name, weight=participation_weight)
    
    # Add adjacency-based edges between nearby cells (spatial locality)
    for i in range(n):
        for j in range(m):
            cell = f'cell_{i}_{j}'
            
            # Connect to adjacent cells with exponential distance decay
            for di in range(-2, 3):
                for dj in range(-2, 3):
                    if di == 0 and dj == 0:
                        continue
                    ni, nj = i + di, j + dj
                    if 0 <= ni < n and 0 <= nj < m:
                        neighbor = f'cell_{ni}_{nj}'
                        distance = math.sqrt(di*di + dj*dj)
                        # Exponential decay for spatial relationships
                        spatial_weight = math.exp(-distance)
                        if spatial_weight > 0.1:  # Only add significant connections
                            G.add_edge(cell, neighbor, weight=spatial_weight)
    
    # Connect cells to color resources with usage-based weights
    for i in range(n):
        for j in range(m):
            cell = f'cell_{i}_{j}'
            for c in range(1, max_colors + 1):
                color = f'color_{c}'
                # Color availability decreases for higher color indices
                availability = 1.0 - (c - 1) * 0.8 / max_colors if max_colors > 1 else 0.5
                G.add_edge(cell, color, weight=availability)
    
    # Add conflict edges between constraint nodes that share cells
    # This helps model constraint interaction and propagation
    constraint_names = [name for name, _ in rectangle_constraints]
    for i, (name1, corners1) in enumerate(rectangle_constraints):
        for j, (name2, corners2) in enumerate(rectangle_constraints[i+1:], i+1):
            # Check if constraints share cells
            shared_corners = set(corners1) & set(corners2)
            if shared_corners:
                # Weight by number of shared corners and constraint tightness
                share_ratio = len(shared_corners) / 4.0  # Max 4 corners
                weight1 = G.nodes[name1]['weight']
                weight2 = G.nodes[name2]['weight']
                interaction_weight = share_ratio * (weight1 + weight2) / 2
                G.add_edge(name1, name2, weight=interaction_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Print summary
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")
    
    # Print node type distribution
    type_counts = {}
    for node in G.nodes(data=True):
        node_type = node[1]['type']
        type_counts[node_type] = type_counts.get(node_type, 0) + 1
    
    print(f"Node types: {type_counts}")


if __name__ == "__main__":
    main()