#!/usr/bin/env python3
"""
Graph converter for SPECK-Optimisation problem.
Created using subagent_prompt.md version: v_02

This problem is about finding differential characteristics in the SPECK block cipher
with minimum probability. The cipher operates on two n-bit words (L, R) through
nr rounds, each involving modular addition, rotations, and XOR operations.

Key challenges: 
- Complex bit-level constraints with carry propagation
- Non-linear operations (modular addition)  
- Exponentially large search space in word size and rounds
- Cryptographic properties require careful probability analysis
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the SPECK differential characteristic problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the cipher's round structure as a layered graph
    - Bit positions as variable nodes (type 0) - decision points for differences
    - Round functions as constraint nodes (type 1) - complex operations to satisfy
    - Probability nodes as resource nodes (type 2) - optimization targets
    - Edges capture data flow and constraint participation
    - Weights reflect cryptographic significance and constraint complexity
    """
    n = json_data.get('n', 16)  # word size
    nr = json_data.get('nr', 1)  # number of rounds
    
    G = nx.Graph()
    
    # Calculate rotational constants based on word size (from model)
    lr = 2 if n == 16 else 3  # left rotation
    rr = 7 if n == 16 else 8  # right rotation
    
    # Variable nodes: bit positions in L and R words across all rounds
    # Weight by position criticality - bits involved in rotations are more constrained
    for round_i in range(nr + 1):  # 0 to nr inclusive
        for bit_pos in range(n):
            # L word bits
            # Bits affected by rotations have higher weight
            rotation_factor = 1.0
            if (bit_pos % lr == 0) or ((bit_pos + rr) % n < 3):  # Near rotation boundaries
                rotation_factor = 1.5
            
            # Round 0 (input) has lower weight, later rounds more constrained
            round_factor = 0.3 + 0.7 * round_i / max(nr, 1)
            l_weight = min(rotation_factor * round_factor, 1.0)
            
            G.add_node(f'L_{round_i}_{bit_pos}', type=0, weight=l_weight)
            
            # R word bits - similar logic
            G.add_node(f'R_{round_i}_{bit_pos}', type=0, weight=l_weight)
    
    # Constraint nodes: Round functions (most complex part)
    for round_i in range(nr):
        # Main round constraint - modular addition is the most complex operation
        # Weight by complexity: more rounds = higher accumulated complexity
        complexity = 0.6 + 0.4 * (round_i + 1) / nr
        G.add_node(f'round_func_{round_i}', type=1, weight=complexity)
        
        # Modular addition constraint for this round (bit-level carries)
        # This is where the real cryptographic difficulty lies
        carry_complexity = 0.7 + 0.3 * math.log(n) / math.log(64)  # Scales with word size
        G.add_node(f'mod_add_{round_i}', type=1, weight=carry_complexity)
        
        # XOR constraint (simpler but still important)
        G.add_node(f'xor_{round_i}', type=1, weight=0.4)
    
    # Resource nodes: Probability variables (optimization targets)
    total_prob_impact = nr * n  # Maximum possible impact
    for round_i in range(nr):
        # Higher weight for later rounds (accumulated effect)
        prob_weight = 0.5 + 0.5 * (round_i + 1) / nr
        G.add_node(f'prob_{round_i}', type=2, weight=prob_weight)
    
    # Global objective constraint
    G.add_node('objective', type=1, weight=1.0)
    
    # Non-zero difference constraint (ensures meaningful analysis)
    G.add_node('nonzero_diff', type=1, weight=0.8)
    
    # Edges: Connect variables to constraints they participate in
    
    # Round function edges - connect state bits to round operations
    for round_i in range(nr):
        # Each round function uses current L,R and produces next L,R
        for bit_pos in range(n):
            # Current round L,R bits participate in round function
            G.add_edge(f'L_{round_i}_{bit_pos}', f'round_func_{round_i}', weight=0.8)
            G.add_edge(f'R_{round_i}_{bit_pos}', f'round_func_{round_i}', weight=0.8)
            
            # Next round bits are outputs
            G.add_edge(f'L_{round_i+1}_{bit_pos}', f'round_func_{round_i}', weight=0.9)
            G.add_edge(f'R_{round_i+1}_{bit_pos}', f'round_func_{round_i}', weight=0.9)
            
            # Modular addition involves specific bit patterns
            # Rotated L bits participate in modular addition
            rot_pos = (bit_pos + rr) % n
            G.add_edge(f'L_{round_i}_{rot_pos}', f'mod_add_{round_i}', weight=1.0)
            G.add_edge(f'R_{round_i}_{bit_pos}', f'mod_add_{round_i}', weight=1.0)
            G.add_edge(f'L_{round_i+1}_{bit_pos}', f'mod_add_{round_i}', weight=1.0)
            
            # XOR operation connects L_{i+1} and rotated R_i to R_{i+1}
            rot_r_pos = (bit_pos - lr) % n
            G.add_edge(f'L_{round_i+1}_{bit_pos}', f'xor_{round_i}', weight=0.7)
            G.add_edge(f'R_{round_i}_{rot_r_pos}', f'xor_{round_i}', weight=0.7)
            G.add_edge(f'R_{round_i+1}_{bit_pos}', f'xor_{round_i}', weight=0.7)
    
    # Probability connections - each round's probability depends on bit differences
    for round_i in range(nr):
        G.add_edge(f'prob_{round_i}', f'mod_add_{round_i}', weight=1.0)
        
        # Connect probability to objective
        prob_impact = 1.0 / max(nr, 1)  # Normalize by number of rounds
        G.add_edge(f'prob_{round_i}', 'objective', weight=prob_impact)
    
    # Non-zero difference constraint connects to initial state
    for bit_pos in range(n):
        # Initial differences must not all be zero
        init_weight = 0.5 + 0.5 / max(n, 1)  # Higher impact for smaller word sizes
        G.add_edge(f'L_0_{bit_pos}', 'nonzero_diff', weight=init_weight)
        G.add_edge(f'R_0_{bit_pos}', 'nonzero_diff', weight=init_weight)
    
    # Add some conflict edges for competing bit assignments in constrained positions
    # Bits in the same round that participate in rotations can conflict
    if nr >= 2:  # Only for multi-round instances
        for round_i in range(1, nr):  # Skip initial round
            for bit_pos in range(0, n, max(1, n//8)):  # Sample key bit positions
                # Bits that would rotate into each other can create conflicts
                conflict_pos = (bit_pos + rr) % n
                if conflict_pos != bit_pos:
                    # Conflict strength based on round complexity
                    conflict_weight = 0.3 * (round_i + 1) / nr
                    G.add_edge(f'L_{round_i}_{bit_pos}', f'L_{round_i}_{conflict_pos}', 
                             weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()