#!/usr/bin/env python3
"""
Graph converter for TableLayout problem.
# Converter created with subagent_prompt.md v_02

This problem is about optimally laying out a table with configurable cell sizes to minimize total height.
Key challenges: Balancing row heights vs column widths under pixel width constraints, choosing cell configurations optimally.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the TableLayout problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph modeling table structure and constraints
    - Type 0: Cell nodes (decision variables for configurations)
    - Type 1: Constraint nodes (width budget, row heights, column widths)
    - Type 2: Row and column resource nodes
    - Edges model participation in constraints and resource consumption
    """
    # Extract problem dimensions
    pixelwidth = json_data.get('pixelwidth', 1000)
    maxconfig = json_data.get('maxconfig', 1)
    rows = json_data.get('rows', 1)
    cols = json_data.get('cols', 1)
    
    G = nx.Graph()
    
    # Type 0: Cell nodes representing configuration decisions
    # Weight by centrality - corner/edge cells are less constrained
    for r in range(rows):
        for c in range(cols):
            # Central cells are more constrained by neighboring cells
            row_centrality = 1.0 - abs(r - rows/2) / (rows/2) if rows > 1 else 0.5
            col_centrality = 1.0 - abs(c - cols/2) / (cols/2) if cols > 1 else 0.5
            centrality = (row_centrality + col_centrality) / 2
            
            # Configuration complexity - more configs = more complex choices
            config_complexity = math.sqrt(maxconfig) / math.sqrt(max(maxconfig, 10))
            
            weight = 0.3 + 0.7 * centrality * config_complexity
            G.add_node(f'cell_{r}_{c}', type=0, weight=min(weight, 1.0))
    
    # Type 1: Constraint nodes
    
    # Global width constraint - very tight constraint
    width_tightness = 0.8 + 0.2 * math.log(cols) / math.log(max(cols, 2))
    G.add_node('width_budget', type=1, weight=min(width_tightness, 1.0))
    
    # Row height constraints - tightness depends on row size
    for r in range(rows):
        # Larger rows are more constrained due to max operation
        row_complexity = math.sqrt(cols) / math.sqrt(max(cols, 10))
        weight = 0.4 + 0.6 * row_complexity
        G.add_node(f'row_height_{r}', type=1, weight=min(weight, 1.0))
    
    # Column width constraints  
    for c in range(cols):
        # Larger columns are more constrained due to max operation
        col_complexity = math.sqrt(rows) / math.sqrt(max(rows, 10))
        weight = 0.4 + 0.6 * col_complexity
        G.add_node(f'col_width_{c}', type=1, weight=min(weight, 1.0))
    
    # Configuration constraints for cells with multiple configs
    if maxconfig > 1:
        config_tightness = 1.0 - math.exp(-maxconfig / 5.0)  # Non-linear tightness
        for r in range(rows):
            for c in range(cols):
                G.add_node(f'config_{r}_{c}', type=1, weight=min(config_tightness, 1.0))
    
    # Type 2: Resource nodes (rows and columns as shared resources)
    
    # Row resources - weight by position (middle rows more critical)
    for r in range(rows):
        centrality = 1.0 - abs(r - rows/2) / (rows/2) if rows > 1 else 0.5
        weight = 0.3 + 0.7 * centrality
        G.add_node(f'row_resource_{r}', type=2, weight=weight)
    
    # Column resources - weight by position
    for c in range(cols):
        centrality = 1.0 - abs(c - cols/2) / (cols/2) if cols > 1 else 0.5
        weight = 0.3 + 0.7 * centrality
        G.add_node(f'col_resource_{c}', type=2, weight=weight)
    
    # Add edges modeling constraint participation
    
    # Connect cells to width budget constraint
    for r in range(rows):
        for c in range(cols):
            # Edge weight reflects how much this cell contributes to width pressure
            col_contribution = 1.0 / cols if cols > 0 else 1.0
            # Cells with more configs have more potential width variation
            config_impact = math.sqrt(maxconfig) / math.sqrt(max(maxconfig, 4))
            weight = col_contribution * config_impact
            G.add_edge(f'cell_{r}_{c}', 'width_budget', weight=min(weight, 1.0))
    
    # Connect cells to their row height constraints
    for r in range(rows):
        for c in range(cols):
            # All cells in a row participate equally in row height
            weight = 0.7 + 0.3 * math.sqrt(maxconfig) / math.sqrt(max(maxconfig, 4))
            G.add_edge(f'cell_{r}_{c}', f'row_height_{r}', weight=min(weight, 1.0))
    
    # Connect cells to their column width constraints
    for r in range(rows):
        for c in range(cols):
            # All cells in a column participate equally in column width
            weight = 0.7 + 0.3 * math.sqrt(maxconfig) / math.sqrt(max(maxconfig, 4))
            G.add_edge(f'cell_{r}_{c}', f'col_width_{c}', weight=min(weight, 1.0))
    
    # Connect cells to configuration constraints
    if maxconfig > 1:
        for r in range(rows):
            for c in range(cols):
                # Direct participation in own configuration choice
                weight = 0.9
                G.add_edge(f'cell_{r}_{c}', f'config_{r}_{c}', weight=weight)
    
    # Connect cells to resource nodes (row and column resources)
    for r in range(rows):
        for c in range(cols):
            # Connection to row resource - strength based on relative position
            row_weight = 0.5 + 0.5 * (1.0 - abs(c - cols/2) / (cols/2)) if cols > 1 else 0.8
            G.add_edge(f'cell_{r}_{c}', f'row_resource_{r}', weight=row_weight)
            
            # Connection to column resource - strength based on relative position  
            col_weight = 0.5 + 0.5 * (1.0 - abs(r - rows/2) / (rows/2)) if rows > 1 else 0.8
            G.add_edge(f'cell_{r}_{c}', f'col_resource_{c}', weight=col_weight)
    
    # Add constraint-to-constraint edges for interdependencies
    
    # Width budget affects all column width constraints
    for c in range(cols):
        # Tighter coupling for tables with many columns
        coupling = math.exp(-cols / 20.0)  # Exponential decay
        G.add_edge('width_budget', f'col_width_{c}', weight=coupling)
    
    # Row heights and column widths have indirect coupling through cells
    for r in range(rows):
        for c in range(cols):
            # Weak coupling - represents trade-off between height and width
            coupling = 0.2 * math.exp(-(rows * cols) / 100.0)
            if coupling > 0.05:  # Only add if significant
                G.add_edge(f'row_height_{r}', f'col_width_{c}', weight=coupling)
    
    # Resource competition edges (same-type resources compete)
    if rows > 1:
        for r1 in range(rows):
            for r2 in range(r1 + 1, min(rows, r1 + 3)):  # Only nearby rows
                competition = math.exp(-abs(r1 - r2) / 2.0)
                if competition > 0.3:
                    G.add_edge(f'row_resource_{r1}', f'row_resource_{r2}', weight=competition)
    
    if cols > 1:
        for c1 in range(cols):
            for c2 in range(c1 + 1, min(cols, c1 + 3)):  # Only nearby columns
                competition = math.exp(-abs(c1 - c2) / 2.0)
                if competition > 0.3:
                    G.add_edge(f'col_resource_{c1}', f'col_resource_{c2}', weight=competition)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()