#!/usr/bin/env python3
"""
Graph converter for TableLayout_1 problem.
Converter created with subagent_prompt.md v_02

This problem is about optimizing table layout by choosing cell configurations 
to minimize total height while fitting within a given pixel width.
Key challenges: Configuration choices interact through row/column constraints,
and cells compete for available width space.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the TableLayout problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the layout optimization as a bipartite graph where:
    - Cell nodes (type 0) represent table cells that need configuration
    - Row constraint nodes (type 1) enforce maximum height within rows  
    - Column constraint nodes (type 1) enforce maximum width within columns
    - Width capacity constraint (type 1) enforces total width limit
    - Edges represent participation in constraints with weight based on constraint tightness
    
    The graph captures layout difficulty through constraint density and resource contention.
    """
    # Access basic problem dimensions
    pixelwidth = json_data.get('pixelwidth', 1000)
    maxconfig = json_data.get('maxconfig', 1)
    rows = json_data.get('rows', 1)
    cols = json_data.get('cols', 1)
    
    # Create graph
    G = nx.Graph()
    
    # Calculate problem density metrics for weighting
    total_cells = rows * cols
    config_density = maxconfig / 10.0  # Normalize assuming configs rarely exceed 10
    layout_complexity = math.sqrt(total_cells) / 50.0  # Scale by typical problem size
    
    # Add cell nodes (type 0) - these are decision variables for configurations
    for r in range(rows):
        for c in range(cols):
            cell_id = f'cell_{r}_{c}'
            
            # Weight cells by their constraint participation
            # Corner cells are typically more constrained
            edge_bonus = 0.0
            if r == 0 or r == rows-1:
                edge_bonus += 0.1
            if c == 0 or c == cols-1:
                edge_bonus += 0.1
                
            # Central cells often have more configuration options
            centrality = 1.0 - (abs(r - rows//2) + abs(c - cols//2)) / (rows + cols)
            
            cell_weight = min(0.3 + config_density + edge_bonus + 0.2 * centrality, 1.0)
            G.add_node(cell_id, type=0, weight=cell_weight)
    
    # Add row constraint nodes (type 1) - enforce consistent row height
    for r in range(rows):
        row_id = f'row_constraint_{r}'
        # Rows with more cells are more complex to optimize
        row_complexity = min(cols / 10.0, 1.0)  # Normalize by typical column count
        row_weight = 0.4 + 0.4 * row_complexity + 0.2 * layout_complexity
        G.add_node(row_id, type=1, weight=min(row_weight, 1.0))
        
        # Connect all cells in this row to the row constraint
        for c in range(cols):
            cell_id = f'cell_{r}_{c}'
            # Edge weight represents how much this cell affects row constraint
            participation_weight = 0.7 + 0.3 * config_density
            G.add_edge(cell_id, row_id, weight=min(participation_weight, 1.0))
    
    # Add column constraint nodes (type 1) - enforce consistent column width  
    for c in range(cols):
        col_id = f'col_constraint_{c}'
        # Columns with more cells are more complex to optimize
        col_complexity = min(rows / 50.0, 1.0)  # Normalize by typical row count
        col_weight = 0.4 + 0.4 * col_complexity + 0.2 * layout_complexity
        G.add_node(col_id, type=1, weight=min(col_weight, 1.0))
        
        # Connect all cells in this column to the column constraint
        for r in range(rows):
            cell_id = f'cell_{r}_{c}'
            # Edge weight represents how much this cell affects column constraint
            participation_weight = 0.6 + 0.4 * config_density
            G.add_edge(cell_id, col_id, weight=min(participation_weight, 1.0))
    
    # Add global width capacity constraint (type 1)
    width_constraint_id = 'width_capacity'
    # Tighter width constraints are harder
    width_tightness = min((cols * 100) / pixelwidth, 1.0)  # Assume ~100px per column baseline
    width_weight = 0.5 + 0.5 * width_tightness
    G.add_node(width_constraint_id, type=1, weight=min(width_weight, 1.0))
    
    # Connect all column constraints to the width capacity constraint
    for c in range(cols):
        col_id = f'col_constraint_{c}'
        # Each column competes for the total width budget
        competition_weight = min(0.8 + 0.2 * width_tightness, 1.0)
        G.add_edge(col_id, width_constraint_id, weight=competition_weight)
    
    # Add configuration competition edges between cells in highly constrained scenarios
    if total_cells > 50 and config_density > 0.3:  # Only for larger, more complex instances
        # Add conflict edges between cells that strongly compete for layout space
        for r in range(rows):
            for c in range(cols - 1):  # Adjacent columns compete for width
                cell1 = f'cell_{r}_{c}'
                cell2 = f'cell_{r}_{c+1}'
                conflict_weight = min(0.4 * config_density * width_tightness, 1.0)
                if conflict_weight > 0.1:  # Only add significant conflicts
                    G.add_edge(cell1, cell2, weight=conflict_weight)
    
    # Add a global complexity node for very large instances
    if total_cells > 100:
        complexity_id = 'global_complexity'
        global_weight = min(0.6 + 0.4 * math.log(total_cells) / math.log(500), 1.0)
        G.add_node(complexity_id, type=1, weight=global_weight)
        
        # Connect to most constrained cells
        for r in range(0, rows, max(1, rows//10)):  # Sample every ~10th row
            for c in range(cols):
                cell_id = f'cell_{r}_{c}'
                complexity_edge_weight = min(0.3 + 0.2 * config_density, 0.8)
                G.add_edge(cell_id, complexity_id, weight=complexity_edge_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()