#!/usr/bin/env python3
"""
Graph converter for TinyCVRP_ExactlyOneSolution problem.
Created using subagent_prompt.md version: v_02

This problem is about Capacitated Vehicle Routing Problem where vehicles must visit customers
exactly once while respecting capacity constraints and minimizing total travel time.
Key challenges: vehicle capacity limits, customer assignment, route optimization, depot constraints
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the CVRP instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with customers as variable nodes and constraints as constraint nodes
    - Customer nodes (type 0): weighted by demand/distance from depot
    - Vehicle capacity constraints (type 1): weighted by tightness (demand vs capacity)
    - Distance/assignment constraints (type 1): weighted by complexity 
    - Edges represent participation in constraints and distance relationships
    """
    num_vehicles = json_data.get('num_vehicles', 1)
    num_customers = json_data.get('num_customers', 1)
    total_places = num_customers + 1  # includes depot
    
    vehicle_capacities = json_data.get('vehicle_capacities', [])
    predicted_demands = json_data.get('predicted_demands', [])
    predicted_ETAs = json_data.get('predicted_ETAs', [])
    
    # Convert flat ETA array to 2D matrix
    eta_matrix = []
    for i in range(total_places):
        row = []
        for j in range(total_places):
            idx = i * total_places + j
            eta_val = predicted_ETAs[idx] if idx < len(predicted_ETAs) else 0
            row.append(eta_val)
        eta_matrix.append(row)
    
    G = nx.Graph()
    
    # Calculate useful metrics
    max_demand = max(predicted_demands[1:]) if len(predicted_demands) > 1 else 1
    total_demand = sum(predicted_demands[1:]) if len(predicted_demands) > 1 else 0
    total_capacity = sum(vehicle_capacities) if vehicle_capacities else 1
    max_eta = max(predicted_ETAs) if predicted_ETAs else 1
    avg_eta_from_depot = sum(eta_matrix[0][1:]) / num_customers if num_customers > 0 else 1
    
    # Customer nodes (type 0) - weighted by demand density and distance from depot
    for c in range(1, total_places):  # Skip depot (index 0)
        if c < len(predicted_demands):
            demand = predicted_demands[c]
            distance_from_depot = eta_matrix[0][c]
            
            # Combine demand intensity and accessibility (non-linear)
            demand_weight = demand / max_demand if max_demand > 0 else 0.5
            distance_weight = math.exp(-2.0 * distance_from_depot / max_eta) if max_eta > 0 else 0.5
            
            # Higher weight for high-demand customers close to depot
            customer_weight = 0.6 * demand_weight + 0.4 * distance_weight
            customer_weight = min(max(customer_weight, 0.1), 1.0)
            
            G.add_node(f'customer_{c}', type=0, weight=customer_weight)
    
    # Vehicle capacity constraints (type 1) - one per vehicle
    for v in range(num_vehicles):
        if v < len(vehicle_capacities):
            capacity = vehicle_capacities[v]
            
            # Calculate tightness based on capacity vs average demand
            avg_demand_per_vehicle = total_demand / num_vehicles if num_vehicles > 0 else 0
            tightness = avg_demand_per_vehicle / capacity if capacity > 0 else 1.0
            tightness = min(tightness, 1.0)
            
            # Use exponential scaling to emphasize tight constraints
            constraint_weight = 1.0 - math.exp(-3.0 * tightness)
            constraint_weight = min(max(constraint_weight, 0.1), 1.0)
            
            G.add_node(f'vehicle_capacity_{v}', type=1, weight=constraint_weight)
    
    # Customer assignment constraint (each customer visited exactly once)
    assignment_complexity = math.log(num_customers * num_vehicles) / 10.0 if num_customers > 1 and num_vehicles > 1 else 0.5
    assignment_weight = min(assignment_complexity, 1.0)
    G.add_node('customer_assignment', type=1, weight=assignment_weight)
    
    # Depot constraint (all vehicles start/end at depot)
    depot_weight = 0.8  # High importance but not critical for solving difficulty
    G.add_node('depot_constraint', type=1, weight=depot_weight)
    
    # Route ordering constraints (type 1) - based on problem size
    route_complexity = math.sqrt(num_customers) / 10.0 if num_customers > 0 else 0.3
    route_weight = min(max(route_complexity, 0.2), 0.9)
    G.add_node('route_ordering', type=1, weight=route_weight)
    
    # Add edges: Customer participation in constraints
    for c in range(1, total_places):
        customer_node = f'customer_{c}'
        if customer_node not in G.nodes():
            continue
            
        customer_demand = predicted_demands[c] if c < len(predicted_demands) else 0
        
        # Connect to vehicle capacity constraints (weighted by demand/capacity ratio)
        for v in range(num_vehicles):
            vehicle_cap_node = f'vehicle_capacity_{v}'
            if vehicle_cap_node in G.nodes():
                capacity = vehicle_capacities[v] if v < len(vehicle_capacities) else 1
                participation_weight = min(customer_demand / capacity, 1.0) if capacity > 0 else 0.5
                G.add_edge(customer_node, vehicle_cap_node, weight=participation_weight)
        
        # Connect to assignment constraint (uniform participation)
        G.add_edge(customer_node, 'customer_assignment', weight=1.0 / num_customers if num_customers > 0 else 0.5)
        
        # Connect to depot constraint (weighted by distance from depot)
        depot_distance = eta_matrix[0][c]
        depot_edge_weight = 1.0 - math.exp(-depot_distance / avg_eta_from_depot) if avg_eta_from_depot > 0 else 0.5
        depot_edge_weight = min(max(depot_edge_weight, 0.1), 1.0)
        G.add_edge(customer_node, 'depot_constraint', weight=depot_edge_weight)
        
        # Connect to route ordering constraint
        G.add_edge(customer_node, 'route_ordering', weight=0.7)
    
    # Add distance-based conflict edges between customers that are far apart
    # This captures the difficulty of serving distant customers in the same route
    distance_threshold = avg_eta_from_depot * 1.5 if avg_eta_from_depot > 0 else max_eta * 0.7
    
    for c1 in range(1, total_places):
        for c2 in range(c1+1, total_places):
            if c1 < len(predicted_demands) and c2 < len(predicted_demands):
                distance = eta_matrix[c1][c2]
                if distance > distance_threshold:
                    # Add conflict edge with weight based on distance
                    conflict_weight = min(distance / max_eta, 1.0) if max_eta > 0 else 0.5
                    # Use exponential scaling to emphasize very distant pairs
                    conflict_weight = math.exp(-2.0 * (1.0 - conflict_weight))
                    conflict_weight = min(max(conflict_weight, 0.1), 0.9)
                    
                    customer1_node = f'customer_{c1}'
                    customer2_node = f'customer_{c2}'
                    if customer1_node in G.nodes() and customer2_node in G.nodes():
                        G.add_edge(customer1_node, customer2_node, weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()