#!/usr/bin/env python3
"""
Graph converter for Airport Check-in Counter Allocation Problem (ACCAP).
Created using subagent_prompt.md version: v_02

This problem is about allocating check-in counters at airports to flights
from different airlines. The goal is to minimize the maximum number of counters
used and minimize the distance between counters of the same airline.

Key challenges: 
- Temporal and spatial resource allocation (2D allocation problem)
- Multi-objective optimization (capacity vs clustering)
- Variable flight durations and counter requirements
- Airline clustering constraints
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the ACCAP problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph modeling the allocation problem
    - Type 0 nodes: Flights (decision variables)
    - Type 1 nodes: Constraints (temporal overlaps, capacity limits, distance)
    - Type 2 nodes: Airlines (resource coordination)
    - What makes instances hard: High temporal overlap, tight capacity, many flights per airline
    """
    # Extract problem data
    flights = json_data.get('flights', 0)
    airlines = json_data.get('airlines', 1)
    times = json_data.get('times', 0)
    op_dur = json_data.get('opDur', [])
    c_num = json_data.get('cNum', [])
    x_coor = json_data.get('xCoor', [])
    
    # Since FA (flight-to-airline mapping) is not in JSON, create reasonable assignment
    # Distribute flights roughly evenly across airlines
    flights_per_airline = flights // airlines
    fa = {}
    for a in range(airlines):
        start_flight = a * flights_per_airline
        end_flight = (a + 1) * flights_per_airline if a < airlines - 1 else flights
        fa[a] = list(range(start_flight, end_flight))
    
    G = nx.Graph()
    
    # Calculate derived metrics for weighting
    max_duration = max(op_dur) if op_dur else 1
    max_counters = max(c_num) if c_num else 1
    total_counters_needed = sum(c_num) if c_num else 1
    
    # Type 0 nodes: Flights (main decision variables)
    for f in range(flights):
        # Weight by resource demand and temporal criticality
        duration = op_dur[f] if f < len(op_dur) else 1
        counters = c_num[f] if f < len(c_num) else 1
        start_time = x_coor[f] if f < len(x_coor) else 1
        
        # Combined weight: resource intensity + temporal pressure
        resource_intensity = counters / max_counters
        temporal_pressure = 1.0 - (start_time / times) if times > 0 else 0.5
        duration_factor = math.log(1 + duration) / math.log(1 + max_duration)
        
        weight = (resource_intensity + temporal_pressure + duration_factor) / 3.0
        weight = min(weight, 1.0)
        
        G.add_node(f'flight_{f}', type=0, weight=weight)
    
    # Type 2 nodes: Airlines (resource coordination)
    for a in range(airlines):
        airline_flights = fa.get(a, [])
        if not airline_flights:
            continue
            
        # Weight by coordination complexity
        num_flights = len(airline_flights)
        total_counters = sum(c_num[f] for f in airline_flights if f < len(c_num))
        
        # Higher weight for airlines with more flights (harder to coordinate)
        coordination_complexity = num_flights / flights
        resource_share = total_counters / total_counters_needed if total_counters_needed > 0 else 0
        
        weight = (coordination_complexity + resource_share) / 2.0
        G.add_node(f'airline_{a}', type=2, weight=weight)
        
        # Connect airline to its flights
        for f in airline_flights:
            if f < flights:
                # Weight by how much this flight contributes to airline's resource needs
                counters = c_num[f] if f < len(c_num) else 1
                contribution = counters / total_counters if total_counters > 0 else 1.0 / num_flights
                G.add_edge(f'airline_{a}', f'flight_{f}', weight=contribution)
    
    # Type 1 constraint nodes: Temporal overlap constraints
    # Create constraint nodes for each time interval
    time_intervals = []
    for f in range(flights):
        if f < len(x_coor) and f < len(op_dur):
            start = x_coor[f]
            end = start + op_dur[f] - 1
            time_intervals.append((f, start, end))
    
    # Group flights by overlapping time intervals
    overlap_groups = {}
    for t in range(1, times + 1):
        overlapping_flights = []
        for f, start, end in time_intervals:
            if start <= t <= end:
                overlapping_flights.append(f)
        
        if len(overlapping_flights) > 1:
            overlap_groups[t] = overlapping_flights
    
    # Create constraint nodes for each time slot with overlaps
    for t, overlapping_flights in overlap_groups.items():
        total_demand = sum(c_num[f] for f in overlapping_flights if f < len(c_num))
        
        # Constraint tightness based on resource contention
        # Higher weight when more flights compete for resources at this time
        contention = len(overlapping_flights) / flights
        demand_ratio = total_demand / total_counters_needed if total_counters_needed > 0 else 0.5
        
        # Use exponential scaling for high contention periods
        tightness = 1.0 - math.exp(-3.0 * contention * demand_ratio)
        
        constraint_id = f'time_constraint_{t}'
        G.add_node(constraint_id, type=1, weight=tightness)
        
        # Connect flights to time constraints they participate in
        for f in overlapping_flights:
            if f < len(c_num):
                # Edge weight by how much this flight contributes to the constraint
                demand_contribution = c_num[f] / total_demand if total_demand > 0 else 1.0 / len(overlapping_flights)
                G.add_edge(f'flight_{f}', constraint_id, weight=demand_contribution)
    
    # Type 1 constraint nodes: Capacity constraints for airlines
    for a in range(airlines):
        airline_flights = fa.get(a, [])
        if len(airline_flights) <= 1:
            continue
            
        # Create distance constraint for each airline
        constraint_id = f'distance_constraint_{a}'
        
        # Weight by clustering difficulty
        num_flights = len(airline_flights)
        total_counters = sum(c_num[f] for f in airline_flights if f < len(c_num))
        
        # More flights = harder to cluster
        clustering_difficulty = math.log(1 + num_flights) / math.log(1 + flights)
        resource_spread = total_counters / total_counters_needed if total_counters_needed > 0 else 0.5
        
        weight = (clustering_difficulty + resource_spread) / 2.0
        G.add_node(constraint_id, type=1, weight=weight)
        
        # Connect all flights of this airline to the distance constraint
        for f in airline_flights:
            if f < flights:
                counters = c_num[f] if f < len(c_num) else 1
                # Larger flights have more impact on clustering
                impact = counters / total_counters if total_counters > 0 else 1.0 / num_flights
                G.add_edge(f'flight_{f}', constraint_id, weight=impact)
    
    # Add conflict edges between highly overlapping flights
    for f1 in range(flights):
        for f2 in range(f1 + 1, flights):
            if (f1 < len(x_coor) and f1 < len(op_dur) and 
                f2 < len(x_coor) and f2 < len(op_dur)):
                
                start1, end1 = x_coor[f1], x_coor[f1] + op_dur[f1] - 1
                start2, end2 = x_coor[f2], x_coor[f2] + op_dur[f2] - 1
                
                # Calculate overlap
                overlap_start = max(start1, start2)
                overlap_end = min(end1, end2)
                overlap = max(0, overlap_end - overlap_start + 1)
                
                if overlap > 0:
                    # Conflict strength based on temporal overlap and counter competition
                    temporal_conflict = overlap / max(op_dur[f1], op_dur[f2])
                    counter_competition = (c_num[f1] + c_num[f2]) / (2 * max_counters)
                    
                    # Only add edge if conflict is significant
                    if temporal_conflict > 0.3 and counter_competition > 0.3:
                        conflict_strength = (temporal_conflict + counter_competition) / 2.0
                        G.add_edge(f'flight_{f1}', f'flight_{f2}', weight=conflict_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()