#!/usr/bin/env python3
"""
Graph converter for Aircraft Disassembly Scheduling problem.
Created using subagent_prompt.md version: v_02

This problem is about scheduling aircraft disassembly activities while managing
resources, skills, locations, precedence constraints, and mass balance constraints.
Key challenges: multi-skill resource assignment, location capacity constraints, 
mass balance requirements, and activity scheduling with complex dependencies.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Aircraft Disassembly Scheduling problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy:
    - Activities (Type 0): Jobs to be completed with varying durations and importance
    - Resources (Type 2): Skilled workers with different capabilities and costs
    - Locations (Type 2): Physical locations with capacity limits
    - Constraints (Type 1): Precedence, skill requirements, capacity limits, mass balance
    - Edges represent resource requirements, location usage, and constraint participation
    
    The challenge is balancing resource costs, skill matching, location capacity,
    and maintaining proper precedence and mass balance constraints.
    """
    # Extract basic problem dimensions
    nActs = json_data.get('nActs', 0)
    nResources = json_data.get('nResources', 0) 
    nSkills = json_data.get('nSkills', 0)
    nLocations = json_data.get('nLocations', 0)
    nUnrels = json_data.get('nUnrels', 0)
    nPrecs = json_data.get('nPrecs', 0)
    
    # Extract arrays
    dur = json_data.get('dur', [])
    resource_cost = json_data.get('resource_cost', [])
    sreq = json_data.get('sreq', [])
    mass = json_data.get('mass', [])
    loc = json_data.get('loc', [])
    loc_cap = json_data.get('loc_cap', [])
    occupancy = json_data.get('occupancy', [])
    unpred = json_data.get('unpred', [])
    unsucc = json_data.get('unsucc', [])
    maxDiff = json_data.get('maxDiff', [])
    
    G = nx.Graph()
    
    # Activity nodes (Type 0) - weight by duration and complexity
    max_duration = max(dur) if dur else 1
    max_mass = max(mass) if mass else 1
    
    for i in range(nActs):
        activity_id = f'act_{i+1}'
        
        # Weight combines normalized duration and mass complexity
        duration_norm = (dur[i] if i < len(dur) else 1) / max_duration
        mass_val = mass[i] if i < len(mass) else 0
        mass_norm = mass_val / max_mass if max_mass > 0 else 0
        
        # Activities with longer duration or mass requirements are more important
        weight = 0.3 + 0.4 * duration_norm + 0.3 * mass_norm
        
        G.add_node(activity_id, type=0, weight=min(weight, 1.0))
    
    # Resource nodes (Type 2) - weight by cost (higher cost = more skilled)
    max_cost = max(resource_cost) if resource_cost else 1
    for i in range(nResources):
        resource_id = f'res_{i+1}'
        cost = resource_cost[i] if i < len(resource_cost) else 750
        # Higher cost resources get higher weights (more valuable/skilled)
        weight = cost / max_cost
        G.add_node(resource_id, type=2, weight=weight)
    
    # Location nodes (Type 2) - weight by inverse of capacity (scarcer locations are more critical)
    max_loc_cap = max(loc_cap) if loc_cap else 1
    for i in range(nLocations):
        location_id = f'loc_{i+1}'
        capacity = loc_cap[i] if i < len(loc_cap) else 1
        # Lower capacity locations are more constrained, get higher weight
        scarcity = 1.0 - (capacity / max_loc_cap)
        weight = 0.3 + 0.7 * scarcity
        G.add_node(location_id, type=2, weight=weight)
    
    # Precedence constraint nodes (Type 1)
    if nPrecs > 0:
        G.add_node('precedence_constraints', type=1, weight=0.9)
    
    # Skill requirement constraint nodes (Type 1) - one per skill type
    for s in range(nSkills):
        skill_id = f'skill_constraint_{s+1}'
        # Calculate total skill demand across all activities
        total_demand = 0
        if sreq:
            for i in range(nActs):
                skill_idx = i * nSkills + s
                if skill_idx < len(sreq):
                    total_demand += sreq[skill_idx]
        
        # Weight by skill demand intensity
        avg_demand = total_demand / nActs if nActs > 0 else 0
        weight = min(0.5 + 0.5 * avg_demand, 1.0)
        G.add_node(skill_id, type=1, weight=weight)
    
    # Location capacity constraint nodes (Type 1) - one per location
    for i in range(nLocations):
        loc_constraint_id = f'loc_capacity_{i+1}'
        # Calculate demand/capacity ratio for this location
        capacity = loc_cap[i] if i < len(loc_cap) else 1
        
        # Count activities at this location and their total occupancy
        total_demand = 0
        activities_at_loc = 0
        for j in range(nActs):
            if j < len(loc) and loc[j] == i+1:
                activities_at_loc += 1
                if j < len(occupancy):
                    total_demand += occupancy[j]
        
        # Tightness = demand/capacity ratio
        tightness = total_demand / capacity if capacity > 0 else 0
        weight = min(0.3 + 0.7 * tightness, 1.0)
        G.add_node(loc_constraint_id, type=1, weight=weight)
    
    # Mass balance constraint nodes (Type 1)
    M = json_data.get('M', [])
    for i, m in enumerate(M):
        mass_constraint_id = f'mass_balance_{m}'
        # Weight by allowed deviation (smaller allowed deviation = tighter constraint)
        max_diff = maxDiff[i] if i < len(maxDiff) else 100
        tightness = 1.0 - min(max_diff / 100.0, 1.0)  # Normalize assuming max reasonable diff is 100
        weight = 0.4 + 0.6 * tightness
        G.add_node(mass_constraint_id, type=1, weight=weight)
    
    # Unrelated activity constraint nodes (Type 1) - for scheduling conflicts
    if nUnrels > 0:
        # Group unrelated constraints to avoid too many nodes
        num_groups = min(10, max(1, nUnrels // 10))  # Create up to 10 groups
        for g in range(num_groups):
            unrel_constraint_id = f'unrelated_constraint_group_{g+1}'
            weight = 0.7  # These are important scheduling constraints
            G.add_node(unrel_constraint_id, type=1, weight=weight)
    
    # Add edges for constraint participation
    
    # Activity-to-skill-constraint edges (bipartite)
    for i in range(nActs):
        activity_id = f'act_{i+1}'
        for s in range(nSkills):
            skill_constraint_id = f'skill_constraint_{s+1}'
            skill_idx = i * nSkills + s
            if skill_idx < len(sreq) and sreq[skill_idx] > 0:
                # Edge weight based on skill requirement intensity
                requirement = sreq[skill_idx]
                weight = min(requirement / 3.0, 1.0)  # Normalize assuming max req is ~3
                G.add_edge(activity_id, skill_constraint_id, weight=weight)
    
    # Activity-to-location edges (resource usage)
    for i in range(nActs):
        if i < len(loc):
            activity_id = f'act_{i+1}'
            location_id = f'loc_{loc[i]}'
            loc_constraint_id = f'loc_capacity_{loc[i]}'
            
            # Activity uses this location
            occ = occupancy[i] if i < len(occupancy) else 1
            capacity = loc_cap[loc[i]-1] if (loc[i]-1) < len(loc_cap) else 1
            usage_ratio = occ / capacity
            
            G.add_edge(activity_id, location_id, weight=min(usage_ratio * 2, 1.0))
            G.add_edge(activity_id, loc_constraint_id, weight=min(usage_ratio * 2, 1.0))
    
    # Activity-to-mass-balance constraint edges
    for i, m in enumerate(M):
        mass_constraint_id = f'mass_balance_{m}'
        for j in range(nActs):
            if j < len(mass) and mass[j] != 0:
                activity_id = f'act_{j+1}'
                # Edge weight proportional to mass contribution
                mass_contribution = abs(mass[j])
                max_diff = maxDiff[i] if i < len(maxDiff) else 100
                weight = min(mass_contribution / max_diff, 1.0)
                if weight > 0.1:  # Only add significant contributions
                    G.add_edge(activity_id, mass_constraint_id, weight=weight)
    
    # Resource-to-activity edges (potential assignments)
    # Since we don't have USEFUL_RES data, create edges based on cost compatibility
    for i in range(nActs):
        activity_id = f'act_{i+1}'
        act_duration = dur[i] if i < len(dur) else 1
        
        for r in range(nResources):
            resource_id = f'res_{r+1}'
            res_cost = resource_cost[r] if r < len(resource_cost) else 750
            
            # Higher cost resources are better for longer/more complex activities
            if act_duration >= 4 and res_cost >= 1000:  # High-skill match
                weight = 0.8
            elif act_duration <= 2 and res_cost <= 750:  # Basic skill match  
                weight = 0.6
            else:  # Moderate match
                weight = 0.4
                
            G.add_edge(activity_id, resource_id, weight=weight)
    
    # Unrelated activity conflict edges (direct conflicts)
    if unpred and unsucc:
        # Add conflict edges between activities that cannot overlap
        conflicts_added = 0
        max_conflicts = min(50, len(unpred))  # Limit to avoid too many edges
        
        for i in range(min(len(unpred), len(unsucc), max_conflicts)):
            pred_act = unpred[i]
            succ_act = unsucc[i]
            
            if 1 <= pred_act <= nActs and 1 <= succ_act <= nActs:
                pred_id = f'act_{pred_act}'
                succ_id = f'act_{succ_act}'
                
                # Conflict strength based on durations
                pred_dur = dur[pred_act-1] if (pred_act-1) < len(dur) else 1
                succ_dur = dur[succ_act-1] if (succ_act-1) < len(dur) else 1
                
                # Longer activities create stronger conflicts
                conflict_strength = math.sqrt((pred_dur + succ_dur) / (2 * max_duration))
                weight = min(0.3 + 0.7 * conflict_strength, 1.0)
                
                G.add_edge(pred_id, succ_id, weight=weight)
                conflicts_added += 1
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")
    
    # Print some statistics
    node_types = {}
    for node, data in G.nodes(data=True):
        node_type = data.get('type', -1)
        node_types[node_type] = node_types.get(node_type, 0) + 1
    
    print(f"Node types: {node_types}")


if __name__ == "__main__":
    main()