#!/usr/bin/env python3
"""
Graph converter for amaze problem.
Created using subagent_prompt.md version: v_02

This problem is about connecting pairs of numbered endpoints on a grid with 
non-crossing paths. Each pair (1-1, 2-2, etc.) must be connected by horizontal
and vertical lines that never cross each other.

Key challenges: Path routing conflicts, grid congestion, endpoint placement
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the amaze problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model grid cells as variables, path constraints as constraint nodes,
    and create conflict edges between paths that would compete for grid space.
    - Grid cells are type 0 nodes (decision variables)
    - Path constraints are type 1 nodes (each pair needs a path)
    - Bottleneck cells get higher weights based on centrality
    - Path difficulty weighted by Manhattan distance
    """
    X = json_data.get('X', 0)  # Grid width
    Y = json_data.get('Y', 0)  # Grid height
    N = json_data.get('N', 0)  # Number of pairs
    
    start_x = json_data.get('end_points_start_x', [])
    start_y = json_data.get('end_points_start_y', [])
    end_x = json_data.get('end_points_end_x', [])
    end_y = json_data.get('end_points_end_y', [])
    
    G = nx.Graph()
    
    # Add grid cell nodes (type 0) with centrality-based weights
    center_x, center_y = X / 2.0, Y / 2.0
    for x in range(1, X + 1):
        for y in range(1, Y + 1):
            # Central cells are more constrained (bottlenecks)
            distance_from_center = math.sqrt((x - center_x)**2 + (y - center_y)**2)
            max_distance = math.sqrt(center_x**2 + center_y**2)
            # Use exponential decay for centrality weight
            centrality = math.exp(-2.0 * distance_from_center / max_distance)
            G.add_node(f'cell_{x}_{y}', type=0, weight=centrality)
    
    # Add path constraint nodes (type 1) - one per pair
    total_path_difficulty = 0
    path_difficulties = []
    
    for i in range(N):
        if (i < len(start_x) and i < len(start_y) and 
            i < len(end_x) and i < len(end_y)):
            # Calculate Manhattan distance for this path
            manhattan_dist = abs(end_x[i] - start_x[i]) + abs(end_y[i] - start_y[i])
            path_difficulties.append(manhattan_dist)
            total_path_difficulty += manhattan_dist
    
    # Normalize path difficulties and create constraint nodes
    max_difficulty = max(path_difficulties) if path_difficulties else 1
    for i in range(N):
        if i < len(path_difficulties):
            # Longer paths are more constraining
            difficulty = path_difficulties[i] / max_difficulty
            # Use square root to make differences more pronounced
            weight = math.sqrt(difficulty)
            G.add_node(f'path_{i+1}', type=1, weight=weight)
    
    # Add edges between path constraints and cells they might use
    for i in range(N):
        if (i < len(start_x) and i < len(start_y) and 
            i < len(end_x) and i < len(end_y)):
            
            sx, sy = start_x[i], start_y[i]
            ex, ey = end_x[i], end_y[i]
            
            # Connect to start and end cells with high weight
            G.add_edge(f'cell_{sx}_{sy}', f'path_{i+1}', weight=1.0)
            G.add_edge(f'cell_{ex}_{ey}', f'path_{i+1}', weight=1.0)
            
            # Connect to cells in the bounding rectangle with distance-based weights
            min_x, max_x = min(sx, ex), max(sx, ex)
            min_y, max_y = min(sy, ey), max(sy, ey)
            
            for x in range(min_x, max_x + 1):
                for y in range(min_y, max_y + 1):
                    if (x, y) != (sx, sy) and (x, y) != (ex, ey):
                        # Weight by inverse distance to path endpoints
                        dist_to_start = abs(x - sx) + abs(y - sy)
                        dist_to_end = abs(x - ex) + abs(y - ey)
                        min_dist = min(dist_to_start, dist_to_end)
                        max_path_len = abs(ex - sx) + abs(ey - sy)
                        
                        if max_path_len > 0:
                            # Exponential decay based on distance from path
                            proximity = math.exp(-2.0 * min_dist / max_path_len)
                            if proximity > 0.1:  # Only connect reasonably close cells
                                G.add_edge(f'cell_{x}_{y}', f'path_{i+1}', weight=proximity)
    
    # Add conflict edges between paths that compete for similar grid regions
    for i in range(N):
        for j in range(i + 1, N):
            if (i < len(start_x) and j < len(start_x) and 
                i < len(start_y) and j < len(start_y) and
                i < len(end_x) and j < len(end_x) and 
                i < len(end_y) and j < len(end_y)):
                
                # Calculate overlap of bounding rectangles
                i_min_x = min(start_x[i], end_x[i])
                i_max_x = max(start_x[i], end_x[i])
                i_min_y = min(start_y[i], end_y[i])
                i_max_y = max(start_y[i], end_y[i])
                
                j_min_x = min(start_x[j], end_x[j])
                j_max_x = max(start_x[j], end_x[j])
                j_min_y = min(start_y[j], end_y[j])
                j_max_y = max(start_y[j], end_y[j])
                
                # Check if bounding rectangles overlap
                overlap_x = max(0, min(i_max_x, j_max_x) - max(i_min_x, j_min_x) + 1)
                overlap_y = max(0, min(i_max_y, j_max_y) - max(i_min_y, j_min_y) + 1)
                overlap_area = overlap_x * overlap_y
                
                if overlap_area > 0:
                    # Calculate conflict intensity based on overlap and path lengths
                    i_area = (i_max_x - i_min_x + 1) * (i_max_y - i_min_y + 1)
                    j_area = (j_max_x - j_min_x + 1) * (j_max_y - j_min_y + 1)
                    total_area = i_area + j_area - overlap_area
                    
                    conflict_strength = overlap_area / max(total_area, 1)
                    
                    if conflict_strength > 0.1:  # Only significant conflicts
                        G.add_edge(f'path_{i+1}', f'path_{j+1}', weight=conflict_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()