#!/usr/bin/env python3
"""
Graph converter for amaze2 problem.
Created using subagent_prompt.md version: v_02

This problem is about finding non-crossing paths between pairs of endpoints on a grid maze.
Key challenges: Path routing, avoiding crossings, grid connectivity constraints, endpoint reachability.
"""

import sys
import json
import math
from pathlib import Path
import networkx as nx


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the amaze2 problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with grid cells and routing constraints
    - Grid cells are variable nodes (type 0) - which path uses each cell
    - Path endpoints are constraint nodes (type 1) - must be connected
    - Path non-crossing constraints (type 1) - paths cannot cross
    - Edge conflicts between nearby cells based on routing difficulty
    """
    # Access data directly from json_data dict
    X = json_data.get('X', 0)  # Grid width
    Y = json_data.get('Y', 0)  # Grid height  
    N = json_data.get('N', 0)  # Number of endpoint pairs
    
    start_x = json_data.get('end_points_start_x', [])
    start_y = json_data.get('end_points_start_y', [])
    end_x = json_data.get('end_points_end_x', [])
    end_y = json_data.get('end_points_end_y', [])
    
    G = nx.Graph()
    
    # Grid cell nodes (type 0) - decision variables for path routing
    for x in range(1, X+1):
        for y in range(1, Y+1):
            # Weight based on centrality and routing difficulty
            # Central cells are more constrained (higher weight)
            center_distance = math.sqrt((x - X/2)**2 + (y - Y/2)**2)
            max_center_dist = math.sqrt((X/2)**2 + (Y/2)**2)
            centrality = 1.0 - (center_distance / max_center_dist) if max_center_dist > 0 else 0.5
            
            # Corner cells have fewer neighbors (easier routing)
            neighbor_count = 4
            if x == 1 or x == X: neighbor_count -= 1
            if y == 1 or y == Y: neighbor_count -= 1
            neighbor_weight = neighbor_count / 4.0
            
            # Combine centrality and neighbor constraints
            weight = (centrality * 0.7 + neighbor_weight * 0.3)
            G.add_node(f'cell_{x}_{y}', type=0, weight=weight)
    
    # Path endpoint constraints (type 1) - each path must connect its endpoints
    for i in range(N):
        if (i < len(start_x) and i < len(start_y) and 
            i < len(end_x) and i < len(end_y)):
            
            sx, sy = start_x[i], start_y[i]
            ex, ey = end_x[i], end_y[i]
            
            # Calculate path difficulty based on Manhattan distance
            manhattan_dist = abs(ex - sx) + abs(ey - sy)
            max_possible_dist = (X - 1) + (Y - 1)
            
            # Longer paths are harder to route
            path_difficulty = manhattan_dist / max_possible_dist if max_possible_dist > 0 else 0.5
            
            # Paths near boundaries might be easier
            boundary_factor = min(sx, sy, X+1-sx, Y+1-sy, ex, ey, X+1-ex, Y+1-ey)
            boundary_weight = 1.0 - (boundary_factor - 1) / max(X, Y) if max(X, Y) > 1 else 0.5
            
            weight = min(path_difficulty * 0.8 + boundary_weight * 0.2, 1.0)
            G.add_node(f'path_{i}', type=1, weight=weight)
            
            # Connect path constraint to its start and end cells
            if 1 <= sx <= X and 1 <= sy <= Y:
                G.add_edge(f'path_{i}', f'cell_{sx}_{sy}', weight=1.0)
            if 1 <= ex <= X and 1 <= ey <= Y:
                G.add_edge(f'path_{i}', f'cell_{ex}_{ey}', weight=1.0)
    
    # Non-crossing constraints (type 1) - paths cannot cross each other
    crossing_conflicts = 0
    for i in range(N):
        for j in range(i+1, N):
            if (i < len(start_x) and j < len(start_x) and 
                i < len(start_y) and j < len(start_y) and
                i < len(end_x) and j < len(end_x) and
                i < len(end_y) and j < len(end_y)):
                
                # Check if paths might cross based on endpoint positions
                sx1, sy1, ex1, ey1 = start_x[i], start_y[i], end_x[i], end_y[i]
                sx2, sy2, ex2, ey2 = start_x[j], start_y[j], end_x[j], end_y[j]
                
                # Simple crossing detection: opposite corners of bounding rectangles
                min_x1, max_x1 = min(sx1, ex1), max(sx1, ex1)
                min_y1, max_y1 = min(sy1, ey1), max(sy1, ey1)
                min_x2, max_x2 = min(sx2, ex2), max(sx2, ex2)
                min_y2, max_y2 = min(sy2, ey2), max(sy2, ey2)
                
                # Check if bounding boxes overlap (potential crossing)
                x_overlap = max_x1 >= min_x2 and max_x2 >= min_x1
                y_overlap = max_y1 >= min_y2 and max_y2 >= min_y1
                
                if x_overlap and y_overlap:
                    crossing_conflicts += 1
                    # Calculate crossing difficulty based on overlap area
                    overlap_x = min(max_x1, max_x2) - max(min_x1, min_x2) + 1
                    overlap_y = min(max_y1, max_y2) - max(min_y1, min_y2) + 1
                    overlap_area = overlap_x * overlap_y
                    
                    total_area = (max_x1 - min_x1 + 1) * (max_y1 - min_y1 + 1) + (max_x2 - min_x2 + 1) * (max_y2 - min_y2 + 1)
                    conflict_strength = min(overlap_area / (total_area / 2.0), 1.0) if total_area > 0 else 0.5
                    
                    constraint_id = f'nocross_{i}_{j}'
                    G.add_node(constraint_id, type=1, weight=conflict_strength)
                    
                    # Connect to both paths
                    if G.has_node(f'path_{i}') and G.has_node(f'path_{j}'):
                        G.add_edge(constraint_id, f'path_{i}', weight=conflict_strength)
                        G.add_edge(constraint_id, f'path_{j}', weight=conflict_strength)
    
    # Add cell adjacency relationships (routing constraints)
    for x in range(1, X+1):
        for y in range(1, Y+1):
            current_cell = f'cell_{x}_{y}'
            
            # Connect to neighboring cells with distance-based weights
            neighbors = []
            if x > 1: neighbors.append((x-1, y))
            if x < X: neighbors.append((x+1, y))
            if y > 1: neighbors.append((x, y-1))
            if y < Y: neighbors.append((x, y+1))
            
            for neighbor_x, neighbor_y in neighbors:
                neighbor_cell = f'cell_{neighbor_x}_{neighbor_y}'
                # Weight based on routing difficulty - central connections are tighter
                routing_weight = min(1.0, 0.3 + 0.7 * (len(neighbors) / 4.0))
                G.add_edge(current_cell, neighbor_cell, weight=routing_weight)
    
    # Add congestion constraint for high-density areas
    if N > 2:  # Only for non-trivial instances
        congestion_weight = min(N / max(X, Y), 1.0)
        G.add_node('congestion_constraint', type=1, weight=congestion_weight)
        
        # Connect congestion constraint to central cells
        center_x, center_y = (X + 1) / 2, (Y + 1) / 2
        for x in range(max(1, int(center_x - 1)), min(X+1, int(center_x + 2))):
            for y in range(max(1, int(center_y - 1)), min(Y+1, int(center_y + 2))):
                if G.has_node(f'cell_{x}_{y}'):
                    G.add_edge('congestion_constraint', f'cell_{x}_{y}', weight=congestion_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2] 
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()