#!/usr/bin/env python3
"""
Graph converter for areas problem.
Created using subagent_prompt.md version: v_02

This problem is about filling a grid where each region must have exactly its target size
and form contiguous areas. Key challenges: balancing region size constraints with
contiguity requirements, avoiding conflicts between regions of equal size.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the areas puzzle instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model grid cells as variables, regions and constraints as constraint nodes
    - Grid cells (type 0): decision points with weights based on constraints and position
    - Region constraints (type 1): ensure each region has exact target size
    - Contiguity constraints (type 1): ensure regions form connected areas
    - Equal-size exclusion constraints (type 1): prevent equal-sized regions from being adjacent
    - Distance constraints (type 1): prevent cells too far from region roots
    """
    # Access data from json_data dict
    w = json_data.get('w', 0)
    h = json_data.get('h', 0) 
    n = json_data.get('n', 0)
    region_size = json_data.get('region_size', [])
    puzzle = json_data.get('puzzle', [])
    
    # Create graph
    G = nx.Graph()
    
    # Helper function to convert 2D coordinates to 1D index
    def get_index(r, c):
        return r * w + c
    
    # Helper function to convert 1D index to 2D coordinates  
    def get_coords(idx):
        return (idx // w, idx % w)
    
    # Add grid cell nodes (type 0) with position and constraint-based weights
    total_cells = w * h
    center_r, center_c = h // 2, w // 2
    
    for r in range(h):
        for c in range(w):
            idx = get_index(r, c)
            cell_value = puzzle[idx] if idx < len(puzzle) else 0
            
            # Weight based on multiple factors:
            # 1. Distance from center (more central = more constrained)
            # 2. Whether pre-filled (pre-filled cells are more constrained)
            # 3. Number of potential regions that could reach this cell
            
            # Distance from center (normalized)
            dist_from_center = abs(r - center_r) + abs(c - center_c)
            max_dist = h//2 + w//2
            centrality = 1.0 - (dist_from_center / max(max_dist, 1))
            
            # Pre-filled cells are highly constrained
            if cell_value > 0:
                constraint_factor = 1.0
            else:
                # Count potential regions that could reach this cell
                reachable_regions = 0
                for region_idx in range(n):
                    region_num = region_idx + 1
                    target_size = region_size[region_idx] if region_idx < len(region_size) else 1
                    
                    # Find if there's a pre-filled cell of this region that can reach here
                    can_reach = False
                    for rr in range(h):
                        for cc in range(w):
                            root_idx = get_index(rr, cc)
                            if root_idx < len(puzzle) and puzzle[root_idx] == region_num:
                                manhattan_dist = abs(r - rr) + abs(c - cc)
                                if manhattan_dist < target_size:
                                    can_reach = True
                                    break
                        if can_reach:
                            break
                    
                    if can_reach:
                        reachable_regions += 1
                
                # More reachable regions = more constraint pressure
                constraint_factor = min(reachable_regions / max(n, 1), 1.0)
            
            # Combine factors with non-linear weighting
            weight = 0.4 * centrality + 0.6 * constraint_factor
            weight = min(max(weight, 0.1), 1.0)  # Ensure bounds
            
            G.add_node(f'cell_{r}_{c}', type=0, weight=weight)
    
    # Add region size constraint nodes (type 1)
    for region_idx in range(n):
        region_num = region_idx + 1
        target_size = region_size[region_idx] if region_idx < len(region_size) else 1
        
        # Weight by relative tightness of size constraint
        # Larger regions are generally easier to satisfy
        size_difficulty = math.exp(-target_size / max(total_cells / n, 1))
        
        G.add_node(f'region_size_{region_num}', type=1, weight=size_difficulty)
    
    # Add contiguity constraint nodes for each region (type 1)
    for region_idx in range(n):
        region_num = region_idx + 1
        target_size = region_size[region_idx] if region_idx < len(region_size) else 1
        
        # Contiguity is harder for larger, more spread out regions
        contiguity_difficulty = math.sqrt(target_size / max(total_cells, 1))
        
        G.add_node(f'contiguity_{region_num}', type=1, weight=contiguity_difficulty)
    
    # Add equal-size exclusion constraints (type 1)
    equal_size_groups = {}
    for region_idx in range(n):
        size = region_size[region_idx] if region_idx < len(region_size) else 1
        if size not in equal_size_groups:
            equal_size_groups[size] = []
        equal_size_groups[size].append(region_idx + 1)
    
    exclusion_id = 0
    for size, regions in equal_size_groups.items():
        if len(regions) > 1:
            # More regions of same size = harder constraint
            exclusion_difficulty = len(regions) / max(n, 1)
            G.add_node(f'equal_size_exclusion_{exclusion_id}', type=1, weight=exclusion_difficulty)
            exclusion_id += 1
    
    # Add distance constraint nodes for region roots (type 1)
    for r in range(h):
        for c in range(w):
            idx = get_index(r, c)
            if idx < len(puzzle) and puzzle[idx] > 0:
                region_num = puzzle[idx]
                region_idx = region_num - 1
                target_size = region_size[region_idx] if region_idx < len(region_size) else 1
                
                # Distance constraints are tighter for smaller regions
                distance_tightness = 1.0 / max(target_size, 1)
                
                G.add_node(f'distance_root_{r}_{c}', type=1, weight=distance_tightness)
    
    # Add edges: cell participation in constraints
    
    # Region size constraints - connect all cells to all region constraints
    for r in range(h):
        for c in range(w):
            cell_node = f'cell_{r}_{c}'
            for region_idx in range(n):
                region_num = region_idx + 1
                constraint_node = f'region_size_{region_num}'
                
                # Edge weight based on likelihood this cell belongs to this region
                idx = get_index(r, c)
                if idx < len(puzzle) and puzzle[idx] == region_num:
                    # Pre-filled cell definitely participates
                    weight = 1.0
                else:
                    # Weight by potential (inverse distance to nearest root)
                    min_dist = float('inf')
                    for rr in range(h):
                        for cc in range(w):
                            root_idx = get_index(rr, cc)
                            if root_idx < len(puzzle) and puzzle[root_idx] == region_num:
                                dist = abs(r - rr) + abs(c - cc)
                                min_dist = min(min_dist, dist)
                    
                    if min_dist == float('inf'):
                        weight = 0.1  # No root found
                    else:
                        target_size = region_size[region_idx] if region_idx < len(region_size) else 1
                        if min_dist >= target_size:
                            weight = 0.1  # Too far
                        else:
                            # Exponential decay with distance
                            weight = math.exp(-2.0 * min_dist / max(target_size, 1))
                
                G.add_edge(cell_node, constraint_node, weight=weight)
    
    # Contiguity constraints - connect cells to their potential region contiguity constraints
    for r in range(h):
        for c in range(w):
            cell_node = f'cell_{r}_{c}'
            # Connect to all region contiguity constraints with distance-based weights
            for region_idx in range(n):
                region_num = region_idx + 1
                contiguity_node = f'contiguity_{region_num}'
                
                # Calculate minimum distance to any root of this region
                min_dist = float('inf')
                for rr in range(h):
                    for cc in range(w):
                        root_idx = get_index(rr, cc)
                        if root_idx < len(puzzle) and puzzle[root_idx] == region_num:
                            dist = abs(r - rr) + abs(c - cc)
                            min_dist = min(min_dist, dist)
                
                if min_dist != float('inf'):
                    target_size = region_size[region_idx] if region_idx < len(region_size) else 1
                    if min_dist < target_size:
                        # Weight decreases with distance from root
                        weight = math.exp(-min_dist / max(target_size / 2, 1))
                        G.add_edge(cell_node, contiguity_node, weight=weight)
    
    # Equal-size exclusion constraints - connect neighboring cells
    exclusion_id = 0
    for size, regions in equal_size_groups.items():
        if len(regions) > 1:
            exclusion_node = f'equal_size_exclusion_{exclusion_id}'
            
            # Connect all cells that could participate in conflicts
            for r in range(h):
                for c in range(w):
                    cell_node = f'cell_{r}_{c}'
                    
                    # Check if this cell could belong to any region of this size
                    could_participate = False
                    for region_num in regions:
                        region_idx = region_num - 1
                        
                        # Check distance to any root of this region
                        for rr in range(h):
                            for cc in range(w):
                                root_idx = get_index(rr, cc)
                                if root_idx < len(puzzle) and puzzle[root_idx] == region_num:
                                    dist = abs(r - rr) + abs(c - cc)
                                    if dist < size:
                                        could_participate = True
                                        break
                            if could_participate:
                                break
                        if could_participate:
                            break
                    
                    if could_participate:
                        # Weight by how many neighboring cells this affects
                        neighbor_count = 0
                        for dr, dc in [(0,1), (0,-1), (1,0), (-1,0)]:
                            nr, nc = r + dr, c + dc
                            if 0 <= nr < h and 0 <= nc < w:
                                neighbor_count += 1
                        
                        weight = neighbor_count / 4.0  # Normalized by max neighbors
                        G.add_edge(cell_node, exclusion_node, weight=weight)
            
            exclusion_id += 1
    
    # Distance constraints - connect cells to their root distance constraints
    for r in range(h):
        for c in range(w):
            idx = get_index(r, c)
            if idx < len(puzzle) and puzzle[idx] > 0:
                region_num = puzzle[idx]
                region_idx = region_num - 1
                target_size = region_size[region_idx] if region_idx < len(region_size) else 1
                root_node = f'distance_root_{r}_{c}'
                
                # Connect all cells within range of this root
                for rr in range(h):
                    for cc in range(w):
                        dist = abs(rr - r) + abs(cc - c)
                        if dist < target_size:
                            cell_node = f'cell_{rr}_{cc}'
                            # Weight by inverse distance (closer = stronger constraint)
                            weight = 1.0 - (dist / max(target_size, 1))
                            G.add_edge(cell_node, root_node, weight=weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()