#!/usr/bin/env python3
"""
Graph converter for ATSP problem.
Converter created with subagent_prompt.md v_02

This problem is about manufacturing scheduling with moulds, colors, and programs.
A complex job-shop scheduling problem with resource constraints, demand fulfillment,
color compatibility, and setup time optimization.

Key challenges: Resource allocation, color compatibility, demand timing, setup optimization
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the ATSP manufacturing scheduling problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the complex interaction structure of manufacturing scheduling
    - Jobs as type 0 nodes (decision variables)
    - Constraints as type 1 nodes (capacity, compatibility, demand)
    - Resources as type 2 nodes (moulds, programs, lines)
    
    What makes instances hard:
    - High demand vs capacity ratios
    - Complex color compatibility patterns
    - Tight due dates vs production capacity
    - Resource contention between demands
    """
    # Extract problem data
    max_jobs = json_data.get('max_jobs', 1)
    num_moulds = json_data.get('num_moulds', 1)
    num_colors = json_data.get('num_colors', 1)
    num_programs = json_data.get('num_programs', 1)
    num_demands = json_data.get('num_demands', 1)
    num_lines = json_data.get('num_lines', 1)
    
    available_moulds = json_data.get('available_moulds', [])
    slots_per_program = json_data.get('slots_per_program', [])
    cycle_time_for_program = json_data.get('cycle_time_for_program', [])
    program_for_mould = json_data.get('program_for_mould', [])
    line_for_mould = json_data.get('line_for_mould', [])
    
    demand_qty = json_data.get('demand_qty', [])
    demand_duedate = json_data.get('demand_duedate', [])
    demand_color = json_data.get('demand_color', [])
    demand_mould = json_data.get('demand_mould', [])
    color_compatibility = json_data.get('color_compatibility', [])
    
    max_colors_per_job = json_data.get('max_colors_per_job', 1)
    min_cycles_per_job = json_data.get('min_cycles_per_job', 1)
    max_cycles_per_job = json_data.get('max_cycles_per_job', 1)
    
    G = nx.Graph()
    
    # Calculate problem statistics for meaningful weights
    total_demand = sum(demand_qty) if demand_qty else 1
    total_capacity = sum(available_moulds) if available_moulds else 1
    max_cycle_time = max(cycle_time_for_program) if cycle_time_for_program else 1
    min_cycle_time = min(cycle_time_for_program) if cycle_time_for_program else 1
    capacity_utilization = min(total_demand / max(total_capacity, 1), 2.0)
    
    # Type 0: Job nodes - decision variables with complexity-based weights
    for j in range(max_jobs):
        # Weight jobs by their potential complexity
        # Central jobs have more setup overhead, weight increases non-linearly
        setup_overhead = math.exp(-2.0 * j / max(max_jobs, 1))
        complexity = (capacity_utilization + setup_overhead) / 2.0
        G.add_node(f'job_{j}', type=0, weight=min(complexity, 1.0))
    
    # Type 2: Resource nodes
    # Mould resources with scarcity-based weights  
    for m in range(num_moulds):
        if m < len(available_moulds):
            # Scarcer moulds are more critical
            scarcity = 1.0 - (available_moulds[m] / max(max(available_moulds) if available_moulds else 1, 1))
            G.add_node(f'mould_{m}', type=2, weight=max(scarcity, 0.1))
        
    # Program resources with efficiency-based weights
    for p in range(num_programs):
        if p < len(cycle_time_for_program) and p < len(slots_per_program):
            # Programs with lower cycle time per slot are more efficient (higher weight)
            efficiency = 1.0 - (cycle_time_for_program[p] / max_cycle_time)
            slot_factor = slots_per_program[p] / max(max(slots_per_program) if slots_per_program else 1, 1)
            weight = (efficiency + slot_factor) / 2.0
            G.add_node(f'program_{p}', type=2, weight=max(weight, 0.1))
    
    # Line resources with load-based weights
    moulds_per_line = {}
    for m in range(min(len(line_for_mould), num_moulds)):
        line = line_for_mould[m] - 1  # Convert to 0-indexed
        if line not in moulds_per_line:
            moulds_per_line[line] = 0
        moulds_per_line[line] += 1
        
    for l in range(num_lines):
        load = moulds_per_line.get(l, 0)
        load_factor = load / max(num_moulds, 1)
        G.add_node(f'line_{l}', type=2, weight=max(load_factor, 0.1))
    
    # Type 1: Constraint nodes with tightness-based weights
    
    # Mould capacity constraints - one per mould
    for m in range(num_moulds):
        if m < len(available_moulds):
            # Calculate demand pressure on this mould
            mould_demand = sum(demand_qty[d] for d in range(len(demand_mould)) 
                             if d < len(demand_mould) and demand_mould[d] == m + 1)  # 1-indexed in data
            capacity = available_moulds[m]
            tightness = min(mould_demand / max(capacity, 1), 1.0)
            G.add_node(f'mould_capacity_{m}', type=1, weight=max(tightness, 0.1))
            
            # Connect to relevant jobs and mould resource
            G.add_edge(f'mould_capacity_{m}', f'mould_{m}', weight=1.0)
            for j in range(max_jobs):
                # Jobs that could use this mould (through program compatibility)
                if m < len(program_for_mould):
                    prog = program_for_mould[m] - 1  # Convert to 0-indexed
                    if prog < num_programs:
                        G.add_edge(f'job_{j}', f'mould_capacity_{m}', weight=0.7)
    
    # Demand fulfillment constraints - one per demand
    for d in range(num_demands):
        if (d < len(demand_qty) and d < len(demand_duedate) and 
            d < len(demand_color) and d < len(demand_mould)):
            
            # Weight by urgency and quantity
            qty = demand_qty[d]
            # Normalize duedate (negative means overdue, more urgent)
            max_duedate = max(demand_duedate) if demand_duedate else 0
            min_duedate = min(demand_duedate) if demand_duedate else 0
            if max_duedate != min_duedate:
                urgency = 1.0 - (demand_duedate[d] - min_duedate) / (max_duedate - min_duedate)
            else:
                urgency = 0.5
            
            quantity_factor = qty / max(total_demand, 1)
            weight = min((urgency + quantity_factor) / 2.0, 1.0)
            G.add_node(f'demand_{d}', type=1, weight=max(weight, 0.1))
            
            # Connect to jobs and relevant resources
            for j in range(max_jobs):
                # Weight connection by how much this job could contribute
                contribution = min(qty / max(total_demand, 1) * 2.0, 1.0)
                G.add_edge(f'job_{j}', f'demand_{d}', weight=contribution)
            
            # Connect to mould and color resources
            mould_idx = demand_mould[d] - 1  # Convert to 0-indexed
            if 0 <= mould_idx < num_moulds:
                G.add_edge(f'demand_{d}', f'mould_{mould_idx}', weight=0.8)
    
    # Color compatibility constraint (global constraint)
    # Calculate color compatibility complexity
    if color_compatibility:
        # Flatten if it's a 2D array
        if isinstance(color_compatibility[0], list):
            compat_values = [val for row in color_compatibility for val in row]
        else:
            compat_values = color_compatibility
            
        incompatible_pairs = sum(1 for val in compat_values if val == 0)
        total_pairs = len(compat_values)
        complexity = incompatible_pairs / max(total_pairs, 1) if total_pairs > 0 else 0.5
        
        G.add_node('color_compatibility', type=1, weight=max(complexity, 0.1))
        
        # Connect to all jobs since they all must respect color compatibility
        for j in range(max_jobs):
            G.add_edge(f'job_{j}', 'color_compatibility', weight=0.6)
    
    # Program-job assignment constraints
    for j in range(max_jobs):
        G.add_node(f'job_program_{j}', type=1, weight=0.5)
        G.add_edge(f'job_{j}', f'job_program_{j}', weight=1.0)
        
        # Connect to program resources
        for p in range(num_programs):
            G.add_edge(f'job_program_{j}', f'program_{p}', weight=0.4)
    
    # Add conflict edges for highly contentious resources
    # Jobs competing for scarce moulds
    for m in range(num_moulds):
        if m < len(available_moulds):
            mould_demand = sum(demand_qty[d] for d in range(len(demand_mould)) 
                             if d < len(demand_mould) and demand_mould[d] == m + 1)
            capacity = available_moulds[m]
            
            if mould_demand > capacity * 1.5:  # Oversubscribed resource
                # Add conflict edges between jobs that could compete for this mould
                competing_jobs = []
                for j in range(min(max_jobs, 5)):  # Limit to reduce complexity
                    competing_jobs.append(j)
                
                for i, j1 in enumerate(competing_jobs):
                    for j2 in competing_jobs[i+1:]:
                        conflict_weight = min(mould_demand / max(capacity, 1) - 1.0, 1.0)
                        if conflict_weight > 0:
                            G.add_edge(f'job_{j1}', f'job_{j2}', weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()