#!/usr/bin/env python3
"""
Graph converter for Black Hole Patience problem.
# Converter created with subagent_prompt.md v_02

This problem is about sequencing playing cards from 17 piles of 3 cards each.
Key challenges: card ordering constraints, pile precedence rules, limited valid moves.
The black hole patience game requires finding a sequence where consecutive cards
are neighbors in the card adjacency graph, while respecting pile ordering.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Black Hole Patience instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model card sequencing with pile constraints
    - Card nodes (type 0): 52 playing cards with position-based weights
    - Constraint nodes (type 1): Pile precedence constraints and adjacency rules
    - Edges capture card relationships and precedence dependencies
    
    The difficulty comes from limited valid moves and pile precedence constraints.
    Cards deeper in piles are harder to access, creating bottlenecks.
    """
    # Extract layout - 51 cards in 17 piles of 3 each (excludes Ace of Spades)
    layout = json_data.get('layout', [])
    if not layout:
        return nx.Graph()  # Empty graph for invalid input
    
    # Reconstruct the 17x3 pile structure from linear layout
    piles = []
    for pile_idx in range(17):
        pile_cards = []
        for depth in range(3):
            card_idx = pile_idx * 3 + depth
            if card_idx < len(layout):
                pile_cards.append(layout[card_idx])
        piles.append(pile_cards)
    
    G = nx.Graph()
    
    # Add card nodes (type 0) with accessibility-based weights
    # Ace of Spades (card 1) has maximum weight as it's always first
    G.add_node('card_1', type=0, weight=1.0)
    
    # Other cards weighted by accessibility (deeper = harder to reach)
    for pile_idx, pile in enumerate(piles):
        for depth, card in enumerate(pile):
            if card and card != 1:  # Skip empty and Ace of Spades
                # Weight decreases with depth and increases with pile centrality
                pile_centrality = 1.0 - abs(pile_idx - 8.5) / 8.5  # Center piles more accessible
                depth_penalty = math.exp(-2.0 * depth)  # Exponential penalty for depth
                accessibility = pile_centrality * depth_penalty
                G.add_node(f'card_{card}', type=0, weight=min(accessibility * 1.2, 1.0))
    
    # Add pile precedence constraint nodes (type 1)
    for pile_idx, pile in enumerate(piles):
        for depth in range(len(pile) - 1):  # Only need depth-1 constraints per pile
            if depth < len(pile) - 1 and pile[depth] and pile[depth + 1]:
                card_top = pile[depth]
                card_below = pile[depth + 1]
                constraint_id = f'precedence_pile_{pile_idx}_depth_{depth}'
                
                # Weight constraint by how restrictive it is (deeper constraints more restrictive)
                restrictiveness = math.exp(depth * 0.5) / math.exp(1.5)  # Normalized exponential
                G.add_node(constraint_id, type=1, weight=restrictiveness)
                
                # Connect constraint to both cards involved
                G.add_edge(f'card_{card_top}', constraint_id, weight=0.8)
                G.add_edge(f'card_{card_below}', constraint_id, weight=0.9)  # Higher weight for blocked card
    
    # Add adjacency constraint node for the global card sequencing rule
    G.add_node('adjacency_constraint', type=1, weight=0.9)  # High weight for global constraint
    
    # Connect all cards to the adjacency constraint
    # Weight by how many valid neighbors each card has (fewer neighbors = more constrained)
    neighbor_counts = {}
    
    # Count neighbors from the neighbors table (hardcoded in MZN file)
    # Simplified: assume each card has approximately 8 neighbors (based on card deck structure)
    # This is an approximation since the full neighbor table is hardcoded
    for pile in piles:
        for card in pile:
            if card and card != 1:
                # Estimate neighbors based on card value patterns
                # Cards with values near boundaries (1, 13, 26, 39, 52) have fewer neighbors
                card_mod = (card - 1) % 13 + 1  # Card rank (1-13)
                if card_mod == 1 or card_mod == 13:  # Aces and Kings
                    neighbor_ratio = 0.6  # Fewer neighbors
                elif card_mod in [2, 12]:  # 2s and Queens  
                    neighbor_ratio = 0.8
                else:
                    neighbor_ratio = 1.0  # Most cards have full neighbors
                
                G.add_edge(f'card_{card}', 'adjacency_constraint', weight=neighbor_ratio)
    
    # Connect Ace of Spades to adjacency constraint with high weight (fixed starting point)
    G.add_edge('card_1', 'adjacency_constraint', weight=1.0)
    
    # Add some pile interaction edges for cards that might create conflicts
    # Connect cards from different piles that are similar values (potential sequence conflicts)
    for pile1_idx, pile1 in enumerate(piles):
        for pile2_idx, pile2 in enumerate(piles):
            if pile1_idx < pile2_idx:  # Avoid duplicate edges
                for card1 in pile1:
                    for card2 in pile2:
                        if card1 and card2:
                            # Create conflict edges for cards with similar values that might compete
                            card1_rank = (card1 - 1) % 13 + 1
                            card2_rank = (card2 - 1) % 13 + 1
                            
                            if abs(card1_rank - card2_rank) <= 1:  # Adjacent ranks
                                # Weight by how close the piles are (closer piles more likely to conflict)
                                pile_distance = abs(pile1_idx - pile2_idx) / 16.0
                                conflict_weight = math.exp(-2.0 * pile_distance) * 0.5
                                if conflict_weight > 0.1:
                                    G.add_edge(f'card_{card1}', f'card_{card2}', weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()