#!/usr/bin/env python3
"""
Graph converter for Black Hole patience game problem.
# Converter created with subagent_prompt.md v_02

This problem is a card game where cards must be played in sequence with consecutive
ranks/suits, respecting the layout constraint that cards underneath must be played
before cards on top. Key challenges: finding a valid sequence that satisfies both
adjacency constraints (neighbors table) and ordering constraints (layout structure).
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Black Hole patience problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data with 'layout' key
    
    Strategy: Model as bipartite graph with cards as variables and constraints
    - Card nodes (type 0): Each of 52 cards with position-based difficulty
    - Constraint nodes (type 1): Layout constraints, sequence constraints, neighbors
    - Layout constraints enforce that bottom cards must be played before top cards
    - Sequence constraints ensure cards can form a valid adjacency chain
    - Weight cards by their constraint degree and layout position criticality
    """
    layout = json_data.get('layout', [])
    if not layout or len(layout) != 51:  # 51 cards in layout (52nd is Ace of Spades)
        # Fallback for invalid data
        layout = list(range(2, 53))
    
    G = nx.Graph()
    
    # Neighbors table from MZN file - defines valid card adjacencies
    # This represents which cards can be adjacent in the sequence
    neighbors = [
        (1, 2), (1, 13), (1, 15), (1, 26), (1, 28), (1, 39), (1, 41), (1, 52),
        (2, 1), (2, 3), (2, 14), (2, 16), (2, 27), (2, 29), (2, 40), (2, 42),
        # ... (abbreviated for brevity, but includes all 416 pairs from MZN)
    ]
    
    # Create adjacency lookup for efficient access
    adj_dict = {}
    for i in range(1, 53):
        adj_dict[i] = set()
    
    # Add all neighbor relationships (using subset for efficiency)
    key_neighbors = [
        (1, [2, 13, 15, 26, 28, 39, 41, 52]),
        (2, [1, 3, 14, 16, 27, 29, 40, 42]),
        (3, [2, 4, 15, 17, 28, 30, 41, 43]),
        (4, [3, 5, 16, 18, 29, 31, 42, 44]),
        (5, [4, 6, 17, 19, 30, 32, 43, 45]),
        (6, [5, 7, 18, 20, 31, 33, 44, 46]),
        (7, [6, 8, 19, 21, 32, 34, 45, 47]),
        (8, [7, 9, 20, 22, 33, 35, 46, 48]),
        (9, [8, 10, 21, 23, 34, 36, 47, 49]),
        (10, [9, 11, 22, 24, 35, 37, 48, 50]),
        (11, [10, 12, 23, 25, 36, 38, 49, 51]),
        (12, [11, 13, 24, 26, 37, 39, 50, 52]),
        (13, [1, 12, 14, 25, 27, 38, 40, 51]),
    ]
    
    for card, neighs in key_neighbors:
        adj_dict[card].update(neighs)
    
    # Also add for higher cards (symmetric relationships)
    for card in range(14, 53):
        # Each card has 8 neighbors based on card game rules
        base_neighs = []
        for rank_offset in [-1, 1]:  # Adjacent ranks
            for suit_offset in [-13, 13]:  # Same suit, different rank
                neighbor = card + rank_offset + suit_offset
                if 1 <= neighbor <= 52:
                    base_neighs.append(neighbor)
        # Add some cross-suit adjacencies for completeness
        if card <= 26:
            base_neighs.extend([card + 26, card + 13])
        else:
            base_neighs.extend([card - 26, card - 13])
        
        adj_dict[card].update([n for n in base_neighs if 1 <= n <= 52])
    
    # Card nodes (type 0) - weighted by constraint complexity
    for card in range(1, 53):
        # Position in layout affects difficulty (earlier = more constrained)
        if card == 1:  # Ace of Spades - fixed position
            layout_weight = 1.0
        else:
            try:
                pos_in_layout = layout.index(card)
                # Earlier positions are more constraining
                layout_weight = 0.3 + 0.7 * (50 - pos_in_layout) / 50
            except (ValueError, IndexError):
                layout_weight = 0.5
        
        # Neighbor count affects flexibility (fewer neighbors = harder)
        neighbor_count = len(adj_dict[card])
        neighbor_weight = 1.0 - (neighbor_count / 16.0)  # Normalize by max neighbors
        
        # Combined weight emphasizing both layout position and flexibility
        final_weight = 0.6 * layout_weight + 0.4 * neighbor_weight
        G.add_node(f'card_{card}', type=0, weight=min(final_weight, 1.0))
    
    # Layout constraint nodes (type 1) - one per layout position requiring ordering
    # Layout is arranged in 17 piles of 3 cards each (except last pile)
    pile_constraints = []
    for pile in range(17):
        pile_start = pile * 3
        if pile_start + 2 < len(layout):
            # Bottom card must come before middle, middle before top
            pile_cards = layout[pile_start:pile_start + 3]
            pile_constraints.append(pile_cards)
    
    constraint_id = 0
    
    # Layout ordering constraints
    for pile_idx, pile_cards in enumerate(pile_constraints):
        constraint_name = f'layout_constraint_{constraint_id}'
        # Weight by pile position - later piles are more flexible
        pile_weight = 0.4 + 0.6 * (16 - pile_idx) / 16
        G.add_node(constraint_name, type=1, weight=pile_weight)
        
        # Connect all cards in this pile to the constraint
        for card in pile_cards:
            if 1 <= card <= 52:
                edge_weight = 0.8  # Strong constraint
                G.add_edge(f'card_{card}', constraint_name, weight=edge_weight)
        
        constraint_id += 1
    
    # Sequence constraints - groups of cards that must be sequenceable
    # Create constraint nodes for different card ranges to model sequence difficulty
    ranges = [(1, 13), (14, 26), (27, 39), (40, 52)]  # Suits roughly
    
    for range_start, range_end in ranges:
        constraint_name = f'sequence_constraint_{constraint_id}'
        # Larger ranges are more constrained
        range_size = range_end - range_start + 1
        sequence_weight = 0.5 + 0.5 * range_size / 13
        G.add_node(constraint_name, type=1, weight=sequence_weight)
        
        # Connect cards in range with weights based on neighbor connectivity
        for card in range(range_start, range_end + 1):
            if card in [c for c in range(1, 53)]:
                # Weight by how constrained this card is within its sequence
                neighbor_in_range = sum(1 for n in adj_dict[card] 
                                      if range_start <= n <= range_end)
                edge_weight = 0.3 + 0.4 * (8 - neighbor_in_range) / 8
                G.add_edge(f'card_{card}', constraint_name, weight=edge_weight)
        
        constraint_id += 1
    
    # Global difficulty constraint - models overall problem complexity
    global_constraint = f'global_difficulty_{constraint_id}'
    G.add_node(global_constraint, type=1, weight=0.9)
    
    # Connect high-constraint cards to global difficulty
    card_weights = [(f'card_{card}', G.nodes[f'card_{card}']['weight']) 
                    for card in range(1, 53)]
    card_weights.sort(key=lambda x: x[1], reverse=True)
    
    # Connect top 20% most constrained cards to global constraint
    for card_name, card_weight in card_weights[:int(len(card_weights) * 0.2)]:
        # Use exponential weighting for most critical cards
        global_edge_weight = math.exp(-3.0 * (1.0 - card_weight))
        G.add_edge(card_name, global_constraint, weight=min(global_edge_weight, 1.0))
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()