#!/usr/bin/env python3
"""
Graph converter for Blocks World problem.
Created using subagent_prompt.md version: v_02

This problem is about rearranging numbered blocks between k piles to reach a goal configuration.
Key challenges: blocks that need to move multiple times, blocks blocking others, cascade effects.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Blocks World problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model blocks as variables, piles as resources, and constraints for:
    - Blocks that must move (misplaced constraint)
    - Pile capacity limits (stack constraints) 
    - Blocking relationships (a block on top blocks access to blocks below)
    - Movement complexity (how many moves a block likely needs)
    
    The graph captures movement difficulty and blocking relationships that make
    instances hard to solve.
    """
    n = json_data.get('n', 0)  # number of blocks
    k = json_data.get('k', 0)  # number of piles
    start = json_data.get('start', [])
    end = json_data.get('end', [])
    
    G = nx.Graph()
    
    # Count blocks in each pile for start and end configurations
    max_pile = max(k, max(start + end) if start and end else k)
    start_pile_counts = [0] * (max_pile + 1)  # pile 0 is empty space
    end_pile_counts = [0] * (max_pile + 1)
    
    for i in range(min(n, len(start))):
        pile = start[i]
        if 0 <= pile <= max_pile:
            start_pile_counts[pile] += 1
    
    for i in range(min(n, len(end))):
        pile = end[i] 
        if 0 <= pile <= max_pile:
            end_pile_counts[pile] += 1
    
    # Add block nodes (type 0) with movement complexity weights
    for i in range(n):
        block_id = i + 1  # blocks are 1-indexed
        
        start_pile = start[i] if i < len(start) else 0
        end_pile = end[i] if i < len(end) else 0
        
        # Calculate movement complexity
        must_move = start_pile != end_pile
        
        # Blocks in dense piles are harder to move (more blocking)
        start_density = start_pile_counts[start_pile] / n if start_pile > 0 and start_pile < len(start_pile_counts) else 0
        end_density = end_pile_counts[end_pile] / n if end_pile > 0 and end_pile < len(end_pile_counts) else 0
        
        # Higher weight for blocks that must move and are in complex configurations
        base_weight = 0.3
        if must_move:
            base_weight = 0.7
            # Add density penalty (more blocks around = harder to move)
            density_penalty = (start_density + end_density) * 0.3
            base_weight = min(1.0, base_weight + density_penalty)
        
        G.add_node(f'block_{block_id}', type=0, weight=base_weight)
    
    # Add pile resource nodes (type 2) with scarcity weights
    # Need to create nodes for all piles actually used in the data
    used_piles = set()
    for pile in start + end:
        if pile > 0:  # pile 0 is not a real pile
            used_piles.add(pile)
    
    for pile in used_piles:
        start_load = start_pile_counts[pile] if pile < len(start_pile_counts) else 0
        end_load = end_pile_counts[pile] if pile < len(end_pile_counts) else 0
        max_load = max(start_load, end_load)
        
        # Pile weight based on utilization - heavily used piles are bottlenecks
        utilization = max_load / n if n > 0 else 0.5
        # Use logarithmic scaling for pile importance
        weight = min(1.0, 0.3 + math.log(1 + utilization * 5) / math.log(6))
        
        G.add_node(f'pile_{pile}', type=2, weight=weight)
    
    # Add constraint nodes (type 1) for different types of constraints
    
    # 1. Movement constraints - one per block that must move
    misplaced_blocks = 0
    for i in range(n):
        block_id = i + 1
        start_pile = start[i] if i < len(start) else 0
        end_pile = end[i] if i < len(end) else 0
        
        if start_pile != end_pile:
            misplaced_blocks += 1
            # Weight by estimated movement difficulty
            movement_distance = abs(start_pile - end_pile) if start_pile > 0 and end_pile > 0 else 1
            # Normalize by maximum possible distance and use exponential scaling
            max_distance = k
            difficulty = 1.0 - math.exp(-2.0 * movement_distance / max_distance)
            
            constraint_id = f'move_constraint_{block_id}'
            G.add_node(constraint_id, type=1, weight=difficulty)
            
            # Connect block to its movement constraint
            G.add_edge(f'block_{block_id}', constraint_id, weight=0.9)
            
            # Connect to source and destination piles
            if start_pile > 0:
                G.add_edge(constraint_id, f'pile_{start_pile}', weight=0.7)
            if end_pile > 0:
                G.add_edge(constraint_id, f'pile_{end_pile}', weight=0.8)
    
    # 2. Pile capacity constraints - model congestion
    for pile in used_piles:
        start_load = start_pile_counts[pile] if pile < len(start_pile_counts) else 0
        end_load = end_pile_counts[pile] if pile < len(end_pile_counts) else 0
        
        if start_load > 1 or end_load > 1:  # Only create if pile has multiple blocks
            max_load = max(start_load, end_load)
            # Congestion increases super-linearly
            congestion = min(1.0, (max_load / n) ** 0.5 * 1.5)
            
            constraint_id = f'pile_capacity_{pile}'
            G.add_node(constraint_id, type=1, weight=congestion)
            G.add_edge(constraint_id, f'pile_{pile}', weight=1.0)
            
            # Connect all blocks that use this pile
            for i in range(n):
                block_id = i + 1
                start_pile = start[i] if i < len(start) else 0
                end_pile = end[i] if i < len(end) else 0
                
                if start_pile == pile or end_pile == pile:
                    # Weight based on how much this block contributes to congestion
                    contribution = 1.0 / max(max_load, 1)
                    G.add_edge(f'block_{block_id}', constraint_id, weight=contribution)
    
    # 3. Add blocking relationships between blocks in same pile
    # Group blocks by pile for both start and end configurations
    pile_groups = {}
    
    for config_name, config in [('start', start), ('end', end)]:
        for i in range(n):
            block_id = i + 1
            pile = config[i] if i < len(config) else 0
            
            if pile > 0:  # Only real piles
                key = f'{config_name}_pile_{pile}'
                if key not in pile_groups:
                    pile_groups[key] = []
                pile_groups[key].append(block_id)
    
    # Add blocking edges between blocks that share piles
    for group_key, blocks in pile_groups.items():
        if len(blocks) > 1:
            # In a pile, blocks can block each other's movement
            for i, block1 in enumerate(blocks):
                for j, block2 in enumerate(blocks[i+1:], i+1):
                    # Blocks in same pile have potential blocking relationship
                    # Weight higher if both blocks need to move
                    block1_moves = start[block1-1] != end[block1-1] if block1-1 < len(start) and block1-1 < len(end) else False
                    block2_moves = start[block2-1] != end[block2-1] if block2-1 < len(start) and block2-1 < len(end) else False
                    
                    if block1_moves and block2_moves:
                        blocking_strength = 0.6  # Both need to move - high conflict
                    elif block1_moves or block2_moves:
                        blocking_strength = 0.4  # One needs to move
                    else:
                        blocking_strength = 0.2  # Stable but still related
                    
                    G.add_edge(f'block_{block1}', f'block_{block2}', weight=blocking_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()