#!/usr/bin/env python3
"""
Graph converter for cargo_coarsePiles problem.
Created using subagent_prompt.md version: v_02

This problem is about scheduling cargo piles from vessels onto a storage pad.
Key challenges: resource conflicts (stacker capacity, pad space, reclaimers), temporal precedence, vessel deadlines.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the cargo stacking problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with explicit constraint nodes
    - Pile nodes (type 0): cargo piles with reclaiming duration and vessel assignment
    - Constraint nodes (type 1): capacity, precedence, spatial, and vessel constraints  
    - Resource nodes (type 2): stacker capacity, pad space, reclaimers
    - Edge weights capture resource consumption and constraint tightness
    """
    # Extract problem data
    nV = json_data.get('nV', 0)  # vessels
    nS = json_data.get('nS', 0)  # piles
    H = json_data.get('H', 1)    # pad height
    T = json_data.get('T', 1)    # time horizon
    stCap = json_data.get('stCap', 1)       # stacker capacity
    reclN = json_data.get('reclN', 1)       # number of reclaimers
    
    eta = json_data.get('eta', [])          # vessel arrival times
    whichV = json_data.get('whichV', [])    # pile to vessel mapping
    dS__ = json_data.get('dS__', [])        # stacking durations (discretized)
    dR = json_data.get('dR', [])            # reclaiming durations
    
    discrStackStart = json_data.get('discrStackStart', 1)
    discrPadPos = json_data.get('discrPadPos', 1)
    mulTonnage = json_data.get('mulTonnage', 1)
    mulPileLen = json_data.get('mulPileLen', 1)
    hourDiscr = json_data.get('hourDiscr', 1)
    
    G = nx.Graph()
    
    # Calculate derived values for weights
    max_dR = max(dR) if dR else 1
    max_dS = max(dS__) if dS__ else 1
    max_eta = max(eta) if eta else T
    
    # Compute pile-specific metrics
    stTonnagePerHour = []
    pile_lengths = []
    for o in range(nS):
        if o < len(dR) and o < len(dS__) and dS__[o] > 0:
            tonnage = dR[o] * mulTonnage // (dS__[o] * discrStackStart)
            length = dR[o] * mulPileLen // hourDiscr
        else:
            tonnage = 0
            length = 0
        stTonnagePerHour.append(tonnage)
        pile_lengths.append(length)
    
    max_tonnage = max(stTonnagePerHour) if stTonnagePerHour else 1
    max_length = max(pile_lengths) if pile_lengths else 1
    
    # Calculate total reclaim duration per vessel
    dRtotal = [0] * nV
    for o in range(nS):
        if o < len(whichV) and o < len(dR) and whichV[o] <= nV:
            vessel_idx = whichV[o] - 1  # Convert to 0-based
            if vessel_idx >= 0:
                dRtotal[vessel_idx] += dR[o]
    
    # Type 0: Pile nodes (variable-like)
    for o in range(nS):
        pile_weight = 0.5  # Base weight
        
        # Weight by processing intensity (higher = more critical)
        if o < len(dR) and max_dR > 0:
            reclaim_intensity = dR[o] / max_dR
            pile_weight = 0.3 + 0.4 * reclaim_intensity
            
        # Boost weight for piles with high tonnage throughput
        if o < len(stTonnagePerHour) and max_tonnage > 0:
            tonnage_ratio = stTonnagePerHour[o] / max_tonnage
            pile_weight = min(1.0, pile_weight + 0.3 * tonnage_ratio)
            
        G.add_node(f'pile_{o}', type=0, weight=pile_weight)
    
    # Type 2: Resource nodes
    # Stacker capacity resource
    total_tonnage_demand = sum(stTonnagePerHour)
    stacker_utilization = total_tonnage_demand / (stCap * nS) if stCap > 0 and nS > 0 else 0.5
    stacker_stress = min(1.0, stacker_utilization)
    G.add_node('stacker_resource', type=2, weight=stacker_stress)
    
    # Pad space resource  
    total_length_demand = sum(pile_lengths)
    pad_utilization = total_length_demand / (H * nS) if H > 0 and nS > 0 else 0.5
    pad_stress = min(1.0, pad_utilization)
    G.add_node('pad_resource', type=2, weight=pad_stress)
    
    # Reclaimer resource
    total_reclaim_demand = sum(dR)
    reclaim_utilization = total_reclaim_demand / (reclN * T) if reclN > 0 and T > 0 else 0.5
    reclaim_stress = min(1.0, reclaim_utilization)
    G.add_node('reclaimer_resource', type=2, weight=reclaim_stress)
    
    # Type 1: Constraint nodes
    
    # Vessel deadline constraints (one per vessel)
    for v in range(nV):
        vessel_idx = v  # Already 0-based
        if vessel_idx < len(eta) and vessel_idx < len(dRtotal):
            # Tightness based on how much work vs available time
            available_time = T - eta[vessel_idx] if eta[vessel_idx] < T else 1
            required_time = dRtotal[vessel_idx]
            tightness = min(1.0, required_time / available_time) if available_time > 0 else 0.9
        else:
            tightness = 0.5
        G.add_node(f'vessel_deadline_{v}', type=1, weight=tightness)
    
    # Precedence constraints between consecutive piles of same vessel
    precedence_count = 0
    for v in range(1, nV + 1):  # 1-based vessel numbering
        vessel_piles = [(o, whichV[o]) for o in range(nS) if o < len(whichV) and whichV[o] == v]
        vessel_piles.sort()
        
        for i in range(len(vessel_piles) - 1):
            pile_a, pile_b = vessel_piles[i][0], vessel_piles[i+1][0]
            # Weight by how tight the precedence is (longer processing = tighter)
            if pile_a < len(dR) and pile_b < len(dR):
                precedence_weight = (dR[pile_a] + dR[pile_b]) / (2 * max_dR) if max_dR > 0 else 0.5
            else:
                precedence_weight = 0.5
            G.add_node(f'precedence_{precedence_count}', type=1, weight=precedence_weight)
            precedence_count += 1
    
    # Cumulative stacker capacity constraint (global)
    G.add_node('stacker_capacity', type=1, weight=stacker_stress)
    
    # Cumulative pad space constraint (global) 
    G.add_node('pad_capacity', type=1, weight=pad_stress)
    
    # Cumulative reclaimer constraint (global)
    G.add_node('reclaimer_capacity', type=1, weight=reclaim_stress)
    
    # Non-overlap (diffn) constraint - represents spatial conflicts
    spatial_conflict_weight = min(1.0, total_length_demand / H) if H > 0 else 0.5
    G.add_node('spatial_nonoverlapw', type=1, weight=spatial_conflict_weight)
    
    # Add edges
    
    # Pile to vessel deadline constraints
    for o in range(nS):
        if o < len(whichV):
            vessel = whichV[o] - 1  # Convert to 0-based
            if 0 <= vessel < nV:
                # Edge weight by pile's contribution to vessel workload
                if o < len(dR) and vessel < len(dRtotal) and dRtotal[vessel] > 0:
                    workload_ratio = dR[o] / dRtotal[vessel]
                else:
                    workload_ratio = 1.0 / nS if nS > 0 else 0.5
                G.add_edge(f'pile_{o}', f'vessel_deadline_{vessel}', weight=workload_ratio)
    
    # Pile to precedence constraints
    precedence_count = 0
    for v in range(1, nV + 1):  # 1-based vessel numbering
        vessel_piles = [(o, whichV[o]) for o in range(nS) if o < len(whichV) and whichV[o] == v]
        vessel_piles.sort()
        
        for i in range(len(vessel_piles) - 1):
            pile_a, pile_b = vessel_piles[i][0], vessel_piles[i+1][0]
            # Connect both piles to this precedence constraint
            G.add_edge(f'pile_{pile_a}', f'precedence_{precedence_count}', weight=0.8)
            G.add_edge(f'pile_{pile_b}', f'precedence_{precedence_count}', weight=0.8)
            precedence_count += 1
    
    # Pile to resource constraints
    for o in range(nS):
        # Stacker capacity usage
        if o < len(stTonnagePerHour) and max_tonnage > 0:
            usage_ratio = stTonnagePerHour[o] / max_tonnage
        else:
            usage_ratio = 1.0 / nS if nS > 0 else 0.5
        G.add_edge(f'pile_{o}', 'stacker_capacity', weight=usage_ratio)
        G.add_edge(f'pile_{o}', 'stacker_resource', weight=usage_ratio)
        
        # Pad space usage
        if o < len(pile_lengths) and max_length > 0:
            length_ratio = pile_lengths[o] / max_length
        else:
            length_ratio = 1.0 / nS if nS > 0 else 0.5
        G.add_edge(f'pile_{o}', 'pad_capacity', weight=length_ratio)
        G.add_edge(f'pile_{o}', 'pad_resource', weight=length_ratio)
        G.add_edge(f'pile_{o}', 'spatial_nonoverlapw', weight=length_ratio)
        
        # Reclaimer usage
        if o < len(dR) and max_dR > 0:
            reclaim_ratio = dR[o] / max_dR
        else:
            reclaim_ratio = 1.0 / nS if nS > 0 else 0.5
        G.add_edge(f'pile_{o}', 'reclaimer_capacity', weight=reclaim_ratio)
        G.add_edge(f'pile_{o}', 'reclaimer_resource', weight=reclaim_ratio)
    
    # Add conflict edges between piles that heavily compete for oversubscribed resources
    if total_tonnage_demand > stCap * 0.8:  # Stacker oversubscribed
        high_tonnage_piles = [(o, stTonnagePerHour[o]) for o in range(min(nS, len(stTonnagePerHour))) 
                              if stTonnagePerHour[o] > max_tonnage * 0.6]
        for i in range(len(high_tonnage_piles)):
            for j in range(i + 1, min(len(high_tonnage_piles), i + 4)):  # Limit conflicts
                o1, tonnage1 = high_tonnage_piles[i]
                o2, tonnage2 = high_tonnage_piles[j]
                conflict_weight = (tonnage1 + tonnage2) / (2 * stCap) if stCap > 0 else 0.5
                G.add_edge(f'pile_{o1}', f'pile_{o2}', weight=min(1.0, conflict_weight))
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()