#!/usr/bin/env python3
"""
Graph converter for cargo_coarsePiles_2 problem.
Converter created with subagent_prompt.md v_02

This problem is about cargo pile management in port terminals where vessels arrive with cargo
that must be stacked in piles and later reclaimed. Key challenges: spatial arrangement of piles,
temporal sequencing of operations, resource capacity limits, and minimizing vessel delays.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the cargo pile management problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with explicit constraint modeling
    - Piles (type 0): Decision variables for pile placement/timing
    - Vessels (type 2): Resource entities with capacity/time constraints  
    - Constraints (type 1): Spatial, temporal, and capacity constraints
    - Focus on pile-pile conflicts, vessel-pile relationships, and resource bottlenecks
    """
    # Extract problem parameters
    nV = json_data.get('nV', 0)  # number of vessels
    nS = json_data.get('nS', 0)  # number of piles
    H = json_data.get('H', 1)    # height capacity
    T = json_data.get('T', 1)    # time horizon
    stCap = json_data.get('stCap', 1)  # stack capacity
    reclN = json_data.get('reclN', 1)  # reclaim capacity
    
    eta = json_data.get('eta', [])      # vessel arrival times
    whichV = json_data.get('whichV', [])  # pile to vessel mapping
    dS__ = json_data.get('dS__', [])    # pile stacking durations
    dR = json_data.get('dR', [])        # pile reclaim durations
    
    discrPadPos = json_data.get('discrPadPos', 1)
    discrStackStart = json_data.get('discrStackStart', 1)
    mulTonnage = json_data.get('mulTonnage', 1)
    mulPileLen = json_data.get('mulPileLen', 1)
    hourDiscr = json_data.get('hourDiscr', 1)
    delayMax = json_data.get('delayMax', 1)
    
    G = nx.Graph()
    
    # Calculate derived values (as in the MZN model)
    max_dR = max(dR) if dR else 1
    max_dS = max(dS__) if dS__ else 1
    max_eta = max(eta) if eta else 1
    
    # Pile nodes (type 0) - decision variables for pile placement/timing
    for pile in range(nS):
        if pile < len(dR) and pile < len(dS__):
            # Weight by processing intensity (reclaim duration / stacking duration ratio)
            dR_val = dR[pile] if pile < len(dR) else 1
            dS_val = dS__[pile] if pile < len(dS__) else 1
            processing_ratio = dR_val / max(dS_val, 1)  # avoid division by zero
            intensity_weight = math.tanh(processing_ratio / 3.0)  # non-linear scaling
            
            # Adjust by pile size relative to capacity
            size_factor = dR_val / max_dR if max_dR > 0 else 0.5
            combined_weight = (intensity_weight * 0.7 + size_factor * 0.3)
            
            G.add_node(f'pile_{pile}', type=0, weight=min(combined_weight, 1.0))
    
    # Vessel nodes (type 2) - resource entities
    vessel_delays = {}
    for vessel in range(nV):
        if vessel < len(eta):
            # Calculate total workload for this vessel
            vessel_piles = [i for i in range(len(whichV)) if whichV[i] == vessel + 1]
            total_work = sum(dR[pile] for pile in vessel_piles if pile < len(dR))
            
            # Weight by urgency (earlier vessels have higher weight) and workload
            if max_eta > 0:
                urgency = 1.0 - (eta[vessel] / max_eta)
            else:
                urgency = 0.5
                
            # Workload intensity
            if max_dR > 0:
                workload_intensity = total_work / (max_dR * max(len(vessel_piles), 1))
            else:
                workload_intensity = 0.5
                
            # Combine urgency and workload with exponential weighting for urgency
            vessel_weight = math.exp(-2.0 * (1 - urgency)) * 0.6 + workload_intensity * 0.4
            vessel_delays[vessel] = total_work
            
            G.add_node(f'vessel_{vessel}', type=2, weight=min(vessel_weight, 1.0))
    
    # Constraint nodes (type 1) - explicit modeling of all major constraints
    
    # 1. Spatial capacity constraint (cumulative height)
    height_utilization = sum(dR[pile] * mulPileLen // max(hourDiscr, 1) 
                           for pile in range(min(nS, len(dR)))) / max(H, 1)
    spatial_tightness = min(height_utilization * 1.2, 1.0)  # slightly amplified
    G.add_node('spatial_capacity', type=1, weight=spatial_tightness)
    
    # 2. Stack capacity constraint (tonnage)
    total_tonnage = sum(dR[pile] * mulTonnage // max(dS__[pile] if pile < len(dS__) else 1, 1)
                       for pile in range(min(nS, len(dR))))
    stack_tightness = min(total_tonnage / max(stCap * T, 1), 1.0)
    G.add_node('stack_capacity', type=1, weight=stack_tightness)
    
    # 3. Reclaim capacity constraint
    reclaim_tightness = min(sum(dR) / max(reclN * T, 1), 1.0) if dR else 0.5
    G.add_node('reclaim_capacity', type=1, weight=reclaim_tightness)
    
    # 4. Individual vessel constraints (precedence within vessel)
    vessel_pile_groups = {}
    for pile in range(len(whichV)):
        vessel_id = whichV[pile]
        if vessel_id not in vessel_pile_groups:
            vessel_pile_groups[vessel_id] = []
        vessel_pile_groups[vessel_id].append(pile)
    
    for vessel_id, pile_list in vessel_pile_groups.items():
        if len(pile_list) > 1:
            # Precedence constraint for this vessel
            total_duration = sum(dR[pile] for pile in pile_list if pile < len(dR))
            if vessel_id - 1 < len(eta) and max_eta > 0:
                time_pressure = 1.0 - (eta[vessel_id - 1] / max_eta)
            else:
                time_pressure = 0.5
            precedence_weight = time_pressure * 0.6 + (len(pile_list) / max(nS, 1)) * 0.4
            G.add_node(f'precedence_vessel_{vessel_id}', type=1, weight=min(precedence_weight, 1.0))
    
    # 5. Delay constraints (one per vessel)
    for vessel in range(nV):
        if vessel < len(eta):
            # Calculate potential delay risk
            vessel_piles = [i for i in range(len(whichV)) if whichV[i] == vessel + 1]
            total_work = sum(dR[pile] for pile in vessel_piles if pile < len(dR))
            available_time = T - eta[vessel] if vessel < len(eta) else T
            
            if available_time > 0:
                delay_risk = total_work / available_time
                delay_constraint_weight = min(math.sqrt(delay_risk), 1.0)
            else:
                delay_constraint_weight = 1.0
                
            G.add_node(f'delay_vessel_{vessel}', type=1, weight=delay_constraint_weight)
    
    # Add bipartite edges: piles to constraints
    for pile in range(nS):
        pile_node = f'pile_{pile}'
        if pile_node in G.nodes:
            # Connect to spatial capacity (all piles use space)
            if pile < len(dR):
                pile_length = dR[pile] * mulPileLen // max(hourDiscr, 1)
                spatial_weight = pile_length / max(H, 1)
                G.add_edge(pile_node, 'spatial_capacity', weight=min(spatial_weight * 2, 1.0))
            
            # Connect to stack capacity (tonnage usage)
            if pile < len(dR) and pile < len(dS__):
                tonnage = dR[pile] * mulTonnage // max(dS__[pile], 1)
                stack_weight = tonnage / max(stCap, 1)
                G.add_edge(pile_node, 'stack_capacity', weight=min(stack_weight * 3, 1.0))
            
            # Connect to reclaim capacity
            if pile < len(dR):
                reclaim_weight = dR[pile] / max(reclN, 1)
                G.add_edge(pile_node, 'reclaim_capacity', weight=min(reclaim_weight / 1000, 1.0))
            
            # Connect to vessel-specific constraints
            if pile < len(whichV):
                vessel_id = whichV[pile]
                
                # Precedence constraint
                precedence_node = f'precedence_vessel_{vessel_id}'
                if precedence_node in G.nodes:
                    pile_position = [i for i in range(len(whichV)) if whichV[i] == vessel_id].index(pile)
                    precedence_weight = (pile_position + 1) / max(len([i for i in range(len(whichV)) if whichV[i] == vessel_id]), 1)
                    G.add_edge(pile_node, precedence_node, weight=precedence_weight)
                
                # Delay constraint
                delay_node = f'delay_vessel_{vessel_id - 1}'
                if delay_node in G.nodes and pile < len(dR):
                    delay_contribution = dR[pile] / max(vessel_delays.get(vessel_id - 1, 1), 1)
                    G.add_edge(pile_node, delay_node, weight=min(delay_contribution * 2, 1.0))
    
    # Add edges: vessels to their piles
    for pile in range(len(whichV)):
        vessel_id = whichV[pile]
        vessel_node = f'vessel_{vessel_id - 1}'
        pile_node = f'pile_{pile}'
        
        if vessel_node in G.nodes and pile_node in G.nodes and pile < len(dR):
            # Weight by pile's contribution to vessel workload
            pile_contribution = dR[pile] / max(vessel_delays.get(vessel_id - 1, 1), 1)
            G.add_edge(vessel_node, pile_node, weight=pile_contribution)
    
    # Add conflict edges for piles that compete for the same critical resources
    # Only for highly contested resources (top 20% of resource usage)
    if nS > 1:
        pile_resource_usage = []
        for pile in range(min(nS, len(dR))):
            if pile < len(dS__):
                usage = (dR[pile] / max(dS__[pile], 1)) + (dR[pile] * mulPileLen // max(hourDiscr, 1))
                pile_resource_usage.append((pile, usage))
        
        pile_resource_usage.sort(key=lambda x: x[1], reverse=True)
        high_usage_piles = pile_resource_usage[:max(1, nS // 5)]  # top 20%
        
        for i in range(len(high_usage_piles)):
            for j in range(i + 1, len(high_usage_piles)):
                pile1, usage1 = high_usage_piles[i]
                pile2, usage2 = high_usage_piles[j]
                
                pile1_node = f'pile_{pile1}'
                pile2_node = f'pile_{pile2}'
                
                if pile1_node in G.nodes and pile2_node in G.nodes:
                    # Conflict strength based on combined resource usage
                    conflict_strength = (usage1 + usage2) / (2 * max(pile_resource_usage[0][1], 1))
                    if conflict_strength > 0.3:  # Only add significant conflicts
                        G.add_edge(pile1_node, pile2_node, weight=min(conflict_strength, 1.0))
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()