#!/usr/bin/env python3
"""
Graph converter for Car Sequencing problem.
# Converter created with subagent_prompt.md v_02

This problem is about sequencing cars on a production line while respecting
capacity constraints for different options (like air conditioning, sunroof, etc.).

Key challenges: 
- Balancing option demand vs capacity constraints
- Managing sequential constraints across production blocks
- Complex interaction between car classes and option requirements
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the car sequencing problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with:
    - Car class nodes (type 0): decision variables representing different car classes
    - Option constraint nodes (type 1): capacity constraints for each option
    - Sequential constraint nodes (type 1): block-based capacity constraints
    - Weights reflect demand pressure, constraint tightness, and option complexity
    """
    # Access data directly from json_data dict
    n_cars = json_data.get('n_cars', 0)
    n_options = json_data.get('n_options', 0)
    n_classes = json_data.get('n_classes', 0)
    option_max_per_block = json_data.get('option_max_per_block', [])
    option_block_size = json_data.get('option_block_size', [])
    cars_in_class = json_data.get('cars_in_class', [])
    class_option_need = json_data.get('class_option_need', [])
    
    G = nx.Graph()
    
    # Calculate total demand for each option
    option_demand = [0] * n_options
    for class_idx in range(n_classes):
        class_demand = cars_in_class[class_idx] if class_idx < len(cars_in_class) else 0
        for opt_idx in range(n_options):
            # class_option_need is flattened: [class][option] = class * n_options + option
            matrix_idx = class_idx * n_options + opt_idx
            if matrix_idx < len(class_option_need) and class_option_need[matrix_idx] == 1:
                option_demand[opt_idx] += class_demand
    
    total_cars_demand = sum(cars_in_class)
    max_class_demand = max(cars_in_class) if cars_in_class else 1
    max_option_demand = max(option_demand) if option_demand else 1
    
    # Car class nodes (type 0): weighted by relative demand and option complexity
    for class_idx in range(n_classes):
        class_demand = cars_in_class[class_idx] if class_idx < len(cars_in_class) else 0
        
        # Count how many options this class requires
        option_count = 0
        for opt_idx in range(n_options):
            matrix_idx = class_idx * n_options + opt_idx
            if matrix_idx < len(class_option_need) and class_option_need[matrix_idx] == 1:
                option_count += 1
        
        # Weight by demand pressure and option complexity
        demand_pressure = class_demand / max_class_demand if max_class_demand > 0 else 0
        option_complexity = option_count / n_options if n_options > 0 else 0
        
        # Use non-linear combination emphasizing high-demand, high-complexity classes
        weight = math.sqrt(demand_pressure) * (0.6 + 0.4 * option_complexity)
        weight = min(max(weight, 0.1), 1.0)  # Clamp to [0.1, 1.0]
        
        G.add_node(f'class_{class_idx}', type=0, weight=weight)
    
    # Option constraint nodes (type 1): weighted by constraint tightness
    for opt_idx in range(n_options):
        block_size = option_block_size[opt_idx] if opt_idx < len(option_block_size) else 1
        max_per_block = option_max_per_block[opt_idx] if opt_idx < len(option_max_per_block) else 1
        
        # Calculate theoretical maximum capacity vs demand
        n_blocks = max(1, n_cars - block_size + 1)  # Number of overlapping blocks
        theoretical_capacity = n_blocks * max_per_block
        actual_demand = option_demand[opt_idx]
        
        # Constraint tightness: how close are we to violating the constraint?
        if theoretical_capacity > 0:
            tightness = actual_demand / theoretical_capacity
            # Use exponential scaling to emphasize near-violations
            weight = min(1.0, 1.0 - math.exp(-3.0 * tightness))
        else:
            weight = 1.0  # Maximum tightness if no capacity
            
        # Also factor in block size complexity (smaller blocks are harder)
        block_complexity = 1.0 - (block_size / n_cars) if n_cars > 0 else 0.5
        weight = 0.7 * weight + 0.3 * block_complexity
        weight = min(max(weight, 0.2), 1.0)  # Clamp to [0.2, 1.0]
        
        G.add_node(f'option_constraint_{opt_idx}', type=1, weight=weight)
    
    # Sequential block constraint nodes (type 1) - model specific block constraints
    for opt_idx in range(n_options):
        block_size = option_block_size[opt_idx] if opt_idx < len(option_block_size) else 1
        max_per_block = option_max_per_block[opt_idx] if opt_idx < len(option_max_per_block) else 1
        
        if block_size > 1:  # Only create for meaningful block constraints
            # Weight by constraint restrictiveness (smaller ratios are more restrictive)
            restrictiveness = max_per_block / block_size if block_size > 0 else 0.5
            weight = 1.0 - restrictiveness  # More restrictive = higher weight
            weight = max(weight, 0.1)
            
            G.add_node(f'block_constraint_{opt_idx}', type=1, weight=weight)
    
    # Bipartite edges: class to option constraints
    for class_idx in range(n_classes):
        class_demand = cars_in_class[class_idx] if class_idx < len(cars_in_class) else 0
        
        for opt_idx in range(n_options):
            matrix_idx = class_idx * n_options + opt_idx
            
            if matrix_idx < len(class_option_need) and class_option_need[matrix_idx] == 1:
                # This class requires this option
                
                # Edge weight reflects the pressure this class puts on the option
                option_total_demand = option_demand[opt_idx]
                pressure = class_demand / option_total_demand if option_total_demand > 0 else 0.5
                
                # Also factor in overall option scarcity
                option_capacity_ratio = option_max_per_block[opt_idx] / option_block_size[opt_idx] if opt_idx < len(option_max_per_block) and opt_idx < len(option_block_size) and option_block_size[opt_idx] > 0 else 0.5
                scarcity = 1.0 - option_capacity_ratio
                
                # Combine pressure and scarcity with non-linear scaling
                weight = pressure + 0.3 * scarcity
                weight = min(max(weight, 0.1), 1.0)
                
                G.add_edge(f'class_{class_idx}', f'option_constraint_{opt_idx}', weight=weight)
                
                # Also connect to block constraints if they exist
                if opt_idx < len(option_block_size) and option_block_size[opt_idx] > 1:
                    block_weight = pressure * 0.8  # Slightly lower weight for block constraints
                    G.add_edge(f'class_{class_idx}', f'block_constraint_{opt_idx}', weight=block_weight)
    
    # Add conflict edges between classes that compete for highly constrained options
    for opt_idx in range(n_options):
        if option_demand[opt_idx] > 0:
            # Find classes that require this option
            competing_classes = []
            for class_idx in range(n_classes):
                matrix_idx = class_idx * n_options + opt_idx
                if matrix_idx < len(class_option_need) and class_option_need[matrix_idx] == 1:
                    class_demand = cars_in_class[class_idx] if class_idx < len(cars_in_class) else 0
                    competing_classes.append((class_idx, class_demand))
            
            # Calculate constraint pressure for this option
            theoretical_capacity = (n_cars - option_block_size[opt_idx] + 1) * option_max_per_block[opt_idx] if opt_idx < len(option_block_size) and opt_idx < len(option_max_per_block) else n_cars
            pressure_ratio = option_demand[opt_idx] / theoretical_capacity if theoretical_capacity > 0 else 1.0
            
            # Add conflict edges if option is highly constrained and has multiple competing classes
            if pressure_ratio > 0.7 and len(competing_classes) > 1:
                competing_classes.sort(key=lambda x: x[1], reverse=True)  # Sort by demand
                
                # Add conflicts between high-demand classes
                for i in range(min(len(competing_classes), 4)):  # Limit to top 4 to avoid too many edges
                    for j in range(i + 1, min(len(competing_classes), 4)):
                        class1_idx, demand1 = competing_classes[i]
                        class2_idx, demand2 = competing_classes[j]
                        
                        # Conflict strength based on combined demand vs capacity
                        conflict_strength = min(1.0, (demand1 + demand2) / theoretical_capacity * pressure_ratio)
                        
                        if conflict_strength > 0.3:  # Only add meaningful conflicts
                            G.add_edge(f'class_{class1_idx}', f'class_{class2_idx}', weight=conflict_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()