#!/usr/bin/env python3
"""
Graph converter for cc_base (Carpet Cutting) problem.
Created using subagent_prompt.md version: v_02

This problem is about cutting carpet shapes from a roll to minimize waste.
Key challenges: 2D packing with rotations, complex rectilinear shapes, stair cutting constraints.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the carpet cutting problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with:
    - Type 0: Rectangle pieces (room and stair carpet rectangles)
    - Type 1: Constraints (roll width, non-overlap, orientations, stair cutting)
    - Type 2: Resources (roll width, carpet rooms/stairs)
    
    Key relationships:
    - Rectangles participate in non-overlap constraints
    - Room rectangles participate in orientation constraints
    - Stair rectangles participate in cutting constraints
    - All rectangles consume roll width and length
    """
    # Extract data
    roll_wid = json_data.get('roll_wid', 1)
    n_rm = json_data.get('n_rm', 0)
    n_rm_rec = json_data.get('n_rm_rec', 0)
    n_st = json_data.get('n_st', 0)
    
    rm_max_len = json_data.get('rm_max_len', [])
    rm_max_wid = json_data.get('rm_max_wid', [])
    rm_rec_len = json_data.get('rm_rec_len', [])
    rm_rec_wid = json_data.get('rm_rec_wid', [])
    
    st_len = json_data.get('st_len', [])
    st_wid = json_data.get('st_wid', [])
    st_no_steps = json_data.get('st_no_steps', [])
    st_min_steps = json_data.get('st_min_steps', [])
    st_max_breaks = json_data.get('st_max_breaks', [])
    
    G = nx.Graph()
    
    # Calculate some useful metrics
    total_area = sum(rm_rec_len[i] * rm_rec_wid[i] for i in range(min(len(rm_rec_len), len(rm_rec_wid))))
    if st_len and st_wid:
        total_area += sum(st_len[i] * st_wid[i] for i in range(min(len(st_len), len(st_wid))))
    
    max_piece_area = 1
    if rm_rec_len and rm_rec_wid:
        max_piece_area = max(max_piece_area, max(rm_rec_len[i] * rm_rec_wid[i] 
                           for i in range(min(len(rm_rec_len), len(rm_rec_wid)))))
    if st_len and st_wid:
        max_piece_area = max(max_piece_area, max(st_len[i] * st_wid[i] 
                           for i in range(min(len(st_len), len(st_wid)))))
    
    max_dimension = max(1, roll_wid)
    if rm_rec_len:
        max_dimension = max(max_dimension, max(rm_rec_len))
    if rm_rec_wid:
        max_dimension = max(max_dimension, max(rm_rec_wid))
    if st_len:
        max_dimension = max(max_dimension, max(st_len))
    if st_wid:
        max_dimension = max(max_dimension, max(st_wid))
    
    # Type 0 nodes: Rectangle pieces
    # Room carpet rectangles
    for i in range(n_rm_rec):
        if i < len(rm_rec_len) and i < len(rm_rec_wid):
            area = rm_rec_len[i] * rm_rec_wid[i]
            # Weight by relative area and aspect ratio (square pieces are easier)
            aspect_ratio = max(rm_rec_len[i], rm_rec_wid[i]) / max(min(rm_rec_len[i], rm_rec_wid[i]), 1)
            area_weight = area / max_piece_area
            aspect_weight = 1.0 / (1.0 + math.log(aspect_ratio))
            weight = (area_weight + aspect_weight) / 2
            G.add_node(f'rm_rec_{i}', type=0, weight=weight)
    
    # Stair carpet rectangles (computed from stair carpets)
    st_rec_id = 0
    for i in range(n_st):
        if (i < len(st_len) and i < len(st_wid) and i < len(st_no_steps)):
            step_len = st_len[i] // max(st_no_steps[i], 1)
            step_wid = st_wid[i]
            
            for step in range(st_no_steps[i]):
                area = step_len * step_wid
                # Weight by area and position in stair (first/last steps may be more constrained)
                area_weight = area / max_piece_area
                position_weight = 1.0 - abs(step - st_no_steps[i]/2) / (st_no_steps[i]/2 + 1)
                weight = (area_weight + position_weight) / 2
                G.add_node(f'st_rec_{st_rec_id}', type=0, weight=weight)
                st_rec_id += 1
    
    # Type 1 nodes: Constraints
    
    # Roll width constraint (affects all rectangles)
    # Tightness based on how much total width is needed vs available
    total_width_demand = 0
    for i in range(n_rm_rec):
        if i < len(rm_rec_wid):
            total_width_demand += rm_rec_wid[i]
    for i in range(n_st):
        if i < len(st_wid) and i < len(st_no_steps):
            total_width_demand += st_wid[i] * st_no_steps[i]
    
    width_pressure = min(1.0, total_width_demand / (roll_wid * 5))  # Assume we can stack up to 5 high
    G.add_node('roll_width_constraint', type=1, weight=width_pressure)
    
    # Non-overlapping constraint (global constraint affecting all rectangles)
    # Weight by total area vs roll area efficiency
    min_roll_len = max(1, total_area // roll_wid)
    efficiency = total_area / (roll_wid * min_roll_len) if min_roll_len > 0 else 0.5
    overlap_tightness = max(0.0, min(1.0, 1.0 - efficiency))
    G.add_node('non_overlap_constraint', type=1, weight=overlap_tightness)
    
    # Room-specific orientation constraints
    for i in range(n_rm):
        if i < len(rm_max_len) and i < len(rm_max_wid):
            # More constrained if dimensions are very different or close to roll width
            max_dim = max(rm_max_len[i], rm_max_wid[i])
            min_dim = min(rm_max_len[i], rm_max_wid[i])
            aspect_constraint = (max_dim - min_dim) / max_dim if max_dim > 0 else 0
            width_constraint = max_dim / roll_wid if roll_wid > 0 else 0.5
            orientation_tightness = min(1.0, (aspect_constraint + width_constraint) / 2)
            G.add_node(f'room_{i}_orientation', type=1, weight=orientation_tightness)
    
    # Stair cutting constraints
    for i in range(n_st):
        if (i < len(st_no_steps) and i < len(st_min_steps) and i < len(st_max_breaks)):
            # More constrained if min_steps is large relative to total or max_breaks is small
            flexibility = (st_max_breaks[i] + 1) / max(st_no_steps[i], 1)
            min_steps_ratio = st_min_steps[i] / max(st_no_steps[i], 1)
            cutting_tightness = min(1.0, min_steps_ratio + (1.0 - flexibility))
            G.add_node(f'stair_{i}_cutting', type=1, weight=cutting_tightness)
    
    # Type 2 nodes: Resources
    
    # Roll width resource
    G.add_node('roll_width_resource', type=2, weight=1.0)
    
    # Individual room carpets as resources (can be rotated)
    for i in range(n_rm):
        if i < len(rm_max_len) and i < len(rm_max_wid):
            area = rm_max_len[i] * rm_max_wid[i]
            importance = min(1.0, area / max_piece_area) if max_piece_area > 0 else 0.5
            G.add_node(f'room_carpet_{i}', type=2, weight=importance)
    
    # Individual stair carpets as resources (can be cut)
    for i in range(n_st):
        if i < len(st_len) and i < len(st_wid):
            area = st_len[i] * st_wid[i]
            importance = min(1.0, area / max_piece_area) if max_piece_area > 0 else 0.5
            G.add_node(f'stair_carpet_{i}', type=2, weight=importance)
    
    # Edges: Bipartite connections and resource usage
    
    # Room rectangles to constraints
    for i in range(n_rm_rec):
        if i < len(rm_rec_len) and i < len(rm_rec_wid):
            rect_node = f'rm_rec_{i}'
            
            # Connect to roll width constraint (participation weight by width usage)
            width_usage = rm_rec_wid[i] / roll_wid if roll_wid > 0 else 0.5
            G.add_edge(rect_node, 'roll_width_constraint', weight=min(1.0, width_usage * 2))
            
            # Connect to non-overlap constraint (weight by area)
            area_weight = min(1.0, (rm_rec_len[i] * rm_rec_wid[i]) / max_piece_area)
            G.add_edge(rect_node, 'non_overlap_constraint', weight=area_weight)
            
            # Connect to roll width resource
            G.add_edge(rect_node, 'roll_width_resource', weight=min(1.0, width_usage))
    
    # Connect room rectangles to their parent room's orientation constraint
    # Need to map rectangles to rooms (using simple heuristic based on order)
    rm_rec_per_room = n_rm_rec // max(n_rm, 1) if n_rm > 0 else 0
    for i in range(n_rm_rec):
        room_id = i // max(rm_rec_per_room, 1) if rm_rec_per_room > 0 else 0
        room_id = min(room_id, n_rm - 1)
        if room_id < n_rm:
            G.add_edge(f'rm_rec_{i}', f'room_{room_id}_orientation', weight=0.8)
            G.add_edge(f'rm_rec_{i}', f'room_carpet_{room_id}', weight=0.9)
    
    # Stair rectangles to constraints
    st_rec_id = 0
    for i in range(n_st):
        if (i < len(st_len) and i < len(st_wid) and i < len(st_no_steps)):
            step_len = st_len[i] // max(st_no_steps[i], 1)
            step_wid = st_wid[i]
            
            for step in range(st_no_steps[i]):
                rect_node = f'st_rec_{st_rec_id}'
                
                # Connect to roll width constraint
                width_usage = step_wid / roll_wid if roll_wid > 0 else 0.5
                G.add_edge(rect_node, 'roll_width_constraint', weight=min(1.0, width_usage * 2))
                
                # Connect to non-overlap constraint
                area_weight = min(1.0, (step_len * step_wid) / max_piece_area)
                G.add_edge(rect_node, 'non_overlap_constraint', weight=area_weight)
                
                # Connect to stair cutting constraint
                G.add_edge(rect_node, f'stair_{i}_cutting', weight=0.9)
                
                # Connect to stair carpet resource
                G.add_edge(rect_node, f'stair_carpet_{i}', weight=0.8)
                
                # Connect to roll width resource
                G.add_edge(rect_node, 'roll_width_resource', weight=min(1.0, width_usage))
                
                st_rec_id += 1
    
    # Add some conflict edges between large pieces that might compete for space
    all_pieces = []
    for i in range(n_rm_rec):
        if i < len(rm_rec_len) and i < len(rm_rec_wid):
            area = rm_rec_len[i] * rm_rec_wid[i]
            all_pieces.append((f'rm_rec_{i}', area))
    
    st_rec_id = 0
    for i in range(n_st):
        if (i < len(st_len) and i < len(st_wid) and i < len(st_no_steps)):
            step_len = st_len[i] // max(st_no_steps[i], 1)
            step_wid = st_wid[i]
            for step in range(st_no_steps[i]):
                area = step_len * step_wid
                all_pieces.append((f'st_rec_{st_rec_id}', area))
                st_rec_id += 1
    
    # Sort by area and add conflicts between largest pieces
    all_pieces.sort(key=lambda x: x[1], reverse=True)
    top_pieces = all_pieces[:min(len(all_pieces), 8)]  # Top 8 largest pieces
    
    for i in range(len(top_pieces)):
        for j in range(i + 1, len(top_pieces)):
            piece1, area1 = top_pieces[i]
            piece2, area2 = top_pieces[j]
            # Conflict weight based on combined area pressure
            combined_area = area1 + area2
            conflict_weight = min(1.0, combined_area / (roll_wid * max_dimension / 4))
            if conflict_weight > 0.3:  # Only add significant conflicts
                G.add_edge(piece1, piece2, weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()