#!/usr/bin/env python3
"""
Graph converter for Carpet Cutting (cc_base_2) problem.
Converter created with subagent_prompt.md v_02

This problem is about cutting carpet shapes from a roll to minimize wastage.
Key challenges: 2D packing with rotation constraints, non-overlap requirements, 
and stair carpet partitioning with break/step constraints.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the carpet cutting instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with:
    - Room carpets and rectangles as variable nodes (type 0)
    - Stair carpets and steps as variable nodes (type 0) 
    - Geometric constraints as constraint nodes (type 1)
    - Roll capacity as resource node (type 2)
    
    Key relationships:
    - Rectangles participate in non-overlap constraints
    - Roll width/length capacity constraints
    - Stair partitioning constraints
    """
    # Parse input data
    roll_wid = json_data.get('roll_wid', 100)
    max_roll_len = json_data.get('max_roll_len', 1000)
    
    # Room carpets
    n_rm = json_data.get('n_rm', 0)
    n_rm_rec = json_data.get('n_rm_rec', 0)
    rm_max_len = json_data.get('rm_max_len', [])
    rm_max_wid = json_data.get('rm_max_wid', [])
    rm_rec_len = json_data.get('rm_rec_len', [])
    rm_rec_wid = json_data.get('rm_rec_wid', [])
    
    # Stair carpets
    n_st = json_data.get('n_st', 0)
    st_len = json_data.get('st_len', [])
    st_wid = json_data.get('st_wid', [])
    st_no_steps = json_data.get('st_no_steps', [])
    st_min_steps = json_data.get('st_min_steps', [])
    st_max_breaks = json_data.get('st_max_breaks', [])
    
    G = nx.Graph()
    
    # Calculate total area for normalization
    rm_total_area = sum(rm_rec_len[i] * rm_rec_wid[i] for i in range(n_rm_rec))
    st_total_area = sum(st_len[i] * st_wid[i] for i in range(n_st))
    total_area = rm_total_area + st_total_area
    roll_area = roll_wid * max_roll_len
    
    # Room carpet nodes (type 0) - weighted by relative area
    max_rm_area = max((rm_max_len[i] * rm_max_wid[i] for i in range(n_rm)), default=1)
    for i in range(n_rm):
        room_area = rm_max_len[i] * rm_max_wid[i]
        # Weight by area importance and aspect ratio constraint difficulty
        aspect_ratio = max(rm_max_len[i], rm_max_wid[i]) / min(rm_max_len[i], rm_max_wid[i])
        difficulty = (room_area / max_rm_area) * (1.0 + math.log(aspect_ratio))
        G.add_node(f'room_{i}', type=0, weight=min(difficulty, 1.0))
    
    # Room rectangle nodes (type 0) - more granular representation
    max_rec_area = max((rm_rec_len[i] * rm_rec_wid[i] for i in range(n_rm_rec)), default=1)
    for i in range(n_rm_rec):
        rec_area = rm_rec_len[i] * rm_rec_wid[i]
        aspect_ratio = max(rm_rec_len[i], rm_rec_wid[i]) / min(rm_rec_len[i], rm_rec_wid[i])
        # Weight by area and shape complexity
        weight = (rec_area / max_rec_area) * math.sqrt(aspect_ratio)
        G.add_node(f'room_rect_{i}', type=0, weight=min(weight, 1.0))
    
    # Stair carpet nodes (type 0) - weighted by complexity
    for i in range(n_st):
        if i < len(st_len):
            stair_area = st_len[i] * st_wid[i]
            steps = st_no_steps[i] if i < len(st_no_steps) else 1
            max_breaks = st_max_breaks[i] if i < len(st_max_breaks) else 0
            min_steps = st_min_steps[i] if i < len(st_min_steps) else 1
            
            # Weight by partitioning complexity: more breaks and step constraints = harder
            partition_complexity = (max_breaks + 1) / steps  # Break density
            step_constraint_tightness = min_steps / steps if steps > 0 else 0.5
            base_weight = stair_area / (st_total_area + 1)
            
            complexity_weight = base_weight * (1.0 + partition_complexity + step_constraint_tightness)
            G.add_node(f'stair_{i}', type=0, weight=min(complexity_weight, 1.0))
            
            # Individual step nodes for fine-grained modeling
            step_area = (st_len[i] / steps) * st_wid[i] if steps > 0 else 0
            for j in range(steps):
                # Steps at partition boundaries are more constrained
                boundary_factor = 1.0
                if j == 0 or j == steps - 1:  # First/last steps
                    boundary_factor = 1.2
                elif j < min_steps or (steps - j) < min_steps:  # Near min-step boundaries
                    boundary_factor = 1.1
                    
                step_weight = (step_area / (stair_area + 1)) * boundary_factor
                G.add_node(f'stair_{i}_step_{j}', type=0, weight=min(step_weight, 1.0))
    
    # Resource node - roll capacity (type 2)
    utilization = total_area / roll_area if roll_area > 0 else 0.5
    G.add_node('roll_capacity', type=2, weight=min(utilization, 1.0))
    
    # Constraint nodes (type 1)
    
    # Roll width constraint - tightness based on largest width requirement
    max_width_req = max([rm_max_wid[i] for i in range(n_rm)] + 
                       [st_wid[i] for i in range(n_st)], default=1)
    width_tightness = max_width_req / roll_wid if roll_wid > 0 else 0.5
    G.add_node('width_constraint', type=1, weight=min(width_tightness, 1.0))
    
    # Non-overlap constraint - complexity based on rectangle density
    total_rectangles = n_rm_rec + sum(st_no_steps[i] for i in range(n_st))
    if total_rectangles > 0:
        # Non-linear scaling of overlap complexity
        overlap_complexity = 1.0 - math.exp(-0.1 * total_rectangles)
        G.add_node('non_overlap', type=1, weight=overlap_complexity)
    
    # Room orientation constraints
    for i in range(n_rm):
        # Orientation constraint difficulty based on aspect ratio
        if i < len(rm_max_len) and i < len(rm_max_wid):
            aspect_ratio = max(rm_max_len[i], rm_max_wid[i]) / min(rm_max_len[i], rm_max_wid[i])
            # High aspect ratio makes orientation more critical
            orientation_weight = 1.0 - math.exp(-0.5 * (aspect_ratio - 1.0))
            G.add_node(f'room_{i}_orientation', type=1, weight=min(orientation_weight, 1.0))
    
    # Stair partitioning constraints
    for i in range(n_st):
        if i < len(st_no_steps):
            steps = st_no_steps[i]
            max_breaks = st_max_breaks[i] if i < len(st_max_breaks) else 0
            min_steps_val = st_min_steps[i] if i < len(st_min_steps) else 1
            
            # Constraint tightness increases with stricter break/step limits
            if steps > 0:
                break_tightness = 1.0 - (max_breaks / steps)
                step_tightness = min_steps_val / steps
                partition_weight = 0.5 * (break_tightness + step_tightness)
                G.add_node(f'stair_{i}_partition', type=1, weight=min(partition_weight, 1.0))
    
    # Edges: Variable-Constraint participation (bipartite)
    
    # Room carpets to constraints
    for i in range(n_rm):
        # Connect to roll capacity based on area consumption
        if i < len(rm_max_len) and i < len(rm_max_wid):
            area_ratio = (rm_max_len[i] * rm_max_wid[i]) / roll_area if roll_area > 0 else 0.1
            G.add_edge(f'room_{i}', 'roll_capacity', weight=min(area_ratio * 2, 1.0))
            
            # Connect to width constraint based on width requirement
            width_ratio = rm_max_wid[i] / roll_wid if roll_wid > 0 else 0.1
            G.add_edge(f'room_{i}', 'width_constraint', weight=min(width_ratio * 1.5, 1.0))
            
            # Connect to orientation constraint
            G.add_edge(f'room_{i}', f'room_{i}_orientation', weight=1.0)
            
            # Connect to non-overlap constraint
            if total_rectangles > 0:
                participation = 1.0 / math.sqrt(total_rectangles)
                G.add_edge(f'room_{i}', 'non_overlap', weight=participation)
    
    # Room rectangles to non-overlap
    for i in range(n_rm_rec):
        if total_rectangles > 0:
            # Individual rectangle participation in non-overlap
            rec_area = rm_rec_len[i] * rm_rec_wid[i] if i < len(rm_rec_len) and i < len(rm_rec_wid) else 1
            participation = math.sqrt(rec_area / max_rec_area) if max_rec_area > 0 else 0.5
            G.add_edge(f'room_rect_{i}', 'non_overlap', weight=min(participation, 1.0))
    
    # Stairs to constraints
    for i in range(n_st):
        if i < len(st_len) and i < len(st_wid):
            # Connect to roll capacity
            area_ratio = (st_len[i] * st_wid[i]) / roll_area if roll_area > 0 else 0.1
            G.add_edge(f'stair_{i}', 'roll_capacity', weight=min(area_ratio * 2, 1.0))
            
            # Connect to width constraint
            width_ratio = st_wid[i] / roll_wid if roll_wid > 0 else 0.1
            G.add_edge(f'stair_{i}', 'width_constraint', weight=min(width_ratio * 1.5, 1.0))
            
            # Connect to partition constraint
            G.add_edge(f'stair_{i}', f'stair_{i}_partition', weight=1.0)
            
            # Connect steps to partitioning constraint
            steps = st_no_steps[i] if i < len(st_no_steps) else 1
            for j in range(steps):
                step_weight = 1.0 / math.sqrt(steps) if steps > 0 else 1.0
                G.add_edge(f'stair_{i}_step_{j}', f'stair_{i}_partition', weight=step_weight)
                
                # Steps also participate in non-overlap
                if total_rectangles > 0:
                    G.add_edge(f'stair_{i}_step_{j}', 'non_overlap', weight=step_weight)
    
    # Add conflict edges between high-competition rectangles/rooms
    if total_area > 0.8 * roll_area:  # High utilization creates conflicts
        # Room-room conflicts for similar sizes competing for space
        for i in range(n_rm):
            for j in range(i + 1, n_rm):
                if (i < len(rm_max_len) and j < len(rm_max_len) and 
                    i < len(rm_max_wid) and j < len(rm_max_wid)):
                    area_i = rm_max_len[i] * rm_max_wid[i]
                    area_j = rm_max_len[j] * rm_max_wid[j]
                    
                    # Conflict weight based on combined area pressure
                    combined_area = area_i + area_j
                    if combined_area > 0.3 * roll_area:  # Significant combined footprint
                        conflict_weight = min((combined_area / roll_area) - 0.3, 0.7)
                        G.add_edge(f'room_{i}', f'room_{j}', weight=conflict_weight)
        
        # Stair-room conflicts for large items
        for i in range(n_st):
            for j in range(n_rm):
                if (i < len(st_len) and i < len(st_wid) and 
                    j < len(rm_max_len) and j < len(rm_max_wid)):
                    stair_area = st_len[i] * st_wid[i]
                    room_area = rm_max_len[j] * rm_max_wid[j]
                    
                    combined_area = stair_area + room_area
                    if combined_area > 0.4 * roll_area:
                        conflict_weight = min((combined_area / roll_area) - 0.4, 0.6)
                        G.add_edge(f'stair_{i}', f'room_{j}', weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()