#!/usr/bin/env python3
"""
Graph converter for city-position problem.
Created using subagent_prompt.md version: v_02

This problem is about positioning N cities on a 2D grid given partial distance constraints.
Key challenges: Geometric embedding with incomplete data, approximation errors, constraint satisfaction.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the city positioning problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph with:
    - City nodes (type 0) representing position variables
    - Distance constraint nodes (type 1) for each distance requirement
    - Position coordination constraints (type 1) for x/y coupling
    
    What makes instances hard:
    - High distance variance (mix of short/long distances)
    - Dense constraint networks (many distance constraints)
    - Conflicting distance requirements (over-constrained systems)
    """
    N = json_data.get('N', 0)  # number of cities
    R = json_data.get('R', 0)  # number of distance constraints
    from_cities = json_data.get('from', [])
    to_cities = json_data.get('to', [])
    distances = json_data.get('distance', [])
    
    G = nx.Graph()
    
    if N == 0 or R == 0:
        return G
    
    # Calculate distance statistics for weight normalization
    max_distance = max(distances) if distances else 1
    min_distance = min(distances) if distances else 1
    avg_distance = sum(distances) / len(distances) if distances else 1
    
    # Count constraints per city for centrality weighting
    city_constraint_count = [0] * (N + 1)  # 1-indexed
    for i in range(R):
        if i < len(from_cities) and i < len(to_cities):
            city_constraint_count[from_cities[i]] += 1
            city_constraint_count[to_cities[i]] += 1
    
    max_constraints = max(city_constraint_count) if city_constraint_count else 1
    
    # Add city nodes (type 0) - represent position variables
    for city in range(1, N + 1):
        # Weight by constraint involvement (more constrained = higher weight)
        constraint_ratio = city_constraint_count[city] / max_constraints if max_constraints > 0 else 0.5
        # Use sqrt for non-linear scaling
        weight = math.sqrt(constraint_ratio)
        G.add_node(f'city_{city}', type=0, weight=weight)
    
    # Add distance constraint nodes (type 1)
    for i in range(R):
        if i < len(from_cities) and i < len(to_cities) and i < len(distances):
            from_city = from_cities[i]
            to_city = to_cities[i]
            dist = distances[i]
            
            # Weight by constraint tightness - longer distances are generally harder to satisfy
            # Use exponential scaling for distance sensitivity
            distance_factor = 1.0 - math.exp(-3.0 * dist / max_distance)
            
            # Also consider relative distance variance from average
            variance_factor = abs(dist - avg_distance) / max_distance
            
            # Combine factors with non-linear weighting
            weight = 0.7 * distance_factor + 0.3 * variance_factor
            weight = min(max(weight, 0.1), 1.0)  # Clamp to [0.1, 1.0]
            
            constraint_id = f'dist_constraint_{i}'
            G.add_node(constraint_id, type=1, weight=weight)
            
            # Connect cities to their distance constraint
            G.add_edge(f'city_{from_city}', constraint_id, weight=0.8)
            G.add_edge(f'city_{to_city}', constraint_id, weight=0.8)
    
    # Add position coordination constraints (x,y coupling for each city)
    for city in range(1, N + 1):
        # Each city has an internal x-y coordination constraint
        coord_weight = 0.6  # Moderate importance
        coord_id = f'coord_constraint_{city}'
        G.add_node(coord_id, type=1, weight=coord_weight)
        G.add_edge(f'city_{city}', coord_id, weight=0.7)
    
    # Add conflict edges for over-constrained city pairs
    # Cities with many shared constraints may conflict
    for i in range(1, N + 1):
        for j in range(i + 1, N + 1):
            # Count shared distance constraints
            shared_constraints = 0
            for k in range(R):
                if k < len(from_cities) and k < len(to_cities):
                    pair = {from_cities[k], to_cities[k]}
                    if i in pair and j in pair:
                        shared_constraints += 1
            
            # Add conflict edge if cities share multiple constraints
            if shared_constraints >= 2:
                # Weight by number of shared constraints (non-linear)
                conflict_weight = min(shared_constraints / 3.0, 1.0)
                conflict_weight = math.sqrt(conflict_weight)  # Non-linear scaling
                G.add_edge(f'city_{i}', f'city_{j}', weight=conflict_weight)
    
    # Add global complexity node for overall problem difficulty
    if N > 4:  # Only for non-trivial instances
        complexity_ratio = R / (N * (N - 1) / 2)  # Constraint density
        complexity_weight = min(math.sqrt(complexity_ratio), 1.0)
        G.add_node('global_complexity', type=1, weight=complexity_weight)
        
        # Connect to most constrained cities
        for city in range(1, N + 1):
            if city_constraint_count[city] > avg_distance / max_distance * max_constraints:
                connection_weight = city_constraint_count[city] / max_constraints
                G.add_edge(f'city_{city}', 'global_complexity', weight=connection_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()