#!/usr/bin/env python3
"""
Graph converter for community-detection problem.
Created using subagent_prompt.md version: v_02

This problem is about partitioning a graph into communities to maximize modularity 
while satisfying must-link and cannot-link constraints.
Key challenges: Balancing modularity optimization with hard constraints, managing
community size constraints, handling sparse vs dense networks.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the community detection problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph with vertices as variables (type 0),
    constraint nodes for must-link/cannot-link constraints (type 1), and
    community nodes as resources (type 2). Weight vertices by their centrality
    in the original network and constraints by their restrictiveness.
    
    - Vertices are weighted by normalized degree (centrality)
    - Must-link constraints weighted by strength of connection in original graph
    - Cannot-link constraints weighted by inverse of modularity penalty
    - Community nodes weighted by capacity utilization
    """
    # Extract data
    n = json_data.get('n', 0)
    k = json_data.get('k', 1)
    nML = json_data.get('nML', 0)
    nCL = json_data.get('nCL', 0)
    deg = json_data.get('deg', [])
    ML = json_data.get('ML', [])
    CL = json_data.get('CL', [])
    A = json_data.get('A', [])
    W = json_data.get('W', [])
    
    if n == 0:
        return nx.Graph()
    
    G = nx.Graph()
    
    # Convert flattened adjacency and weight matrices to 2D
    A_matrix = [[A[i * n + j] if i * n + j < len(A) else 0 for j in range(n)] for i in range(n)]
    W_matrix = [[W[i * n + j] if i * n + j < len(W) else 0 for j in range(n)] for i in range(n)]
    
    # Add vertex nodes (type 0) - weighted by normalized degree (centrality)
    max_deg = max(deg) if deg else 1
    for i in range(n):
        vertex_deg = deg[i] if i < len(deg) else 0
        # Higher degree vertices are more central/important
        centrality = vertex_deg / max_deg if max_deg > 0 else 0.5
        G.add_node(f'vertex_{i}', type=0, weight=centrality)
    
    # Add community resource nodes (type 2) - weighted by expected utilization
    avg_community_size = n / k if k > 0 else n
    max_size = json_data.get('maxsize', n)
    for c in range(k):
        # Weight by capacity utilization expectation
        utilization = min(avg_community_size / max_size, 1.0) if max_size > 0 else 0.5
        G.add_node(f'community_{c}', type=2, weight=utilization)
    
    # Add must-link constraint nodes (type 1)
    for m in range(0, len(ML), 2):  # ML is flattened pairs
        if m + 1 < len(ML):
            v1, v2 = ML[m], ML[m + 1]
            # Convert to 0-based indexing
            v1 -= 1
            v2 -= 1
            if 0 <= v1 < n and 0 <= v2 < n:
                # Weight by strength of connection in original graph
                original_weight = W_matrix[v1][v2] if v1 < len(W_matrix) and v2 < len(W_matrix[v1]) else 0
                # Normalize to positive weight - higher positive weights indicate stronger must-link
                max_weight = max(max(row) for row in W_matrix) if W_matrix else 1
                min_weight = min(min(row) for row in W_matrix) if W_matrix else -1
                weight_range = max_weight - min_weight if max_weight > min_weight else 1
                normalized_weight = (original_weight - min_weight) / weight_range if weight_range > 0 else 0.5
                
                constraint_id = f'mustlink_{m//2}'
                G.add_node(constraint_id, type=1, weight=normalized_weight)
                
                # Connect constraint to both vertices
                G.add_edge(f'vertex_{v1}', constraint_id, weight=0.9)
                G.add_edge(f'vertex_{v2}', constraint_id, weight=0.9)
    
    # Add cannot-link constraint nodes (type 1)
    for c in range(0, len(CL), 2):  # CL is flattened pairs
        if c + 1 < len(CL):
            v1, v2 = CL[c], CL[c + 1]
            # Convert to 0-based indexing
            v1 -= 1
            v2 -= 1
            if 0 <= v1 < n and 0 <= v2 < n:
                # Weight by inverse modularity penalty - stronger penalty = more important constraint
                original_weight = W_matrix[v1][v2] if v1 < len(W_matrix) and v2 < len(W_matrix[v1]) else 0
                # For cannot-link, negative weights indicate penalty for being together
                # Convert to constraint importance using exponential decay
                penalty_strength = abs(original_weight) if original_weight < 0 else 1
                max_penalty = max(abs(min(row)) for row in W_matrix) if W_matrix else 1
                constraint_weight = math.exp(-penalty_strength / max_penalty) if max_penalty > 0 else 0.5
                
                constraint_id = f'cannotlink_{c//2}'
                G.add_node(constraint_id, type=1, weight=constraint_weight)
                
                # Connect constraint to both vertices
                G.add_edge(f'vertex_{v1}', constraint_id, weight=0.8)
                G.add_edge(f'vertex_{v2}', constraint_id, weight=0.8)
    
    # Add edges from vertices to potential communities (modeling assignment decisions)
    for i in range(n):
        vertex_deg = deg[i] if i < len(deg) else 0
        for c in range(k):
            # Edge weight reflects assignment preference based on vertex importance
            assignment_strength = (vertex_deg + 1) / (max_deg + 1) if max_deg > 0 else 0.5
            G.add_edge(f'vertex_{i}', f'community_{c}', weight=assignment_strength)
    
    # Add structural edges between vertices based on original graph adjacency
    # This captures the underlying network structure
    for i in range(n):
        for j in range(i + 1, n):
            if A_matrix[i][j] > 0:  # There's an edge in original graph
                # Weight by normalized modularity contribution
                weight_val = W_matrix[i][j] if i < len(W_matrix) and j < len(W_matrix[i]) else 0
                if weight_val > 0:  # Positive weights encourage same community
                    max_pos_weight = max(max(max(row), 0) for row in W_matrix) if W_matrix else 1
                    edge_weight = weight_val / max_pos_weight if max_pos_weight > 0 else 0.5
                    G.add_edge(f'vertex_{i}', f'vertex_{j}', weight=edge_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()