#!/usr/bin/env python3
"""
Graph converter for concert-hall-cap problem.
Converter created with subagent_prompt.md v_02

This problem is about assigning concert offers to halls with capacity constraints.
Each offer has a time interval, price, and audience requirement.
Each hall has a fixed capacity.
Key challenges: overlapping time conflicts and capacity matching.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the concert hall scheduling problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Bipartite model with explicit constraint nodes
    - Offers as type 0 nodes (decision variables)
    - Halls as type 2 nodes (resources) 
    - Time-overlap constraints as type 1 nodes
    - Capacity constraints as type 1 nodes
    - Edge weights reflect conflict intensity and resource utilization
    """
    num_offers = json_data.get('num_offers', 0)
    num_halls = json_data.get('num_halls', 0)
    start = json_data.get('start', [])
    end = json_data.get('end', [])
    price = json_data.get('price', [])
    capacity = json_data.get('capacity', [])
    requirement = json_data.get('requirement', [])
    
    G = nx.Graph()
    
    # Add offer nodes (type 0 - variables)
    max_price = max(price) if price else 1
    max_duration = max(end[i] - start[i] for i in range(num_offers)) if num_offers > 0 else 1
    
    for i in range(num_offers):
        offer_start = start[i] if i < len(start) else 0
        offer_end = end[i] if i < len(end) else offer_start + 1
        offer_price = price[i] if i < len(price) else 0
        duration = offer_end - offer_start
        
        # Weight by value density (price per time unit) - normalized
        value_density = offer_price / max(duration, 1) / max_price if max_price > 0 else 0.5
        # Add urgency factor based on duration (shorter = more constrained)
        urgency = math.exp(-2.0 * duration / max_duration) if max_duration > 0 else 0.5
        
        weight = min(0.7 * value_density + 0.3 * urgency, 1.0)
        G.add_node(f'offer_{i}', type=0, weight=max(weight, 0.1))
    
    # Add hall nodes (type 2 - resources)
    max_capacity = max(capacity) if capacity else 1
    
    for h in range(num_halls):
        hall_capacity = capacity[h] if h < len(capacity) else 0
        # Weight by relative scarcity (smaller halls are scarcer)
        scarcity = 1.0 - (hall_capacity / max_capacity) if max_capacity > 0 else 0.5
        # Add utilization pressure based on how many offers can fit
        feasible_offers = sum(1 for i in range(num_offers) 
                            if i < len(requirement) and requirement[i] <= hall_capacity)
        utilization_pressure = min(feasible_offers / max(num_offers, 1), 1.0)
        
        weight = 0.6 * scarcity + 0.4 * utilization_pressure
        G.add_node(f'hall_{h}', type=2, weight=max(weight, 0.1))
    
    # Create time overlap constraint nodes (type 1)
    # Group offers by overlapping time intervals
    time_groups = []
    processed = set()
    
    for i in range(num_offers):
        if i in processed:
            continue
            
        offer_start = start[i] if i < len(start) else 0
        offer_end = end[i] if i < len(end) else offer_start + 1
        
        # Find all offers that overlap with offer i
        overlapping = {i}
        for j in range(num_offers):
            if j != i and j not in processed:
                other_start = start[j] if j < len(start) else 0
                other_end = end[j] if j < len(end) else other_start + 1
                
                # Check for overlap
                if max(offer_start, other_start) < min(offer_end, other_end):
                    overlapping.add(j)
        
        if len(overlapping) > 1:  # Only create constraint for actual conflicts
            time_groups.append(overlapping)
            processed.update(overlapping)
    
    # Add time constraint nodes
    for idx, group in enumerate(time_groups):
        # Weight by conflict intensity (more offers = tighter constraint)
        conflict_intensity = min(len(group) / max(num_offers, 1) * 3, 1.0)
        G.add_node(f'time_conflict_{idx}', type=1, weight=conflict_intensity)
        
        # Connect all offers in this time conflict
        for offer_id in group:
            G.add_edge(f'offer_{offer_id}', f'time_conflict_{idx}', weight=1.0)
    
    # Create capacity constraint nodes (type 1) for each hall-requirement combination
    capacity_constraints = []
    for h in range(num_halls):
        hall_capacity = capacity[h] if h < len(capacity) else 1
        
        # Find offers that can fit in this hall
        compatible_offers = []
        total_demand = 0
        for i in range(num_offers):
            req = requirement[i] if i < len(requirement) else 0
            if req <= hall_capacity:
                compatible_offers.append(i)
                total_demand += req
        
        if compatible_offers:
            # Calculate constraint tightness
            tightness = min(total_demand / hall_capacity, 1.0) if hall_capacity > 0 else 1.0
            tightness = max(tightness, 0.1)
            
            constraint_name = f'capacity_{h}'
            G.add_node(constraint_name, type=1, weight=tightness)
            capacity_constraints.append((constraint_name, compatible_offers, hall_capacity))
            
            # Connect compatible offers to this capacity constraint
            for offer_id in compatible_offers:
                req = requirement[offer_id] if offer_id < len(requirement) else 0
                utilization = req / hall_capacity if hall_capacity > 0 else 0.5
                G.add_edge(f'offer_{offer_id}', constraint_name, weight=min(utilization * 2, 1.0))
    
    # Add hall-offer feasibility edges (resource allocation)
    for h in range(num_halls):
        hall_capacity = capacity[h] if h < len(capacity) else 1
        for i in range(num_offers):
            req = requirement[i] if i < len(requirement) else 0
            if req <= hall_capacity:  # Feasible assignment
                utilization = req / hall_capacity if hall_capacity > 0 else 0.5
                # Use exponential scaling to emphasize high utilization
                weight = min(math.exp(utilization * 2) / math.exp(2), 1.0)
                G.add_edge(f'offer_{i}', f'hall_{h}', weight=weight)
    
    # Add conflict edges between offers competing for oversubscribed halls
    for h in range(num_halls):
        hall_capacity = capacity[h] if h < len(capacity) else 1
        competing_offers = []
        
        for i in range(num_offers):
            req = requirement[i] if i < len(requirement) else 0
            if req <= hall_capacity:
                competing_offers.append((i, req))
        
        # If total demand exceeds capacity significantly, add conflicts
        total_demand = sum(req for _, req in competing_offers)
        if total_demand > hall_capacity * 1.5 and len(competing_offers) > 1:
            # Sort by requirement (descending) to prioritize large consumers
            competing_offers.sort(key=lambda x: x[1], reverse=True)
            
            # Add conflict edges between top competitors
            for idx1 in range(min(len(competing_offers), 4)):
                for idx2 in range(idx1 + 1, min(len(competing_offers), 4)):
                    i1, req1 = competing_offers[idx1]
                    i2, req2 = competing_offers[idx2]
                    
                    if req1 + req2 > hall_capacity:  # Cannot both fit
                        conflict_strength = (req1 + req2) / (hall_capacity * 2)
                        G.add_edge(f'offer_{i1}', f'offer_{i2}', 
                                 weight=min(conflict_strength, 1.0))
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()