#!/usr/bin/env python3
"""
Graph converter for Cable Tree Wiring (CTW) problem.
Created using subagent_prompt.md version: v_02

This problem is about optimally routing cables through cavities in a wiring system.
Key challenges: Complex constraint interactions, disjunctive constraints, cable adjacency requirements, and multi-objective optimization.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the CTW problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with positions as decision variables and constraints as explicit nodes.
    - Position nodes (type 0): Decision points for cable placement
    - Constraint nodes (type 1): Various constraint types with weights reflecting tightness
    - Cable nodes (type 2): Cable resources with weights based on adjacency requirements
    
    The graph captures constraint interactions and cable conflicts that determine solving difficulty.
    """
    k = json_data.get('k', 0)  # Total positions
    b = json_data.get('b', 0)  # Number of cable starts
    
    atomic_constraints = json_data.get('AtomicConstraints', [])
    disjunctive_constraints = json_data.get('DisjunctiveConstraints', [])
    direct_successors = json_data.get('DirectSuccessors', [])
    soft_atomic_constraints = json_data.get('SoftAtomicConstraints', [])
    
    G = nx.Graph()
    
    # Position nodes (type 0) - decision variables for cavity assignments
    # Weight by centrality and constraint density
    for pos in range(1, k + 1):
        # Count constraints involving this position
        constraint_count = 0
        
        # Count atomic constraints
        for i in range(0, len(atomic_constraints), 2):
            if i + 1 < len(atomic_constraints):
                if atomic_constraints[i] == pos or atomic_constraints[i + 1] == pos:
                    constraint_count += 1
        
        # Count disjunctive constraints
        for i in range(0, len(disjunctive_constraints), 4):
            if i + 3 < len(disjunctive_constraints):
                if (disjunctive_constraints[i] == pos or disjunctive_constraints[i + 1] == pos or
                    disjunctive_constraints[i + 2] == pos or disjunctive_constraints[i + 3] == pos):
                    constraint_count += 1
        
        # Count soft constraints
        for i in range(0, len(soft_atomic_constraints), 2):
            if i + 1 < len(soft_atomic_constraints):
                if soft_atomic_constraints[i] == pos or soft_atomic_constraints[i + 1] == pos:
                    constraint_count += 1
        
        # Normalize constraint density and apply non-linear scaling
        max_constraints = max(k * 2, 1)  # Rough upper bound
        density = constraint_count / max_constraints
        weight = math.sqrt(density)  # Non-linear scaling for better sensitivity
        
        G.add_node(f'pos_{pos}', type=0, weight=min(weight, 1.0))
    
    # Cable nodes (type 2) - represent cable resources and adjacency requirements
    for cable in range(1, b + 1):
        # Check if cable has direct successor constraint
        has_direct_successor = cable in direct_successors or (cable + b) in direct_successors
        
        # Cables with direct successor constraints are more critical
        weight = 0.8 if has_direct_successor else 0.4
        
        G.add_node(f'cable_{cable}', type=2, weight=weight)
        G.add_node(f'cable_{cable + b}', type=2, weight=weight)
    
    # Atomic constraint nodes (type 1) - hard precedence constraints
    constraint_id = 0
    for i in range(0, len(atomic_constraints), 2):
        if i + 1 < len(atomic_constraints):
            pos1, pos2 = atomic_constraints[i], atomic_constraints[i + 1]
            
            # Weight by constraint scope and problem scale
            scope_weight = 2.0 / k  # Normalized by problem size
            weight = min(0.7 + scope_weight, 1.0)
            
            G.add_node(f'atomic_{constraint_id}', type=1, weight=weight)
            
            # Connect to involved positions
            if 1 <= pos1 <= k:
                G.add_edge(f'pos_{pos1}', f'atomic_{constraint_id}', weight=0.8)
            if 1 <= pos2 <= k:
                G.add_edge(f'pos_{pos2}', f'atomic_{constraint_id}', weight=0.8)
            
            constraint_id += 1
    
    # Disjunctive constraint nodes (type 1) - either-or precedence constraints
    disjunctive_id = 0
    for i in range(0, len(disjunctive_constraints), 4):
        if i + 3 < len(disjunctive_constraints):
            pos1, pos2, pos3, pos4 = (disjunctive_constraints[i], disjunctive_constraints[i + 1],
                                    disjunctive_constraints[i + 2], disjunctive_constraints[i + 3])
            
            # Disjunctive constraints are complex - higher weight
            weight = 0.9
            
            G.add_node(f'disjunctive_{disjunctive_id}', type=1, weight=weight)
            
            # Connect to all involved positions
            if 1 <= pos1 <= k:
                G.add_edge(f'pos_{pos1}', f'disjunctive_{disjunctive_id}', weight=0.9)
            if 1 <= pos2 <= k:
                G.add_edge(f'pos_{pos2}', f'disjunctive_{disjunctive_id}', weight=0.9)
            if 1 <= pos3 <= k:
                G.add_edge(f'pos_{pos3}', f'disjunctive_{disjunctive_id}', weight=0.7)
            if 1 <= pos4 <= k:
                G.add_edge(f'pos_{pos4}', f'disjunctive_{disjunctive_id}', weight=0.7)
            
            disjunctive_id += 1
    
    # Direct successor constraint nodes (type 1) - adjacency requirements
    successor_id = 0
    for cable_pos in direct_successors:
        # Each cable in DirectSuccessors has an adjacency constraint
        weight = 0.8  # Adjacency constraints are moderately complex
        
        G.add_node(f'successor_{successor_id}', type=1, weight=weight)
        
        # Connect to the cable position
        if 1 <= cable_pos <= k:
            G.add_edge(f'pos_{cable_pos}', f'successor_{successor_id}', weight=0.8)
        
        # Connect to cable node and its pair
        if cable_pos <= b:
            G.add_edge(f'cable_{cable_pos}', f'successor_{successor_id}', weight=0.9)
            # Also connect to the paired position
            if cable_pos + b <= k:
                G.add_edge(f'pos_{cable_pos + b}', f'successor_{successor_id}', weight=0.6)
                G.add_edge(f'cable_{cable_pos + b}', f'successor_{successor_id}', weight=0.7)
        else:
            cable_id = cable_pos - b
            if cable_id >= 1:
                G.add_edge(f'cable_{cable_id}', f'successor_{successor_id}', weight=0.9)
                # Also connect to the paired position
                if cable_pos - b >= 1:
                    G.add_edge(f'pos_{cable_pos - b}', f'successor_{successor_id}', weight=0.6)
        
        successor_id += 1
    
    # Ensure all cable nodes are connected to the constraint graph
    # Add cable coordination constraint nodes to connect isolated cables
    for cable in range(1, b + 1):
        coord_weight = 0.6
        G.add_node(f'cable_coord_{cable}', type=1, weight=coord_weight)
        
        # Connect cable pair
        G.add_edge(f'cable_{cable}', f'cable_coord_{cable}', weight=0.7)
        G.add_edge(f'cable_{cable + b}', f'cable_coord_{cable}', weight=0.7)
        
        # Connect to positions if they exist
        if cable <= k:
            G.add_edge(f'pos_{cable}', f'cable_coord_{cable}', weight=0.5)
        if cable + b <= k:
            G.add_edge(f'pos_{cable + b}', f'cable_coord_{cable}', weight=0.5)
    
    # Soft atomic constraint nodes (type 1) - soft precedence constraints
    soft_id = 0
    for i in range(0, len(soft_atomic_constraints), 2):
        if i + 1 < len(soft_atomic_constraints):
            pos1, pos2 = soft_atomic_constraints[i], soft_atomic_constraints[i + 1]
            
            # Soft constraints have lower weight since violations are allowed
            weight = 0.5
            
            G.add_node(f'soft_{soft_id}', type=1, weight=weight)
            
            # Connect to involved positions with lower edge weights
            if 1 <= pos1 <= k:
                G.add_edge(f'pos_{pos1}', f'soft_{soft_id}', weight=0.5)
            if 1 <= pos2 <= k:
                G.add_edge(f'pos_{pos2}', f'soft_{soft_id}', weight=0.5)
            
            soft_id += 1
    
    # Add conflict edges between cables that compete for similar positions
    # This models the implicit conflicts that arise from cable routing constraints
    if b > 1:
        for cable1 in range(1, min(b + 1, 10)):  # Limit to avoid too many edges
            for cable2 in range(cable1 + 1, min(b + 1, 10)):
                # Cables with overlapping position ranges create conflicts
                # Add exponential decay based on cable distance
                distance = abs(cable2 - cable1)
                conflict_weight = math.exp(-2.0 * distance / b)
                
                if conflict_weight > 0.1:  # Only add significant conflicts
                    G.add_edge(f'cable_{cable1}', f'cable_{cable2}', weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()