#!/usr/bin/env python3
"""
Graph converter for cutstock problem.
Created using subagent_prompt.md version: v_02

This problem is about cutting items of different lengths from stock pieces
of fixed length to minimize waste while satisfying demand.
Key challenges: Bin packing complexity, item combination optimization,
material waste minimization.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the cutstock problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with items and constraints
    - Items (type 0): Different item types to be cut
    - Demand constraints (type 1): Ensure each item demand is met
    - Capacity constraints (type 1): Ensure stock piece capacity isn't exceeded
    - Conflicts modeled via shared capacity constraints
    """
    # Extract problem data
    N = json_data.get('N', 0)  # Number of item types
    L = json_data.get('L', 1)  # Stock length
    i_length = json_data.get('i_length', [])
    i_demand = json_data.get('i_demand', [])
    
    # Calculate upper bound K (number of stock pieces)
    K = 0
    for i in range(N):
        if i < len(i_length) and i < len(i_demand):
            pieces_per_stock = L // i_length[i] if i_length[i] > 0 else 1
            pieces_needed = (i_demand[i] + pieces_per_stock - 1) // pieces_per_stock  # Ceiling division
            K += pieces_needed
    K = max(K, 1)  # Ensure at least 1
    
    G = nx.Graph()
    
    # Node 1: Item type nodes (type 0)
    max_demand = max(i_demand) if i_demand else 1
    max_length = max(i_length) if i_length else 1
    total_demand = sum(i_demand)
    
    for i in range(N):
        if i < len(i_length) and i < len(i_demand):
            length = i_length[i]
            demand = i_demand[i]
            
            # Weight by value density: high demand + large size = more critical
            length_factor = length / max_length
            demand_factor = demand / max_demand
            # Non-linear combination emphasizing both factors
            criticality = math.sqrt(length_factor * demand_factor)
            
            G.add_node(f'item_{i}', type=0, weight=criticality)
    
    # Node 2: Demand constraint nodes (type 1) - one per item type
    for i in range(N):
        if i < len(i_demand):
            demand = i_demand[i]
            # Weight by demand pressure
            demand_weight = demand / max_demand if max_demand > 0 else 0.5
            G.add_node(f'demand_constraint_{i}', type=1, weight=demand_weight)
    
    # Node 3: Stock piece capacity constraints (type 1)
    # Create constraints for potential stock piece configurations
    # Weight by expected utilization difficulty
    
    # Calculate efficiency/waste for different item combinations
    total_volume = sum(i_length[i] * i_demand[i] for i in range(min(N, len(i_length), len(i_demand))))
    theoretical_stocks = total_volume / L if L > 0 else 1
    
    for k in range(min(K, 20)):  # Limit to 20 stock constraints to avoid explosion
        # Weight based on expected utilization
        # Later stocks are harder to fill efficiently
        utilization_difficulty = 1.0 - math.exp(-2.0 * k / max(K, 1))
        G.add_node(f'stock_{k}', type=1, weight=utilization_difficulty)
    
    # Edges: Items to demand constraints (bipartite)
    for i in range(N):
        if i < len(i_demand):
            # Strong connection - each item must satisfy its demand
            G.add_edge(f'item_{i}', f'demand_constraint_{i}', weight=1.0)
    
    # Edges: Items to stock capacity constraints
    # Model potential cutting patterns
    stock_count = min(K, 20)
    for i in range(N):
        if i < len(i_length):
            length = i_length[i]
            space_ratio = length / L if L > 0 else 0.5
            
            # Connect to multiple stocks with decreasing probability/weight
            for k in range(stock_count):
                # Weight represents utilization of stock capacity
                # Exponential decay - first few stocks are more likely to be used
                usage_probability = math.exp(-0.5 * k) * space_ratio
                if usage_probability > 0.1:  # Only add meaningful connections
                    G.add_edge(f'item_{i}', f'stock_{k}', weight=usage_probability)
    
    # Add conflict edges between items that compete for space
    # Items with large sizes create more conflicts
    for i in range(N):
        for j in range(i + 1, N):
            if (i < len(i_length) and j < len(i_length) and 
                i < len(i_demand) and j < len(i_demand)):
                
                combined_length = i_length[i] + i_length[j]
                # If items are too big to fit together, they conflict
                if combined_length > L:
                    # Weight by how much they exceed capacity
                    conflict_strength = min((combined_length - L) / L, 1.0)
                    # Also consider their demands - more demanded items have stronger conflicts
                    demand_factor = (i_demand[i] + i_demand[j]) / (2 * max_demand) if max_demand > 0 else 0.5
                    final_weight = conflict_strength * demand_factor
                    
                    if final_weight > 0.1:  # Lower threshold for better connectivity
                        G.add_edge(f'item_{i}', f'item_{j}', weight=final_weight)
                else:
                    # Items that can fit together also have relationships (weaker)
                    # This creates better connectivity
                    space_efficiency = combined_length / L if L > 0 else 0.5
                    cooperation_weight = space_efficiency * 0.3  # Weaker cooperation edge
                    if cooperation_weight > 0.1:
                        G.add_edge(f'item_{i}', f'item_{j}', weight=cooperation_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()