#!/usr/bin/env python3
"""
Graph converter for DCMST (Diameter Constrained Minimum Spanning Tree) problem.
# Converter created with subagent_prompt.md v_02

This problem is about finding a minimum weight spanning tree with diameter constraint.
Key challenges: Balancing tree weight minimization with diameter constraint satisfaction.
The diameter constraint adds structural complexity that standard MST algorithms don't handle.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the DCMST problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph with nodes and structural constraints
    - Node entities: vertices in the original graph (type 0)
    - Constraint entities: diameter constraint and edge selection constraints (type 1)
    - Edge entities: potential edges with their costs (type 2)
    - Model the tension between minimizing cost and satisfying diameter constraint
    """
    # Access data from json_data dict
    nbV = json_data.get('nbV', 0)
    nbE = json_data.get('nbE', 0)
    diameter = json_data.get('diameter', 1)
    radius = json_data.get('radius', 1)
    ws = json_data.get('ws', [])
    en_flat = json_data.get('en', [])
    
    # Parse edge endpoints from flat array
    edges = []
    for i in range(0, len(en_flat), 2):
        if i + 1 < len(en_flat):
            edges.append((en_flat[i], en_flat[i+1]))
    
    G = nx.Graph()
    
    # Vertex nodes (type 0) - decision variables in the tree
    # Weight by centrality potential and radius constraint pressure
    for v in range(1, nbV + 1):
        # Central vertices are more constrained by diameter requirement
        # Weight higher for vertices that could serve as good centers
        max_distance_from_center = max(abs(v - (nbV + 1) // 2), 1)
        centrality = 1.0 / max_distance_from_center
        # Apply non-linear scaling based on radius constraint
        radius_pressure = math.exp(-2.0 * max_distance_from_center / radius)
        weight = 0.3 + 0.7 * (centrality + radius_pressure) / 2
        G.add_node(f'vertex_{v}', type=0, weight=min(weight, 1.0))
    
    # Edge resource nodes (type 2) - potential edges with their costs
    # Weight by cost efficiency and structural importance
    if ws:
        max_weight = max(ws)
        min_weight = min(ws)
        weight_range = max(max_weight - min_weight, 1)
    
    for i, (u, v) in enumerate(edges):
        if i < len(ws):
            edge_weight = ws[i]
            # Invert weight - lower cost edges are more valuable
            cost_efficiency = 1.0 - (edge_weight - min_weight) / weight_range
            # Add structural importance based on potential to reduce diameter
            node_distance = abs(u - v)
            diameter_contribution = math.exp(-node_distance / (nbV // 2)) if nbV > 2 else 0.5
            weight = 0.4 * cost_efficiency + 0.6 * diameter_contribution
            G.add_node(f'edge_{i}', type=2, weight=weight)
    
    # Constraint nodes (type 1)
    
    # Diameter constraint - very high weight as it's the key constraint
    G.add_node('diameter_constraint', type=1, weight=1.0)
    
    # Tree structure constraint (exactly nbV-1 edges must be selected)
    tree_tightness = min(nbE / ((nbV - 1) * 2), 1.0)  # How constrained the tree selection is
    G.add_node('tree_structure_constraint', type=1, weight=0.8 + 0.2 * tree_tightness)
    
    # Connectivity constraints for each vertex (ensure each vertex is reachable)
    avg_degree_pressure = nbE / nbV / nbV if nbV > 0 else 0
    for v in range(1, nbV + 1):
        # Higher weight for vertices with fewer potential connections
        vertex_degree = sum(1 for u, w in edges if u == v or w == v)
        scarcity = 1.0 - min(vertex_degree / nbV, 1.0) if nbV > 0 else 0.5
        constraint_weight = 0.5 + 0.3 * scarcity + 0.2 * avg_degree_pressure
        G.add_node(f'connectivity_{v}', type=1, weight=constraint_weight)
    
    # Add bipartite edges: vertices to constraints they participate in
    
    # All vertices participate in diameter constraint
    for v in range(1, nbV + 1):
        # Weight by how much this vertex affects diameter (central vertices have higher impact)
        centrality_impact = 1.0 / max(abs(v - (nbV + 1) // 2), 1)
        diameter_impact = math.sqrt(centrality_impact)  # Non-linear relationship
        G.add_edge(f'vertex_{v}', 'diameter_constraint', weight=diameter_impact)
    
    # Vertices to their connectivity constraints
    for v in range(1, nbV + 1):
        G.add_edge(f'vertex_{v}', f'connectivity_{v}', weight=1.0)
    
    # Edges to tree structure constraint (each edge can be selected for the tree)
    for i in range(len(edges)):
        if i < len(ws):
            # Weight by importance to tree formation
            edge_weight = ws[i]
            importance = 1.0 - (edge_weight - min_weight) / weight_range if weight_range > 0 else 0.5
            G.add_edge(f'edge_{i}', 'tree_structure_constraint', weight=0.3 + 0.7 * importance)
    
    # Edges to vertex connectivity constraints
    for i, (u, v) in enumerate(edges):
        # Each edge contributes to connectivity of its endpoints
        if i < len(ws):
            edge_cost = ws[i]
            # Better (cheaper) edges are more likely to be used for connectivity
            utility = 1.0 - (edge_cost - min_weight) / weight_range if weight_range > 0 else 0.5
            G.add_edge(f'edge_{i}', f'connectivity_{u}', weight=0.4 + 0.6 * utility)
            G.add_edge(f'edge_{i}', f'connectivity_{v}', weight=0.4 + 0.6 * utility)
    
    # Edges to diameter constraint (edges that could violate diameter get higher weights)
    for i, (u, v) in enumerate(edges):
        if i < len(ws):
            # Long edges (between distant nodes) are more likely to contribute to diameter violations
            node_distance = abs(u - v)
            diameter_risk = node_distance / max(nbV - 1, 1) if nbV > 1 else 0
            # Use exponential to emphasize high-risk edges
            risk_weight = math.exp(2.0 * diameter_risk) / math.exp(2.0)
            G.add_edge(f'edge_{i}', 'diameter_constraint', weight=0.2 + 0.8 * risk_weight)
    
    # Add conflict edges between expensive edges competing for tree inclusion
    if ws:
        # Sort edges by weight (most expensive first)
        expensive_edges = [(i, ws[i]) for i in range(len(edges)) if i < len(ws)]
        expensive_edges.sort(key=lambda x: x[1], reverse=True)
        
        # Add conflicts between most expensive edges (they compete for exclusion)
        top_expensive = expensive_edges[:min(10, len(expensive_edges)//3)]
        for i in range(len(top_expensive)):
            for j in range(i+1, len(top_expensive)):
                edge1_idx, weight1 = top_expensive[i]
                edge2_idx, weight2 = top_expensive[j]
                # Conflict strength based on combined expense
                conflict_strength = (weight1 + weight2) / (2 * max_weight) if max_weight > 0 else 0.5
                G.add_edge(f'edge_{edge1_idx}', f'edge_{edge2_idx}', weight=conflict_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()