#!/usr/bin/env python3
"""
Graph converter for ECP (Extended Competition Problem) - Soccer Tournament Ranking.
# Converter created with subagent_prompt.md v_02

This problem is about determining final rankings in a soccer tournament where teams play 
additional games and must meet position constraints. Key challenges include:
- Complex interaction between game results and final positions
- Position constraints create tight coupling between teams
- Point differences create ranking conflicts that must be resolved
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the ECP tournament instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with constraint nodes
    - Team nodes (type 0) represent decision variables for positions
    - Game constraint nodes (type 1) enforce valid game outcomes
    - Position constraint nodes (type 1) enforce ranking requirements
    - Competition pressure creates conflicts between teams with similar points
    """
    n = json_data.get('n', 0)
    iPoints = json_data.get('iPoints', [])
    games = json_data.get('games', [])
    positionConstraints = json_data.get('positionConstraints', [])
    
    if n == 0:
        return nx.Graph()
    
    G = nx.Graph()
    
    # Parse games array (flat list: team1, team2, team1, team2, ...)
    parsed_games = []
    for i in range(0, len(games), 2):
        if i + 1 < len(games):
            parsed_games.append((games[i] - 1, games[i + 1] - 1))  # Convert to 0-based
    
    # Calculate statistics for weighting
    max_points = max(iPoints) if iPoints else 1
    min_points = min(iPoints) if iPoints else 0
    point_range = max_points - min_points if max_points > min_points else 1
    
    # Calculate how many games each team plays for pressure weighting
    team_game_counts = [0] * n
    for team1, team2 in parsed_games:
        if 0 <= team1 < n:
            team_game_counts[team1] += 1
        if 0 <= team2 < n:
            team_game_counts[team2] += 1
    
    max_game_count = max(team_game_counts) if team_game_counts else 1
    
    # Team nodes (decision variables for positions)
    for i in range(n):
        initial_points = iPoints[i] if i < len(iPoints) else 0
        
        # Weight by competitive pressure: combination of initial points and games to play
        point_pressure = (initial_points - min_points) / point_range
        game_pressure = team_game_counts[i] / max_game_count
        # Teams with mid-range points and many games have highest uncertainty
        uncertainty = 1.0 - abs(point_pressure - 0.5) * 2.0  # Peak at 0.5, decay to edges
        
        # Combine pressures with non-linear weighting
        weight = 0.3 * point_pressure + 0.3 * game_pressure + 0.4 * uncertainty
        weight = max(0.1, min(1.0, weight))  # Ensure valid range
        
        G.add_node(f'team_{i}', type=0, weight=weight)
    
    # Game constraint nodes - each game creates a constraint
    for game_idx, (team1, team2) in enumerate(parsed_games):
        if 0 <= team1 < n and 0 <= team2 < n:
            # Weight by competitiveness of matchup
            points1 = iPoints[team1] if team1 < len(iPoints) else 0
            points2 = iPoints[team2] if team2 < len(iPoints) else 0
            point_diff = abs(points1 - points2)
            
            # Closer games are more critical (higher weight)
            competitiveness = 1.0 - (point_diff / point_range) if point_range > 0 else 0.5
            # Apply exponential function to emphasize very close matchups
            weight = math.exp(-2.0 * point_diff / point_range) if point_range > 0 else 0.5
            weight = max(0.1, min(1.0, weight))
            
            G.add_node(f'game_{game_idx}', type=1, weight=weight)
            
            # Connect teams to their games
            G.add_edge(f'team_{team1}', f'game_{game_idx}', weight=0.8)
            G.add_edge(f'team_{team2}', f'game_{game_idx}', weight=0.8)
    
    # Position constraint nodes
    if len(positionConstraints) >= 3:
        num_constraints = len(positionConstraints) // 3
        
        for i in range(num_constraints):
            base_idx = i * 3
            if base_idx + 2 < len(positionConstraints):
                team_id = positionConstraints[base_idx] - 1  # Convert to 0-based
                constraint_type = positionConstraints[base_idx + 1]
                target_position = positionConstraints[base_idx + 2]
                
                if 0 <= team_id < n:
                    # Weight by constraint tightness
                    # More restrictive constraints (requiring specific positions) are weighted higher
                    if constraint_type == 1:  # Exact position
                        tightness = 1.0
                    elif constraint_type in [4, 5]:  # Less than / less than or equal
                        tightness = target_position / n  # Earlier positions are tighter
                    else:  # Greater than / greater than or equal
                        tightness = (n - target_position + 1) / n  # Later positions are tighter
                    
                    # Apply logarithmic scaling for position constraints
                    weight = math.log(1 + 9 * tightness) / math.log(10)  # Maps [0,1] to [0,1] with log curve
                    weight = max(0.2, min(1.0, weight))
                    
                    G.add_node(f'pos_constraint_{i}', type=1, weight=weight)
                    G.add_edge(f'team_{team_id}', f'pos_constraint_{i}', weight=tightness)
    
    # Add competition pressure edges between teams with similar initial points
    # This models the fact that teams with similar points compete for similar final positions
    point_tolerance = point_range * 0.3  # Teams within 30% of point range compete directly
    
    for i in range(n):
        for j in range(i + 1, n):
            points_i = iPoints[i] if i < len(iPoints) else 0
            points_j = iPoints[j] if j < len(iPoints) else 0
            point_diff = abs(points_i - points_j)
            
            if point_diff <= point_tolerance and point_range > 0:
                # Competition intensity based on how close the teams are in points
                competition = 1.0 - (point_diff / point_tolerance)
                # Use square root to moderate the effect
                weight = math.sqrt(competition) * 0.7  # Scale down conflict edges
                weight = max(0.1, min(0.7, weight))  # Cap conflict edges lower than constraint edges
                
                G.add_edge(f'team_{i}', f'team_{j}', weight=weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()