#!/usr/bin/env python3
"""
Graph converter for fastfood (facility location) problem.
Converter created with subagent_prompt.md v_02

This problem is about placing a given number of depots at restaurant locations
to minimize the total distance from all restaurants to their nearest depot.
Key challenges: determining optimal depot spacing along a line to minimize service distances.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the fastfood facility location problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with restaurants, potential depot locations,
    and constraints. Focus on proximity relationships and depot coverage decisions.
    - Restaurants are type 0 nodes (decision points for coverage)
    - Depot location constraints are type 1 nodes (capacity/placement rules)
    - Depot candidates are type 2 nodes (limited resources)
    - Edge weights reflect distance costs and placement viability
    """
    nr = json_data.get('nr', 0)
    number_of_depots = json_data.get('number_of_depots', 1)
    k = json_data.get('k', [])
    
    if not k or nr == 0:
        # Return minimal graph for empty instances
        G = nx.Graph()
        G.add_node('dummy', type=0, weight=0.5)
        return G
    
    G = nx.Graph()
    
    # Calculate problem characteristics for weighting
    max_pos = max(k) if k else 1
    min_pos = min(k) if k else 0
    span = max_pos - min_pos if max_pos > min_pos else 1
    unique_positions = len(set(k))
    
    # Type 0: Restaurant nodes (decision points for coverage)
    # Weight by isolation (distance to nearest other restaurant)
    for i in range(nr):
        pos = k[i]
        
        # Calculate isolation weight (higher for more isolated restaurants)
        min_dist_to_other = float('inf')
        for j in range(nr):
            if i != j:
                dist = abs(k[j] - pos)
                if dist > 0:
                    min_dist_to_other = min(min_dist_to_other, dist)
        
        if min_dist_to_other == float('inf'):
            isolation = 1.0
        else:
            # Use exponential decay for isolation weight
            isolation = min(1.0, math.exp(-2.0 * min_dist_to_other / span))
        
        # Weight also considers position extremity (edge restaurants are harder to cover)
        edge_factor = 1.0
        if pos == min_pos or pos == max_pos:
            edge_factor = 1.5
        
        weight = min(1.0, isolation * edge_factor)
        G.add_node(f'restaurant_{i}', type=0, weight=weight)
    
    # Type 2: Potential depot locations (resource nodes)
    # Each unique position is a potential depot location
    unique_pos = sorted(set(k))
    for pos in unique_pos:
        # Weight by coverage potential (how many restaurants this position can serve well)
        coverage_score = 0.0
        for rest_pos in k:
            dist = abs(pos - rest_pos)
            # Exponential decay for coverage contribution
            coverage_score += math.exp(-3.0 * dist / span)
        
        # Normalize by number of restaurants
        coverage_weight = min(1.0, coverage_score / nr)
        G.add_node(f'depot_loc_{pos}', type=2, weight=coverage_weight)
    
    # Type 1: Constraint nodes
    
    # 1. Depot count constraint (must place exactly number_of_depots depots)
    depot_density = number_of_depots / unique_positions if unique_positions > 0 else 0.5
    G.add_node('depot_count_constraint', type=1, weight=min(1.0, depot_density))
    
    # 2. Ordering constraint (depots must be ordered by position)
    # More complex with more depots
    ordering_complexity = min(1.0, number_of_depots / 10.0)
    G.add_node('ordering_constraint', type=1, weight=ordering_complexity)
    
    # 3. Coverage constraints for each restaurant (must be served by some depot)
    for i in range(nr):
        pos = k[i]
        
        # Calculate serving difficulty (average distance to potential depot locations)
        total_dist = sum(abs(pos - depot_pos) for depot_pos in unique_pos)
        avg_dist = total_dist / len(unique_pos) if unique_pos else 0
        
        # Normalize and invert (higher weight for harder to serve restaurants)
        serving_difficulty = min(1.0, avg_dist / span) if span > 0 else 0.5
        G.add_node(f'coverage_constraint_{i}', type=1, weight=serving_difficulty)
    
    # Edges: Restaurant-depot location relationships
    for i in range(nr):
        rest_pos = k[i]
        for depot_pos in unique_pos:
            dist = abs(rest_pos - depot_pos)
            
            # Distance-based weight (closer = stronger relationship)
            if span > 0:
                # Use exponential decay for distance relationships
                distance_weight = math.exp(-4.0 * dist / span)
            else:
                distance_weight = 1.0 if dist == 0 else 0.1
            
            G.add_edge(f'restaurant_{i}', f'depot_loc_{depot_pos}', 
                      weight=min(1.0, distance_weight))
    
    # Edges: Restaurants to their coverage constraints
    for i in range(nr):
        G.add_edge(f'restaurant_{i}', f'coverage_constraint_{i}', weight=1.0)
    
    # Edges: Depot locations to count constraint
    for depot_pos in unique_pos:
        # Weight by how much this location contributes to depot selection complexity
        selection_weight = min(1.0, 1.0 / number_of_depots) if number_of_depots > 0 else 0.5
        G.add_edge(f'depot_loc_{depot_pos}', 'depot_count_constraint', 
                  weight=selection_weight)
    
    # Edges: Adjacent depot locations to ordering constraint
    for i in range(len(unique_pos) - 1):
        pos1, pos2 = unique_pos[i], unique_pos[i + 1]
        # Weight by gap size (larger gaps create more ordering complexity)
        gap = pos2 - pos1
        gap_weight = min(1.0, gap / span) if span > 0 else 0.5
        
        G.add_edge(f'depot_loc_{pos1}', 'ordering_constraint', weight=gap_weight)
        G.add_edge(f'depot_loc_{pos2}', 'ordering_constraint', weight=gap_weight)
    
    # Add conflict edges between restaurants that are far apart
    # (they create tension in depot placement decisions)
    for i in range(nr):
        for j in range(i + 1, nr):
            dist = abs(k[i] - k[j])
            # Only add conflict edges for restaurants that are far apart
            if span > 0 and dist > span * 0.3:
                conflict_weight = min(1.0, dist / span)
                G.add_edge(f'restaurant_{i}', f'restaurant_{j}', 
                          weight=conflict_weight * 0.3)  # Scale down conflict weights
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()