#!/usr/bin/env python3
"""
Graph converter for fillomino problem.
Converter created with subagent_prompt.md v_02

This problem is about partitioning a grid into areas where each cell contains
a number equal to the size of its area, and adjacent areas must have different sizes.
Key challenges: area boundary placement, ensuring connectivity, satisfying size constraints.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the fillomino instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model the spatial structure and constraints
    - Cell nodes (type 0): Each grid cell as a variable node
    - Area constraint nodes (type 1): Constraints for area formation
    - Adjacency constraint nodes (type 1): Different-size constraints for adjacent areas
    - Size constraint nodes (type 1): Constraints ensuring cells match area size
    """
    rows = json_data.get('rows', 0)
    cols = json_data.get('cols', 0)
    puzzle = json_data.get('puzzle', [])
    
    G = nx.Graph()
    
    # Convert flat puzzle array to 2D for easier access
    grid = {}
    for r in range(rows):
        for c in range(cols):
            idx = r * cols + c
            grid[(r, c)] = puzzle[idx] if idx < len(puzzle) else 0
    
    # Cell nodes (type 0) - weight by constraint level
    clue_positions = set()
    max_clue = max(puzzle) if puzzle else 1
    
    for r in range(rows):
        for c in range(cols):
            clue = grid[(r, c)]
            if clue > 0:
                clue_positions.add((r, c))
                # Clue cells are highly constrained - higher weight for larger clues
                weight = 0.7 + 0.3 * (clue / max_clue)
            else:
                # Empty cells get moderate weight, higher for central positions
                centrality = 1.0 - (abs(r - rows//2) + abs(c - cols//2)) / (rows + cols)
                weight = 0.3 + 0.4 * centrality
            
            G.add_node(f'cell_{r}_{c}', type=0, weight=weight)
    
    # Area formation constraints (type 1) - one per potential area size
    # Weight by how restrictive the size is
    total_cells = rows * cols
    for size in range(1, min(10, total_cells + 1)):  # fillomino typically uses 1-9
        # Larger areas are harder to place - exponential difficulty
        difficulty = math.exp(-size / 3.0)  # Exponential decay for larger sizes
        G.add_node(f'area_size_{size}', type=1, weight=difficulty)
    
    # Adjacency constraints (type 1) - different sizes for adjacent areas
    adjacency_count = 0
    for r in range(rows):
        for c in range(cols):
            # Right adjacency
            if c < cols - 1:
                # Weight by potential conflict - higher if both cells have clues
                clue1 = grid[(r, c)]
                clue2 = grid[(r, c + 1)]
                if clue1 > 0 and clue2 > 0 and clue1 == clue2:
                    # Same clues adjacent - very constraining
                    weight = 0.9
                elif clue1 > 0 or clue2 > 0:
                    # One clue - moderately constraining
                    weight = 0.6
                else:
                    # No clues - basic constraint
                    weight = 0.4
                
                G.add_node(f'adj_h_{r}_{c}', type=1, weight=weight)
                G.add_edge(f'cell_{r}_{c}', f'adj_h_{r}_{c}', weight=0.8)
                G.add_edge(f'cell_{r}_{c+1}', f'adj_h_{r}_{c}', weight=0.8)
                adjacency_count += 1
            
            # Down adjacency
            if r < rows - 1:
                clue1 = grid[(r, c)]
                clue2 = grid[(r + 1, c)]
                if clue1 > 0 and clue2 > 0 and clue1 == clue2:
                    weight = 0.9
                elif clue1 > 0 or clue2 > 0:
                    weight = 0.6
                else:
                    weight = 0.4
                
                G.add_node(f'adj_v_{r}_{c}', type=1, weight=weight)
                G.add_edge(f'cell_{r}_{c}', f'adj_v_{r}_{c}', weight=0.8)
                G.add_edge(f'cell_{r+1}_{c}', f'adj_v_{r}_{c}', weight=0.8)
                adjacency_count += 1
    
    # Size consistency constraints (type 1) - each clue creates a size constraint
    for r, c in clue_positions:
        clue = grid[(r, c)]
        # Higher clue values are more constraining
        tightness = min(1.0, clue / 6.0)  # Normalize, higher clues are tighter
        G.add_node(f'size_clue_{r}_{c}', type=1, weight=tightness)
        G.add_edge(f'cell_{r}_{c}', f'size_clue_{r}_{c}', weight=1.0)
        
        # Connect to the area size constraint
        G.add_edge(f'area_size_{clue}', f'size_clue_{r}_{c}', weight=0.9)
    
    # Connect cells to area size constraints based on potential
    for r in range(rows):
        for c in range(cols):
            cell_node = f'cell_{r}_{c}'
            clue = grid[(r, c)]
            
            if clue > 0:
                # Clue cells strongly connect to their required size
                G.add_edge(cell_node, f'area_size_{clue}', weight=1.0)
            else:
                # Empty cells connect to all possible sizes with distance-based weights
                for size in range(1, min(10, total_cells + 1)):
                    # Connection strength decreases with larger sizes (harder to form)
                    strength = math.exp(-size / 4.0)
                    if strength > 0.1:  # Only add meaningful connections
                        G.add_edge(cell_node, f'area_size_{size}', weight=strength)
    
    # Add connectivity constraints within potential areas (weak connections)
    # Connect nearby cells that could be in the same area
    for r in range(rows):
        for c in range(cols):
            for dr, dc in [(0, 1), (1, 0)]:  # Right and down
                nr, nc = r + dr, c + dc
                if 0 <= nr < rows and 0 <= nc < cols:
                    clue1, clue2 = grid[(r, c)], grid[(nr, nc)]
                    
                    # Same clues or both empty - could be same area
                    if (clue1 == clue2 and clue1 > 0) or (clue1 == 0 and clue2 == 0):
                        # Weak connection for potential same-area cells
                        G.add_edge(f'cell_{r}_{c}', f'cell_{nr}_{nc}', weight=0.3)
                    elif clue1 > 0 and clue2 > 0 and clue1 != clue2:
                        # Different clues - conflict edge (must be different areas)
                        conflict_strength = 0.8
                        G.add_edge(f'cell_{r}_{c}', f'cell_{nr}_{nc}', weight=conflict_strength)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()