#!/usr/bin/env python3
"""
Graph converter for filter problem.
Created using subagent_prompt.md version: v_02

This problem is about filter scheduling and resource assignment.
Operations (add/mul) need to be scheduled with dependencies and resource constraints.
Key challenges: dependency ordering, resource contention, makespan minimization
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the filter scheduling problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create bipartite graph modeling operations and constraints
    - Operations (type 0): add and mul operations with different complexities
    - Constraints (type 1): resource constraints, dependency constraints
    - Resource nodes (type 2): adder and multiplier resources
    
    Note: Dependencies array missing from JSON - using available data structure
    """
    # Access data from json_data dict
    n = json_data.get('n', 0)
    add_ops = json_data.get('add', [])
    mul_ops = json_data.get('mul', [])
    last_ops = json_data.get('Last', [])
    
    del_add = json_data.get('del_add', 1)
    del_mul = json_data.get('del_mul', 2)
    number_add = json_data.get('number_add', 1)
    number_mul = json_data.get('number_mul', 1)
    
    G = nx.Graph()
    
    # Type 0: Operation nodes
    # Add operations - typically simpler, shorter duration
    for op in add_ops:
        # Weight by relative complexity and resource scarcity
        complexity = del_add / max(del_add + del_mul, 1)  # Lower for adds
        resource_pressure = len(add_ops) / max(number_add, 1)  # Higher when many adds compete
        # Final weight considers whether this is a critical (last) operation
        criticality_bonus = 0.3 if op in last_ops else 0.0
        weight = min(complexity + math.log(resource_pressure + 1) / 5 + criticality_bonus, 1.0)
        G.add_node(f'add_op_{op}', type=0, weight=weight)
    
    # Multiplication operations - typically more complex, longer duration
    for op in mul_ops:
        # Weight by relative complexity and resource scarcity
        complexity = del_mul / max(del_add + del_mul, 1)  # Higher for muls
        resource_pressure = len(mul_ops) / max(number_mul, 1)  # Higher when many muls compete
        # Final weight considers whether this is a critical (last) operation
        criticality_bonus = 0.3 if op in last_ops else 0.0
        weight = min(complexity + math.log(resource_pressure + 1) / 5 + criticality_bonus, 1.0)
        G.add_node(f'mul_op_{op}', type=0, weight=weight)
    
    # Type 2: Resource nodes (limited resources that operations compete for)
    # Adder resources
    if number_add > 0:
        add_utilization = len(add_ops) / number_add if number_add > 0 else 1.0
        add_scarcity = min(add_utilization / 2, 1.0)  # More scarce = higher weight
        G.add_node('adder_resource', type=2, weight=add_scarcity)
    
    # Multiplier resources  
    if number_mul > 0:
        mul_utilization = len(mul_ops) / number_mul if number_mul > 0 else 1.0
        mul_scarcity = min(mul_utilization / 2, 1.0)  # More scarce = higher weight
        G.add_node('multiplier_resource', type=2, weight=mul_scarcity)
    
    # Type 1: Constraint nodes
    # Resource capacity constraints for adds
    if add_ops and number_add > 0:
        # Tightness based on utilization
        utilization = len(add_ops) / number_add
        tightness = min(utilization / 3, 1.0)  # Normalized tightness
        G.add_node('add_capacity_constraint', type=1, weight=tightness)
        
        # Connect all add operations to their capacity constraint
        for op in add_ops:
            # Edge weight based on operation's resource consumption relative to total capacity
            consumption_ratio = 1.0 / number_add  # Each operation consumes 1 unit
            G.add_edge(f'add_op_{op}', 'add_capacity_constraint', weight=consumption_ratio)
            
            # Connect operations to their resource
            if 'adder_resource' in G.nodes:
                G.add_edge(f'add_op_{op}', 'adder_resource', weight=consumption_ratio)
    
    # Resource capacity constraints for muls
    if mul_ops and number_mul > 0:
        # Tightness based on utilization
        utilization = len(mul_ops) / number_mul
        tightness = min(utilization / 3, 1.0)  # Normalized tightness
        G.add_node('mul_capacity_constraint', type=1, weight=tightness)
        
        # Connect all mul operations to their capacity constraint
        for op in mul_ops:
            # Edge weight based on operation's resource consumption relative to total capacity
            consumption_ratio = 1.0 / number_mul  # Each operation consumes 1 unit
            G.add_edge(f'mul_op_{op}', 'mul_capacity_constraint', weight=consumption_ratio)
            
            # Connect operations to their resource
            if 'multiplier_resource' in G.nodes:
                G.add_edge(f'mul_op_{op}', 'multiplier_resource', weight=consumption_ratio)
    
    # Makespan constraint (all operations must complete before objective)
    if last_ops:
        # Weight by how many operations contribute to makespan
        makespan_pressure = len(last_ops) / max(n, 1)
        G.add_node('makespan_constraint', type=1, weight=makespan_pressure)
        
        # Connect last operations to makespan constraint
        for op in last_ops:
            op_node = f'add_op_{op}' if op in add_ops else f'mul_op_{op}'
            if op_node in G.nodes:
                # Higher weight for longer operations affecting makespan
                duration = del_mul if op in mul_ops else del_add
                duration_weight = duration / max(del_add + del_mul, 1)
                G.add_edge(op_node, 'makespan_constraint', weight=duration_weight)
    
    # Add some conflict edges for highly contended resources
    # Operations of same type compete more directly
    if len(add_ops) > number_add:
        # Add conflicts between add operations when oversubscribed
        for i, op1 in enumerate(add_ops[:min(len(add_ops), 5)]):  # Top 5 to avoid too many edges
            for op2 in add_ops[i+1:min(len(add_ops), 5)]:
                conflict_intensity = (len(add_ops) - number_add) / max(len(add_ops), 1)
                G.add_edge(f'add_op_{op1}', f'add_op_{op2}', weight=conflict_intensity)
    
    if len(mul_ops) > number_mul:
        # Add conflicts between mul operations when oversubscribed
        for i, op1 in enumerate(mul_ops[:min(len(mul_ops), 5)]):  # Top 5 to avoid too many edges
            for op2 in mul_ops[i+1:min(len(mul_ops), 5)]:
                conflict_intensity = (len(mul_ops) - number_mul) / max(len(mul_ops), 1)
                G.add_edge(f'mul_op_{op1}', f'mul_op_{op2}', weight=conflict_intensity)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()