#!/usr/bin/env python3
"""
Graph converter for foxgeesecorn problem.
# Converter created with subagent_prompt.md v_02

This problem is about the classic "Fox, Geese, and Corn" optimization problem.
Key challenges: capacity constraints, predator-prey relationships, multi-trip planning.
The alone predicate prevents geese from eating corn and foxes from eating geese when left alone.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the foxgeesecorn problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: 
    - Items (foxes, geese, corn) are variable nodes with value-based weights
    - Trips are variable nodes with capacity constraints
    - Capacity constraints, predator-prey constraints are constraint nodes
    - Trip sequences create dependencies between consecutive trips
    """
    # Access data directly from json_data dict
    f = json_data.get('f', 0)  # number of foxes
    g = json_data.get('g', 0)  # number of geese
    c = json_data.get('c', 0)  # number of corn
    k = json_data.get('k', 1)  # capacity per trip
    t = json_data.get('t', 1)  # maximum trips
    pf = json_data.get('pf', 1)  # profit per fox
    pg = json_data.get('pg', 1)  # profit per goose
    pc = json_data.get('pc', 1)  # profit per corn
    
    # Create graph
    G = nx.Graph()
    
    # Calculate total items and max profit for normalization
    total_items = f + g + c
    max_profit = max(pf, pg, pc) if max(pf, pg, pc) > 0 else 1
    
    # Item type nodes (what needs to be transported)
    # Weight by profit density and quantity
    if f > 0:
        profit_weight = pf / max_profit
        quantity_weight = f / total_items if total_items > 0 else 0
        G.add_node('fox_items', type=0, weight=(profit_weight + quantity_weight) / 2)
    
    if g > 0:
        profit_weight = pg / max_profit
        quantity_weight = g / total_items if total_items > 0 else 0
        G.add_node('geese_items', type=0, weight=(profit_weight + quantity_weight) / 2)
    
    if c > 0:
        profit_weight = pc / max_profit
        quantity_weight = c / total_items if total_items > 0 else 0
        G.add_node('corn_items', type=0, weight=(profit_weight + quantity_weight) / 2)
    
    # Trip nodes (decision variables for each trip)
    for i in range(1, t + 1):
        # Weight trips by their importance in the sequence
        # Earlier trips are more critical as they enable later trips
        trip_importance = 1.0 - (i - 1) / t if t > 1 else 1.0
        G.add_node(f'trip_{i}', type=0, weight=trip_importance)
    
    # Capacity constraint nodes (one per trip)
    for i in range(1, t + 1):
        # Weight by capacity utilization pressure
        # Tighter capacity creates more constraints
        if k > 0:
            max_demand = min(total_items, 3)  # max 3 types of items
            capacity_pressure = min(max_demand / k, 1.0)
        else:
            capacity_pressure = 1.0
        G.add_node(f'capacity_constraint_{i}', type=1, weight=capacity_pressure)
    
    # Predator-prey constraint nodes
    # Fox-geese constraint (foxes eat geese when alone)
    if f > 0 and g > 0:
        # Weight by the severity of the predator-prey imbalance
        predator_pressure = min(f / max(g, 1), 2.0) / 2.0  # normalize to [0,1]
        G.add_node('fox_geese_constraint', type=1, weight=predator_pressure)
    
    # Geese-corn constraint (geese eat corn when alone)
    if g > 0 and c > 0:
        predator_pressure = min(g / max(c, 1), 2.0) / 2.0  # normalize to [0,1]
        G.add_node('geese_corn_constraint', type=1, weight=predator_pressure)
    
    # Trip sequence constraint (odd/even trip dependency)
    if t > 1:
        # Weight by complexity of trip sequencing
        sequence_complexity = min(t / 10.0, 1.0)  # normalize large t values
        G.add_node('trip_sequence_constraint', type=1, weight=sequence_complexity)
    
    # Add edges for relationships
    
    # Items participate in capacity constraints
    for i in range(1, t + 1):
        cap_constraint = f'capacity_constraint_{i}'
        trip_node = f'trip_{i}'
        
        # Connect trip to its capacity constraint
        G.add_edge(trip_node, cap_constraint, weight=1.0)
        
        # Connect item types to capacity constraints with weight based on demand
        if f > 0:
            demand_ratio = min(f / (k * t), 1.0) if k > 0 and t > 0 else 0.5
            G.add_edge('fox_items', cap_constraint, weight=demand_ratio)
        
        if g > 0:
            demand_ratio = min(g / (k * t), 1.0) if k > 0 and t > 0 else 0.5
            G.add_edge('geese_items', cap_constraint, weight=demand_ratio)
        
        if c > 0:
            demand_ratio = min(c / (k * t), 1.0) if k > 0 and t > 0 else 0.5
            G.add_edge('corn_items', cap_constraint, weight=demand_ratio)
    
    # Predator-prey relationships
    if f > 0 and g > 0:
        # Fox and geese participate in predator-prey constraint
        conflict_strength = min((f + g) / total_items, 1.0) if total_items > 0 else 0.5
        G.add_edge('fox_items', 'fox_geese_constraint', weight=conflict_strength)
        G.add_edge('geese_items', 'fox_geese_constraint', weight=conflict_strength)
    
    if g > 0 and c > 0:
        # Geese and corn participate in predator-prey constraint
        conflict_strength = min((g + c) / total_items, 1.0) if total_items > 0 else 0.5
        G.add_edge('geese_items', 'geese_corn_constraint', weight=conflict_strength)
        G.add_edge('corn_items', 'geese_corn_constraint', weight=conflict_strength)
    
    # Trip sequence dependencies
    if t > 1:
        for i in range(1, t + 1):
            # Each trip participates in the sequence constraint
            sequence_weight = 1.0 / t  # weight decreases as more trips are involved
            G.add_edge(f'trip_{i}', 'trip_sequence_constraint', weight=sequence_weight)
    
    # Add conflicts between incompatible items (direct predator-prey conflicts)
    if f > 0 and g > 0:
        # Direct conflict when foxes and geese might be left alone together
        conflict_intensity = math.exp(-2.0 * k / max(f + g, 1))  # exponential decay with capacity
        G.add_edge('fox_items', 'geese_items', weight=conflict_intensity)
    
    if g > 0 and c > 0:
        # Direct conflict when geese and corn might be left alone together
        conflict_intensity = math.exp(-2.0 * k / max(g + c, 1))  # exponential decay with capacity
        G.add_edge('geese_items', 'corn_items', weight=conflict_intensity)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()