#!/usr/bin/env python3
"""
Graph converter for freepizza problem.
Created using subagent_prompt.md version: v_02

This problem is about optimizing pizza ordering with vouchers.
Key challenges: Deciding which pizzas to assign to vouchers (buy vs free) to minimize cost.
The vouchers create complex interactions where expensive pizzas should be bought to get cheaper ones free.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the freepizza problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph with pizzas as type 0 nodes and vouchers as type 1 nodes.
    - Pizza nodes weighted by normalized price (higher price = higher weight)
    - Voucher nodes weighted by efficiency (free/buy ratio, normalized)
    - Edge weights represent compatibility/attraction between pizzas and vouchers
    - Add conflict edges between pizzas that compete for the same voucher slot
    """
    # Access data directly from json_data dict
    n = json_data.get('n', 0)  # number of pizzas
    m = json_data.get('m', 0)  # number of vouchers
    prices = json_data.get('price', [])
    buy_amounts = json_data.get('buy', [])
    free_amounts = json_data.get('free', [])
    
    # Create graph
    G = nx.Graph()
    
    # Calculate normalization factors
    max_price = max(prices) if prices else 1
    min_price = min(prices) if prices else 1
    price_range = max_price - min_price if max_price > min_price else 1
    
    # Add pizza nodes (type 0) with price-based weights
    for i in range(n):
        if i < len(prices):
            price = prices[i]
            # Higher price pizzas get higher weights as they're more valuable for optimization
            price_weight = (price - min_price) / price_range if price_range > 0 else 0.5
            # Use square root to avoid too extreme differences
            normalized_weight = math.sqrt(price_weight) * 0.8 + 0.2  # Range [0.2, 1.0]
        else:
            normalized_weight = 0.5
        
        G.add_node(f'pizza_{i}', type=0, weight=normalized_weight)
    
    # Calculate voucher efficiencies for valid vouchers only
    valid_vouchers = []
    voucher_efficiencies = []
    max_efficiency = 0
    
    for j in range(m):
        buy = buy_amounts[j] if j < len(buy_amounts) else 1
        free = free_amounts[j] if j < len(free_amounts) else 0
        
        # Skip invalid vouchers (buy=0 or free=0)
        if buy == 0 or free == 0:
            continue
            
        valid_vouchers.append(j)
        efficiency = free / buy
        voucher_efficiencies.append(efficiency)
        max_efficiency = max(max_efficiency, efficiency)
    
    # Add voucher nodes (type 1) with efficiency-based weights
    for j in range(m):
        buy = buy_amounts[j] if j < len(buy_amounts) else 1
        free = free_amounts[j] if j < len(free_amounts) else 0
        
        # Handle invalid vouchers with minimal weight
        if buy == 0 or free == 0:
            voucher_weight = 0.1
        else:
            efficiency = free / buy
            # Higher efficiency vouchers are more valuable
            voucher_weight = (efficiency / max_efficiency) if max_efficiency > 0 else 0.5
            # Use logarithmic scaling for efficiency to avoid extreme weights
            voucher_weight = math.log(1 + voucher_weight * 9) / math.log(10)  # Range [0, 1]
            voucher_weight = voucher_weight * 0.8 + 0.2  # Range [0.2, 1.0]
        
        G.add_node(f'voucher_{j}', type=1, weight=voucher_weight)
    
    # Add bipartite edges between pizzas and valid vouchers
    for i in range(n):
        pizza_price = prices[i] if i < len(prices) else 0
        
        for j in valid_vouchers:
            buy = buy_amounts[j]
            free = free_amounts[j]
                
            # Edge weight represents how attractive this voucher is for this pizza
            # High-price pizzas are more attractive for vouchers (should be bought to get free ones)
            # But also consider voucher efficiency
            
            price_attractiveness = (pizza_price - min_price) / price_range if price_range > 0 else 0.5
            voucher_efficiency = free / buy
            
            # Combine both factors with logarithmic scaling
            combined_weight = (price_attractiveness + voucher_efficiency) / 2
            edge_weight = math.log(1 + combined_weight * 9) / math.log(10)  # Normalize to [0,1]
            edge_weight = max(0.1, min(1.0, edge_weight))  # Ensure valid range
            
            G.add_edge(f'pizza_{i}', f'voucher_{j}', weight=edge_weight)
    
    # Add connections from invalid vouchers to some pizzas to avoid isolation
    # This represents that even invalid vouchers could theoretically be used
    invalid_vouchers = [j for j in range(m) if j not in valid_vouchers]
    for j in invalid_vouchers:
        # Connect to a few random pizzas with low weight
        for i in range(min(3, n)):  # Connect to first 3 pizzas
            G.add_edge(f'pizza_{i}', f'voucher_{j}', weight=0.05)
    
    # Add conflict edges between expensive pizzas (they compete for voucher slots)
    # Sort pizzas by price
    pizza_price_pairs = [(i, prices[i] if i < len(prices) else 0) for i in range(n)]
    pizza_price_pairs.sort(key=lambda x: x[1], reverse=True)
    
    # Add conflicts between top expensive pizzas (they compete for buy slots in vouchers)
    top_pizzas = pizza_price_pairs[:min(n, 8)]  # Top 8 or all if fewer
    
    for idx1 in range(len(top_pizzas)):
        for idx2 in range(idx1 + 1, len(top_pizzas)):
            pizza1_id, price1 = top_pizzas[idx1]
            pizza2_id, price2 = top_pizzas[idx2]
            
            # Conflict weight based on price similarity (similar prices compete more)
            price_diff = abs(price1 - price2)
            conflict_strength = 1.0 - (price_diff / price_range) if price_range > 0 else 0.5
            
            # Use exponential decay for conflict strength
            conflict_weight = math.exp(-2.0 * conflict_strength) * 0.4 + 0.1  # Range [0.1, 0.5]
            
            G.add_edge(f'pizza_{pizza1_id}', f'pizza_{pizza2_id}', weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()