#!/usr/bin/env python3
"""
Graph converter for Road Network Maintenance Scheduling (FULL) problem.
Converter created with subagent_prompt.md v_02

This problem is about scheduling maintenance worksheets on a road network to minimize 
disruption while respecting resource constraints and precedence rules.
Key challenges: resource allocation, precedence constraints, road blocking limits,
time windows, and perturbation minimization.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the road network maintenance scheduling problem.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Create a bipartite graph with:
    - Type 0: Worksheets (decision variables)
    - Type 1: Various constraints (time windows, precedence, capacity, blocking)
    - Type 2: Resources (work centers, roads)
    
    Key relationships:
    - Worksheets participate in constraints
    - Resource usage and conflicts
    - Precedence dependencies
    """
    # Extract basic parameters
    days = json_data.get('days', 0)
    roads = json_data.get('roads', 0) 
    centers = json_data.get('centers', 0)
    worksheets = json_data.get('worksheets', 0)
    activities = json_data.get('activities', 0)
    
    # Extract arrays (some may be missing from JSON)
    blocked_max = json_data.get('blocked_max', 0)
    precedences = json_data.get('precedences', 0)
    preceeds = json_data.get('preceeds', [])
    succeeds = json_data.get('succeeds', [])
    road_array = json_data.get('road', [])
    workers_array = json_data.get('workers', [])
    perterb = json_data.get('perterb', [])
    blocked_max_amount = json_data.get('blocked_max_amount', [])
    
    # Create graph
    G = nx.Graph()
    
    # Type 0: Worksheet nodes (decision variables)
    # Weight worksheets by their complexity (duration * activities used)
    max_duration = 1
    max_workers_per_worksheet = 1
    
    # Calculate worksheet complexity metrics
    for w in range(worksheets):
        activity_count = 0
        total_workers = 0
        roads_used = set()
        
        for a in range(activities):
            idx = w * activities + a
            if idx < len(road_array) and idx < len(workers_array):
                road_id = road_array[idx]
                worker_count = workers_array[idx]
                
                if road_id >= 0 and worker_count > 0:  # Valid activity
                    activity_count += 1
                    total_workers += worker_count
                    roads_used.add(road_id)
        
        # Worksheet complexity based on scope and resource usage
        complexity = (activity_count + 1) * (total_workers + 1) * (len(roads_used) + 1)
        max_duration = max(max_duration, complexity)
        max_workers_per_worksheet = max(max_workers_per_worksheet, total_workers)
    
    # Add worksheet nodes
    for w in range(worksheets):
        activity_count = 0
        total_workers = 0
        roads_used = set()
        
        for a in range(activities):
            idx = w * activities + a
            if idx < len(road_array) and idx < len(workers_array):
                road_id = road_array[idx]
                worker_count = workers_array[idx]
                
                if road_id >= 0 and worker_count > 0:
                    activity_count += 1
                    total_workers += worker_count
                    roads_used.add(road_id)
        
        # Weight by complexity using non-linear scaling
        complexity = (activity_count + 1) * (total_workers + 1) * (len(roads_used) + 1)
        weight = math.sqrt(complexity / max_duration) if max_duration > 0 else 0.5
        
        G.add_node(f'worksheet_{w}', type=0, weight=min(weight, 1.0))
    
    # Type 1: Constraint nodes
    
    # Precedence constraints
    for p in range(precedences):
        if p < len(preceeds) and p < len(succeeds):
            pred = preceeds[p]
            succ = succeeds[p]
            
            # Weight by how constraining the precedence is (based on time pressure)
            constraint_weight = 0.8  # High weight as precedences are critical
            G.add_node(f'precedence_{p}', type=1, weight=constraint_weight)
            
            # Connect to involved worksheets
            if pred < worksheets and succ < worksheets:
                G.add_edge(f'worksheet_{pred}', f'precedence_{p}', weight=0.9)
                G.add_edge(f'worksheet_{succ}', f'precedence_{p}', weight=0.9)
    
    # Road blocking constraints (for each blocked rule)
    for b in range(blocked_max):
        if b < len(blocked_max_amount):
            max_amount = blocked_max_amount[b]
            # Weight by how restrictive the blocking limit is
            constraint_weight = 1.0 - (max_amount / max(roads, 1))
            G.add_node(f'blocking_{b}', type=1, weight=min(constraint_weight, 1.0))
    
    # Work center capacity constraints (one per center)
    for c in range(centers):
        # High weight as capacity constraints are critical
        G.add_node(f'capacity_{c}', type=1, weight=0.9)
    
    # Type 2: Resource nodes
    
    # Work center resources
    for c in range(centers):
        # Weight by scarcity (inverse of available capacity)
        # We don't have available_workers from JSON, so use uniform weight
        G.add_node(f'center_{c}', type=2, weight=0.6)
    
    # Road resources (weight by usage frequency)
    road_usage = {}
    for idx in range(len(road_array)):
        road_id = road_array[idx]
        if road_id >= 0:
            road_usage[road_id] = road_usage.get(road_id, 0) + 1
    
    max_usage = max(road_usage.values()) if road_usage else 1
    for road_id in range(roads):
        usage = road_usage.get(road_id, 0)
        # High usage roads are more critical resources
        weight = math.log(usage + 1) / math.log(max_usage + 1) if max_usage > 0 else 0.3
        G.add_node(f'road_{road_id}', type=2, weight=min(weight, 1.0))
    
    # Edges: Worksheet-constraint relationships
    
    # Connect worksheets to capacity constraints based on work center assignment
    # Since we don't have work_center data in JSON, we'll estimate based on worksheet patterns
    for w in range(worksheets):
        # Assume worksheets are distributed across centers
        center = w % centers
        consumption_weight = 0.7  # Default consumption weight
        G.add_edge(f'worksheet_{w}', f'capacity_{center}', weight=consumption_weight)
        G.add_edge(f'worksheet_{w}', f'center_{center}', weight=consumption_weight)
    
    # Connect worksheets to roads they use
    for w in range(worksheets):
        roads_used = {}
        total_workers_on_roads = 0
        
        for a in range(activities):
            idx = w * activities + a
            if idx < len(road_array) and idx < len(workers_array):
                road_id = road_array[idx]
                worker_count = workers_array[idx]
                
                if road_id >= 0 and worker_count > 0:
                    roads_used[road_id] = roads_used.get(road_id, 0) + worker_count
                    total_workers_on_roads += worker_count
        
        # Connect to road resources
        for road_id, worker_count in roads_used.items():
            if road_id < roads:
                # Weight by intensity of road usage
                usage_intensity = worker_count / max(total_workers_on_roads, 1)
                weight = math.sqrt(usage_intensity)  # Non-linear scaling
                G.add_edge(f'worksheet_{w}', f'road_{road_id}', weight=min(weight, 1.0))
    
    # Connect worksheets to blocking constraints
    # Since we don't have blocked_roads data, we'll create approximate connections
    for w in range(worksheets):
        roads_used = set()
        for a in range(activities):
            idx = w * activities + a
            if idx < len(road_array):
                road_id = road_array[idx]
                if road_id >= 0:
                    roads_used.add(road_id)
        
        # Connect to blocking constraints if worksheet uses many roads
        if len(roads_used) > 1:  # Only connect if worksheet is road-intensive
            for b in range(blocked_max):
                # Weight by how much this worksheet contributes to blocking
                blocking_impact = len(roads_used) / max(roads, 1)
                weight = math.exp(-2.0 * (1.0 - blocking_impact))  # Exponential scaling
                if weight > 0.1:  # Only add significant connections
                    G.add_edge(f'worksheet_{w}', f'blocking_{b}', weight=min(weight, 1.0))
    
    # Add conflict edges between worksheets that compete for the same scarce resources
    resource_conflicts = {}
    for w in range(worksheets):
        roads_used = []
        for a in range(activities):
            idx = w * activities + a
            if idx < len(road_array) and idx < len(workers_array):
                road_id = road_array[idx]
                worker_count = workers_array[idx]
                if road_id >= 0 and worker_count > 0:
                    roads_used.append((road_id, worker_count))
        resource_conflicts[w] = roads_used
    
    # Add conflict edges between worksheets that heavily use the same roads
    for w1 in range(worksheets):
        for w2 in range(w1 + 1, worksheets):
            common_resource_pressure = 0
            roads1 = {road_id: workers for road_id, workers in resource_conflicts.get(w1, [])}
            roads2 = {road_id: workers for road_id, workers in resource_conflicts.get(w2, [])}
            
            for road_id in set(roads1.keys()) & set(roads2.keys()):
                # High conflict if both worksheets heavily use the same road
                pressure = (roads1[road_id] + roads2[road_id]) / 10.0  # Scale down
                common_resource_pressure += pressure
            
            if common_resource_pressure > 0.5:  # Only add significant conflicts
                conflict_weight = min(common_resource_pressure / 2.0, 1.0)
                G.add_edge(f'worksheet_{w1}', f'worksheet_{w2}', weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()