#!/usr/bin/env python3
"""
Graph converter for gfd-schedule problem.
Created using subagent_prompt.md version: v_02

This problem is about scheduling items of different kinds using limited facilities.
Key challenges: facility conflicts, deadline pressure, grouping constraints, and daily capacity limits.
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the gfd-schedule problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with explicit constraint nodes
    - Items (type 0): decision variables needing scheduling
    - Constraints (type 1): various scheduling constraints
    - Facilities (type 2): limited shared resources
    
    Key relationships:
    - Items connect to facility constraints based on availability
    - Items connect to deadline constraints based on urgency
    - Items connect to kind constraints based on grouping requirements
    - Daily capacity constraints connect to all items
    """
    # Access data directly from json_data dict
    N = json_data.get('N', 0)
    F = json_data.get('F', 1)
    MaxItemsPerDay = json_data.get('MaxItemsPerDay', 1)
    MaxDay = json_data.get('MaxDay', 1)
    kind = json_data.get('kind', [])
    producedDay = json_data.get('producedDay', [])
    deadLineDay = json_data.get('deadLineDay', [])
    
    G = nx.Graph()
    
    # Calculate problem statistics for weight normalization
    max_deadline_pressure = max((deadLineDay[i] - producedDay[i]) for i in range(N)) if N > 0 else 1
    unique_kinds = len(set(kind)) if kind else 1
    
    # Type 0 nodes: Items (decision variables)
    for i in range(N):
        # Weight by deadline pressure and processing window
        if i < len(producedDay) and i < len(deadLineDay):
            processing_window = deadLineDay[i] - producedDay[i]
            deadline_pressure = 1.0 - (processing_window / max_deadline_pressure) if max_deadline_pressure > 0 else 0.5
            # Add urgency based on how close we are to deadline
            urgency_factor = 1.0 - (producedDay[i] / MaxDay) if MaxDay > 0 else 0.5
            # Combine deadline pressure with urgency using non-linear function
            item_weight = math.sqrt(deadline_pressure * urgency_factor)
        else:
            item_weight = 0.5
            
        G.add_node(f'item_{i}', type=0, weight=min(item_weight, 1.0))
    
    # Type 2 nodes: Facilities (shared resources)
    for f in range(1, F + 1):
        # Weight facilities by their expected usage (inversely by availability)
        facility_weight = min(1.0, N / (F * MaxDay)) if F > 0 and MaxDay > 0 else 0.5
        G.add_node(f'facility_{f}', type=2, weight=facility_weight)
    
    # Type 1 nodes: Kind constraints (grouping constraints)
    kind_items = {}
    for i in range(N):
        if i < len(kind):
            k = kind[i]
            if k not in kind_items:
                kind_items[k] = []
            kind_items[k].append(i)
    
    for k, items in kind_items.items():
        # Weight by complexity - larger groups are more constrained
        group_size_factor = len(items) / N if N > 0 else 0.5
        # More kinds = more complex grouping decisions
        kind_complexity = unique_kinds / max(F, 1)  # Normalize by facility availability
        constraint_weight = math.sqrt(group_size_factor * kind_complexity)
        G.add_node(f'kind_constraint_{k}', type=1, weight=min(constraint_weight, 1.0))
    
    # Type 1 nodes: Daily capacity constraints
    for d in range(1, MaxDay + 1):
        # Weight by expected contention on this day
        items_needing_this_day = sum(1 for i in range(N) 
                                   if i < len(producedDay) and i < len(deadLineDay)
                                   and producedDay[i] <= d <= deadLineDay[i])
        capacity_pressure = items_needing_this_day / max(MaxItemsPerDay, 1)
        # Use exponential function to emphasize high-pressure days
        day_weight = 1.0 - math.exp(-2.0 * capacity_pressure) if capacity_pressure > 0 else 0.1
        G.add_node(f'day_capacity_{d}', type=1, weight=min(day_weight, 1.0))
    
    # Type 1 nodes: Individual deadline constraints
    for i in range(N):
        if i < len(deadLineDay) and i < len(producedDay):
            # Weight by deadline tightness
            processing_window = deadLineDay[i] - producedDay[i]
            tightness = 1.0 - (processing_window / max_deadline_pressure) if max_deadline_pressure > 0 else 0.5
            G.add_node(f'deadline_constraint_{i}', type=1, weight=tightness)
    
    # Edges: Item to kind constraints
    for k, items in kind_items.items():
        for i in items:
            # Weight by how much this item contributes to kind constraint complexity
            contribution = 1.0 / len(items) if len(items) > 0 else 1.0
            G.add_edge(f'item_{i}', f'kind_constraint_{k}', weight=contribution)
    
    # Edges: Item to facility nodes (all items can potentially use all facilities)
    for i in range(N):
        for f in range(1, F + 1):
            # Weight by facility contention
            facility_demand = 1.0 / F  # Equal demand assumption
            G.add_edge(f'item_{i}', f'facility_{f}', weight=facility_demand)
    
    # Edges: Item to daily capacity constraints
    for i in range(N):
        if i < len(producedDay) and i < len(deadLineDay):
            # Connect item to all feasible days (within its window)
            feasible_days = deadLineDay[i] - producedDay[i] + 1
            for d in range(producedDay[i] + 1, deadLineDay[i] + 1):
                if d <= MaxDay:
                    # Weight by how critical this day is for this item
                    day_criticality = 1.0 / feasible_days if feasible_days > 0 else 1.0
                    G.add_edge(f'item_{i}', f'day_capacity_{d}', weight=day_criticality)
    
    # Edges: Item to deadline constraints
    for i in range(N):
        G.add_edge(f'item_{i}', f'deadline_constraint_{i}', weight=1.0)
    
    # Add conflict edges between items of different kinds that compete for same facilities
    # Only add for highly constrained scenarios
    if F > 0 and N / F > 2:  # High facility contention
        for k1, items1 in kind_items.items():
            for k2, items2 in kind_items.items():
                if k1 < k2:  # Avoid duplicate edges
                    # Add conflicts between items of different kinds
                    for i1 in items1[:min(3, len(items1))]:  # Limit to avoid too many edges
                        for i2 in items2[:min(3, len(items2))]:
                            # Weight by facility scarcity
                            conflict_weight = min(1.0, N / (F * MaxDay)) if F > 0 and MaxDay > 0 else 0.3
                            G.add_edge(f'item_{i1}', f'item_{i2}', weight=conflict_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()