#!/usr/bin/env python3
"""
Graph converter for ghoulomb problem.
Created using subagent_prompt.md version: v_02

This problem is about finding three Golomb rulers sequentially where only the middle
one's length is minimized. It's a variant of the classic Golomb ruler problem with
intentional redundancy to make the search more challenging.

Key challenges: 
- Three interleaved ruler construction processes
- Optimization focuses only on middle ruler
- Uses cumulative constraints instead of all_different
- Complex precedence and distinctness requirements
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the ghoulomb problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model three Golomb rulers with precedence and distinctness constraints
    - Variables are mark positions for each ruler
    - Constraints enforce ordering, distinctness, and symmetry breaking
    - Middle ruler is more critical (objective focus)
    - Difference variables create additional constraint structure
    """
    # Extract ruler sizes
    m1 = json_data.get('m1', 0)
    m2 = json_data.get('m2', 0)
    m3 = json_data.get('m3', 0)
    
    if m1 == 0 or m2 == 0 or m3 == 0:
        # Fallback for empty data
        return nx.Graph()
    
    G = nx.Graph()
    
    # Variable nodes: mark positions for all three rulers
    # Middle ruler (m2) has higher importance since it's optimized
    for i in range(m1):
        # First ruler marks - lower weight since not optimized
        centrality = (i + 1) / m1  # Position importance
        G.add_node(f'mark1_{i}', type=0, weight=0.3 * centrality)
    
    for i in range(m2):
        # Second ruler marks - highest weight (optimization target)
        centrality = (i + 1) / m2
        # Extra weight for being the optimization target
        G.add_node(f'mark2_{i}', type=0, weight=0.8 * centrality + 0.2)
    
    for i in range(m3):
        # Third ruler marks - lower weight since not optimized
        centrality = (i + 1) / m3
        G.add_node(f'mark3_{i}', type=0, weight=0.3 * centrality)
    
    # Constraint nodes for each ruler's structure
    
    # Ordering constraints for each ruler
    for ruler, m in [('1', m1), ('2', m2), ('3', m3)]:
        # Ordering constraints: mark[i] < mark[i+1]
        for i in range(m - 1):
            constraint_weight = 1.0  # All ordering constraints equally critical
            G.add_node(f'order_{ruler}_{i}', type=1, weight=constraint_weight)
            # Connect adjacent marks to ordering constraint
            G.add_edge(f'mark{ruler}_{i}', f'order_{ruler}_{i}', weight=0.8)
            G.add_edge(f'mark{ruler}_{i+1}', f'order_{ruler}_{i}', weight=0.8)
    
    # Distinctness constraints for difference variables
    for ruler, m in [('1', m1), ('2', m2), ('3', m3)]:
        num_diffs = (m * (m - 1)) // 2
        if num_diffs > 0:
            # Weight distinctness by complexity - more marks = harder
            complexity = math.log(num_diffs + 1) / math.log(50)  # Normalize to ~[0,1]
            distinct_weight = min(complexity, 1.0)
            
            # Create constraint node for distinctness of differences
            G.add_node(f'distinct_{ruler}', type=1, weight=distinct_weight)
            
            # Connect all marks to distinctness constraint 
            # (since all pairs contribute to differences)
            for i in range(m):
                # Connection strength based on how many differences involve this mark
                involvement = (2 * (m - 1) - abs(2 * i - (m - 1))) / (2 * (m - 1))
                edge_weight = 0.6 * involvement
                G.add_edge(f'mark{ruler}_{i}', f'distinct_{ruler}', weight=edge_weight)
    
    # Symmetry breaking constraints
    for ruler, m in [('1', m1), ('2', m2), ('3', m3)]:
        if m > 1:
            # Symmetry breaking: differences[1] < differences[last]
            G.add_node(f'symmetry_{ruler}', type=1, weight=0.5)
            # Connect first and last marks (involved in symmetry breaking)
            G.add_edge(f'mark{ruler}_0', f'symmetry_{ruler}', weight=0.7)
            G.add_edge(f'mark{ruler}_{m-1}', f'symmetry_{ruler}', weight=0.7)
    
    # Cumulative constraint modeling (distinct predicate using cumulative)
    for ruler, m in [('1', m1), ('2', m2), ('3', m3)]:
        # The distinct predicate uses cumulative with specific capacity
        G.add_node(f'cumulative_{ruler}', type=1, weight=0.6)
        # Connect all difference-generating marks to cumulative constraint
        for i in range(m):
            G.add_edge(f'mark{ruler}_{i}', f'cumulative_{ruler}', weight=0.5)
    
    # Cross-ruler complexity relationships
    # Rulers interact through search ordering but not direct constraints
    # Add weak connections between corresponding positions to model search interaction
    max_common = min(m1, m2, m3)
    for i in range(min(max_common, 3)):  # Only first few positions to avoid clutter
        # Search interdependencies - decisions in one ruler affect others
        search_weight = 0.2 * math.exp(-i * 0.5)  # Exponential decay
        
        if i < m1 and i < m2:
            G.add_edge(f'mark1_{i}', f'mark2_{i}', weight=search_weight)
        if i < m2 and i < m3:
            G.add_edge(f'mark2_{i}', f'mark3_{i}', weight=search_weight)
        if i < m1 and i < m3:
            G.add_edge(f'mark1_{i}', f'mark3_{i}', weight=search_weight * 0.5)
    
    # Global difficulty node reflecting overall problem complexity
    total_marks = m1 + m2 + m3
    total_constraints = sum((m * (m - 1)) // 2 for m in [m1, m2, m3])
    difficulty = min(math.log(total_marks * total_constraints + 1) / math.log(1000), 1.0)
    
    G.add_node('global_complexity', type=2, weight=difficulty)
    
    # Connect global complexity to most critical elements
    # Objective variable (last mark of middle ruler)
    if m2 > 0:
        G.add_edge(f'mark2_{m2-1}', 'global_complexity', weight=0.9)
    
    # Most constrained distinctness constraints
    for ruler, m in [('1', m1), ('2', m2), ('3', m3)]:
        if m > 2:  # Only for non-trivial rulers
            constraint_complexity = m * (m - 1) // 2
            connection_weight = min(0.4 + 0.3 * constraint_complexity / 50, 0.8)
            G.add_edge(f'distinct_{ruler}', 'global_complexity', weight=connection_weight)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()