#!/usr/bin/env python3
"""
Graph converter for Ghoulomb problem.
Created using subagent_prompt.md version: v_02

This problem is about constructing three Golomb rulers sequentially where only 
the middle one is minimized. A Golomb ruler places marks such that all pairwise
differences are distinct.

Key challenges: 
- Complex interaction between three rulers
- Non-linear constraint structure with quadratic difference variables
- Artificial redundancy designed to make search harder
- Uses cumulative scheduling instead of all_different for obfuscation
"""

import sys
import json
import math
import networkx as nx
from pathlib import Path


def build_graph(mzn_file, json_data):
    """
    Build graph representation of the Ghoulomb problem instance.
    
    Args:
        mzn_file: Path to .mzn file (for reference)
        json_data: Dict containing parsed DZN data
    
    Strategy: Model as bipartite graph with variable and constraint nodes
    - Variables: Mark positions for each ruler (type 0)
    - Constraints: Ordering, distinctness, and ruler-specific constraints (type 1)
    - Weights reflect importance in optimization and constraint complexity
    
    Key insight: The middle ruler (m2) is most critical since it's the objective.
    Larger rulers create more complex constraint interactions.
    """
    # Access data from json_data dict
    m1 = json_data.get('m1', 0)
    m2 = json_data.get('m2', 0) 
    m3 = json_data.get('m3', 0)
    
    # Create graph
    G = nx.Graph()
    
    # Variable nodes: Mark positions for each ruler
    # Weight by ruler importance (m2 is objective) and position criticality
    rulers = [('ruler1', m1, 0.3), ('ruler2', m2, 1.0), ('ruler3', m3, 0.3)]
    
    for ruler_name, m, ruler_importance in rulers:
        for i in range(1, m + 1):  # marks 1 to m (mark[1] = 0 is fixed)
            # First and last marks are more critical (boundary effects)
            position_criticality = 1.0 if i == 1 or i == m else 0.7
            weight = ruler_importance * position_criticality
            G.add_node(f'{ruler_name}_mark_{i}', type=0, weight=weight)
    
    # Constraint nodes: Model the complex constraint structure
    
    # 1. Ordering constraints for each ruler (mark[i] < mark[i+1])
    for ruler_name, m, ruler_importance in rulers:
        for i in range(1, m):  # m-1 ordering constraints per ruler
            # Weight by ruler importance and position (later positions more constrained)
            position_factor = math.sqrt(i / m)  # Non-linear: later constraints harder
            weight = ruler_importance * (0.5 + 0.5 * position_factor)
            G.add_node(f'{ruler_name}_order_{i}', type=1, weight=weight)
    
    # 2. Distinctness constraints for difference variables
    # Each ruler has (m*(m-1))/2 difference variables that must be distinct
    for ruler_name, m, ruler_importance in rulers:
        if m > 1:
            num_differences = (m * (m - 1)) // 2
            # Weight by complexity: more differences = harder constraint
            complexity = math.log(num_differences + 1) / math.log(100)  # Logarithmic scaling
            weight = ruler_importance * min(0.8 + 0.2 * complexity, 1.0)
            G.add_node(f'{ruler_name}_distinct', type=1, weight=weight)
    
    # 3. Symmetry breaking constraints (differences[1] < differences[last])
    for ruler_name, m, ruler_importance in rulers:
        if m > 2:  # Only meaningful for m > 2
            weight = ruler_importance * 0.6  # Moderately important
            G.add_node(f'{ruler_name}_symbreak', type=1, weight=weight)
    
    # 4. Cumulative scheduling constraints (the evil twist)
    # Each ruler uses cumulative instead of all_different
    for ruler_name, m, ruler_importance in rulers:
        # Weight by problem size and ruler importance
        scheduling_complexity = 1.0 - math.exp(-m / 10.0)  # Exponential growth
        weight = ruler_importance * scheduling_complexity
        G.add_node(f'{ruler_name}_cumulative', type=1, weight=weight)
    
    # Edges: Variable-constraint participation (bipartite structure)
    
    # Variables participate in ordering constraints
    for ruler_name, m, _ in rulers:
        for i in range(1, m):  # For each ordering constraint
            # Each ordering constraint involves two consecutive marks
            G.add_edge(f'{ruler_name}_mark_{i}', f'{ruler_name}_order_{i}', weight=1.0)
            G.add_edge(f'{ruler_name}_mark_{i+1}', f'{ruler_name}_order_{i}', weight=1.0)
    
    # Variables participate in distinctness constraints
    for ruler_name, m, _ in rulers:
        if m > 1:
            # All marks participate in the distinctness constraint
            for i in range(1, m + 1):
                # Weight by how many differences this mark participates in
                participation = (i - 1) + (m - i)  # Differences involving mark i
                max_participation = m - 1
                weight = 0.5 + 0.5 * (participation / max_participation) if max_participation > 0 else 0.5
                G.add_edge(f'{ruler_name}_mark_{i}', f'{ruler_name}_distinct', weight=weight)
    
    # Variables participate in symmetry breaking
    for ruler_name, m, _ in rulers:
        if m > 2:
            # First and last marks are most relevant for symmetry breaking
            G.add_edge(f'{ruler_name}_mark_1', f'{ruler_name}_symbreak', weight=0.8)
            G.add_edge(f'{ruler_name}_mark_{m}', f'{ruler_name}_symbreak', weight=0.8)
            # Other marks have lower participation
            for i in range(2, m):
                G.add_edge(f'{ruler_name}_mark_{i}', f'{ruler_name}_symbreak', weight=0.3)
    
    # Variables participate in cumulative constraints
    for ruler_name, m, _ in rulers:
        for i in range(1, m + 1):
            # All marks participate equally in the cumulative constraint
            G.add_edge(f'{ruler_name}_mark_{i}', f'{ruler_name}_cumulative', weight=0.7)
    
    # Cross-ruler interactions: Search order dependencies
    # The search goes mark1++mark2++mark3, creating subtle interactions
    if m1 > 0 and m2 > 0:
        # Add weak coupling between rulers due to search order
        interaction_weight = 0.2 * min(m1, m2) / max(max(m1, m2), 1)
        G.add_node('ruler1_ruler2_coupling', type=1, weight=interaction_weight)
        # Connect first marks of each ruler (search order effect)
        if m1 >= 1 and m2 >= 1:
            G.add_edge('ruler1_mark_1', 'ruler1_ruler2_coupling', weight=0.3)
            G.add_edge('ruler2_mark_1', 'ruler1_ruler2_coupling', weight=0.5)
    
    if m2 > 0 and m3 > 0:
        interaction_weight = 0.2 * min(m2, m3) / max(max(m2, m3), 1)
        G.add_node('ruler2_ruler3_coupling', type=1, weight=interaction_weight)
        if m2 >= 1 and m3 >= 1:
            G.add_edge('ruler2_mark_1', 'ruler2_ruler3_coupling', weight=0.5)
            G.add_edge('ruler3_mark_1', 'ruler2_ruler3_coupling', weight=0.3)
    
    return G


def main():
    if len(sys.argv) != 4:
        print("Usage: python converter.py <mzn_file> <dzn_file> <json_file>")
        sys.exit(1)
    
    mzn_file = sys.argv[1]
    dzn_file = sys.argv[2]
    json_file = sys.argv[3]
    
    # Load JSON data
    with open(json_file, 'r') as f:
        json_data = json.load(f)
    
    # Build graph
    G = build_graph(mzn_file, json_data)
    
    # Graph is returned by build_graph for direct feature extraction
    print(f"Graph built: {G.number_of_nodes()} nodes, {G.number_of_edges()} edges")


if __name__ == "__main__":
    main()